// -*- C++ -*-

/* 
 * Gnome Crystal
 * application.cc 
 *
 * Copyright (C) 2001-2003
 *
 * Developed by Jean Bréfort <jean.brefort@ac-dijon.fr>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "config.h"
#include <gtk/gtk.h>
#include <libgnome/libgnome.h>
#include <libgnomeui/libgnomeui.h>
#include <glade/glade.h>
#include <libgnomeprint/gnome-print.h>
#include <libgnomeprint/gnome-print-job.h>
#include <libgnomeprintui/gnome-print-dialog.h>
//#include <libgnomeprintui/gnome-print-preview.h>
#include <libgnomeprintui/gnome-print-job-preview.h>
#include "application.h"
#include "globals.h"
#include "filesel.h"
#include "prefs.h"
#include "view-settings.h"
std::list<gcApplication*> Apps;

static unsigned short nNewDocs = 1;
guint TabPos =  0;

//Callbacks
static void on_file_new(GtkWidget *widget, gcApplication *app)
{
	app->OnFileNew();
}

static void on_file_open(GtkWidget *widget, gcApplication *app)
{
	app->OnFileOpen();
}

static void on_file_open_new_win(GtkWidget *widget, gcApplication *app)
{
	gcApplication* pApp = new gcApplication();
	Apps.push_back(pApp);
	pApp->OnFileOpen();
}

static void on_file_save(GtkWidget *widget, gcApplication *app)
{
	app->OnFileSave();
}

static void on_file_save_as(GtkWidget *widget, gcApplication *app)
{
	app->OnFileSaveAs();
}

static void on_file_close(GtkWidget *widget, gcApplication *app)
{
	app->OnFileClose();
}

static void on_file_print(GtkWidget *widget, gcApplication *app)
{
	app->OnFilePrint();
}

#ifdef HAVE_LIBJPEG
static void on_export_jpeg(GtkWidget *widget, gcApplication *app)
{
	app->OnExportJPEG();
}
#endif

#ifdef HAVE_PNG
static void on_export_png(GtkWidget *widget, gcApplication *app)
{
	app->OnExportPNG();
}
#endif

static void on_export_vrml(GtkWidget *widget, gcApplication *app)
{
	app->OnExportVRML();
}

static void on_view_new(GtkWidget *widget, gcApplication *app)
{
	app->OnViewNew();
}

static void on_view_close(GtkWidget *widget, gcApplication *app)
{
	app->OnViewClose();
}

static void on_about()
{
	char * authors[]={"Jean Bréfort",NULL};
	char * documentors[] = {NULL};
	GdkPixbuf* pixbuf = gdk_pixbuf_new_from_file ( DATADIR"/pixmaps/gnome-crystal-logo.png", NULL);
/*Note for translators: please add an acute accent to the e in Brefort*/
	char * translators = _(\
"Jean Brefort <jean.brefort@ac-dijon.fr>: French\n\
Tigrux <tigrux@avantel.net>: Spanish\n\
Costantino Ceoldo <inverness1@virgilio.it>: Italian"
			);
	GtkWidget* about = gnome_about_new(_("Gnome Crystal"), VERSION,
/*Note for translators: please add an acute accent to the e in Brefort*/
										_("(C) 1999-2001 by Jean Brefort"),
										_("Gnome Crystal is a lightweight crystal structures viewer for Gnome"),
										(const gchar**) authors, (const gchar**) documentors, translators, pixbuf);
	gtk_widget_show_all(about);
}

static void on_help()
{
/*	gchar* helpfile = gnome_help_file_find_file("gcrystal", "index.html");
	if (helpfile != NULL)
	{
		gchar* url = g_strconcat("ghelp:", helpfile, NULL);
		gnome_help_goto(NULL, url);
		g_free(url);
		g_free(helpfile);
	}
	else
	{
		gnome_error_dialog(_("Couldn't find the Gnome Crystal manual"));
	}*/
}

static void on_lattice(GtkWidget *widget, gcApplication *app)
{
	if (app) app->OnCrystalDefine(0);
}

static void on_atoms(GtkWidget *widget, gcApplication *app)
{
	if (app) app->OnCrystalDefine(1);
}

static void on_bonds(GtkWidget *widget, gcApplication *app)
{
	if (app) app->OnCrystalDefine(2);
}

static void on_size(GtkWidget *widget, gcApplication *app)
{
	if (app) app->OnCrystalDefine(3);
}

static void on_cleavages(GtkWidget *widget, gcApplication *app)
{
	if (app) app->OnCrystalDefine(4);
}

static void on_view_settings(GtkWidget *widget, gcApplication *app)
{
	if (app) app->OnViewSettings();
}

extern std::list<gcDocument*> Docs;

static bool on_quit(GtkWidget *widget, gcApplication *app)
{
	while (Apps.size())
	{
		if (Apps.front()->OnFileClose())
			while (gtk_events_pending()) gtk_main_iteration();
		else break;
	}
	if (Docs.size()) return false;
	delete Apps.front();
	Apps.pop_front();
	if (Apps.size() != 0) return false;
	gtk_main_quit();
	return true;
}

static void on_select(GtkWidget* widget, gcApplication *app)
{
	GtkWidget* w = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "page"));
	app->OnSelectView(w);
}

static void on_change_page(GtkWidget* widget, GtkNotebook* book, int page, gcApplication *app)
{
	app->OnChangePage(page);
}

static bool on_delete(GtkWidget* widget, GdkEventAny *event, gcApplication *app)
{
	while (app->OnViewClose()) while (gtk_events_pending()) gtk_main_iteration();
	std::list<gcApplication*>::iterator i;
	for (i = Apps.begin(); i != Apps.end(); i++)
	{
		if (app == *i)
		{
			if (!app->IsEmpty()) return true;
			Apps.remove(app);
			delete app;
			return false;
		}
	}
	return false;
}

static void on_destroy(GtkWidget* widget, gcApplication *app)
{
	if (Apps.empty()) gtk_main_quit();
}

static void on_prefs(GtkWidget* widget, gcApplication *app)
{
	gcPrefsDlg* box = new gcPrefsDlg();
}

//Helper functions
static bool do_load(const gchar* filename, gcView* pView)
{
	gcApplication* pApp = pView->GetApp();
	pView = pApp->GetDocView(filename);
	gcDocument* pDoc = pView->GetDocument();
	pApp = pView->GetApp();
	if (pDoc->GetFileName() && !strcmp(pDoc->GetFileName(), filename))
	{
		pApp->SelectView(pView);
		if (!pDoc->IsDirty()) return true;
		else
		{
			gchar* str = g_strdup_printf(_("``%s'' has been modified since last saving. Do you wish to come back to saved version?"), pDoc->GetTitle());
			GtkWidget* mbox = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO, str);
			int res = gtk_dialog_run(GTK_DIALOG(mbox));
			if (res != GTK_RESPONSE_YES) return true;
		}
	}
	if (pDoc->Load(filename))
	{
		gtk_label_set_text(pView->GetLabel(), pDoc->GetTitle());
		GtkLabel *pLabel = pView->GetMenuLabel();
		if (pLabel) gtk_label_set_text(pLabel, pDoc->GetTitle());
		return true;
	}
	nNewDocs++;
	return false;
}

static void do_save_as(const gchar* filename, gcView* pView)
{
	gcDocument *pDoc = pView->GetDocument();
	pDoc->SetFileName(filename);
	pDoc->Save();
}

#ifdef HAVE_LIBJPEG
static void do_export_jpeg(const gchar *filename, gcView* pView)
{
	if (!pView) return;
	pView->ExportJPG(filename);
}
#endif

#ifdef HAVE_PNG
static void do_export_png(const gchar *filename, gcView* pView)
{
	if (!pView) return;
	pView->ExportPNG(filename);
}
#endif

static void do_export_vrml(const gchar *filename, gcView* pView)
{
	gcDocument *pDoc = pView->GetDocument();
	if (pDoc) pDoc->OnExportVRML(filename, pView);;
}

bool RequestApp(gcView* pView)
{
	gcDocument* pDoc = pView->GetDocument();
	gcApplication* pApp = NULL;
	std::list<gcApplication*>::iterator i;
	for (i = Apps.begin(); i!= Apps.end(); i++)
	{
		if (!(*i)->HasDocument(pDoc))
		{
			pApp = *i;
			break;
		}
	}
	if (!pApp)
	{
		pApp = new gcApplication();
		Apps.push_back(pApp);
	}
	if (pApp)
	{
		pApp->AddView(pView);
		gtk_label_set_text(pView->GetLabel(), pDoc->GetTitle());
		GtkLabel *pLabel = pView->GetMenuLabel();
		if (pLabel) gtk_label_set_text(pLabel, pDoc->GetTitle());
	}
	return (pApp != NULL);
}

//Implementation of gcApplication class

gcApplication::gcApplication()
{
	GladeXML *xml ;
	xml = glade_xml_new(DATADIR"/gcrystal/glade/application.glade", "app", NULL);


	m_App = glade_xml_get_widget(xml, "app");
	g_signal_connect(G_OBJECT(m_App), "delete_event", (GtkSignalFunc)on_delete, this);
	g_signal_connect(G_OBJECT(m_App), "destroy", (GtkSignalFunc)on_destroy, this);
	m_Notebook = GTK_NOTEBOOK(glade_xml_get_widget(xml, "notebook"));
	g_signal_connect(G_OBJECT(m_Notebook), "switch_page", (GtkSignalFunc)on_change_page, this); 

	m_WindowsMenu = (GtkMenu*) ((glade_xml_get_widget(xml, "Placeholder"))->parent);	//Connect callbacks to menu items
	GObject *obj;
	obj = G_OBJECT(glade_xml_get_widget(xml, "exit"));
	g_signal_connect(obj, "activate", (GtkSignalFunc)on_quit, this); 
	obj = G_OBJECT(glade_xml_get_widget(xml, "quit_btn"));
	g_signal_connect(obj, "clicked", (GtkSignalFunc)on_quit, this); 
	obj = G_OBJECT(glade_xml_get_widget(xml, "new"));
	g_signal_connect(obj, "activate", (GtkSignalFunc)on_file_new, this); 
	obj = G_OBJECT(glade_xml_get_widget(xml, "new_btn"));
	g_signal_connect(obj, "clicked", (GtkSignalFunc)on_file_new, this); 
	obj = G_OBJECT(glade_xml_get_widget(xml, "open"));
	g_signal_connect(obj, "activate", (GtkSignalFunc)on_file_open, this); 
	obj = G_OBJECT(glade_xml_get_widget(xml, "open_btn"));
	g_signal_connect(obj, "clicked", (GtkSignalFunc)on_file_open, this); 
	obj = G_OBJECT(glade_xml_get_widget(xml, "open_new_win"));
	g_signal_connect(obj, "activate", (GtkSignalFunc)on_file_open_new_win, this); 
	obj = G_OBJECT(glade_xml_get_widget(xml, "save"));
	g_signal_connect(obj, "activate", (GtkSignalFunc)on_file_save, this); 
	obj = G_OBJECT(glade_xml_get_widget(xml, "save_btn"));
	g_signal_connect(obj, "clicked", (GtkSignalFunc)on_file_save, this); 
	obj = G_OBJECT(glade_xml_get_widget(xml, "save_as"));
	g_signal_connect(obj, "activate", (GtkSignalFunc)on_file_save_as, this); 
	obj = G_OBJECT(glade_xml_get_widget(xml, "close"));
	g_signal_connect(obj, "activate", (GtkSignalFunc)on_file_close, this); 
	obj = G_OBJECT(glade_xml_get_widget(xml, "close_btn"));
	g_signal_connect(obj, "clicked", (GtkSignalFunc)on_file_close, this); 
	obj = G_OBJECT(glade_xml_get_widget(xml, "print"));
	g_signal_connect(obj, "activate", (GtkSignalFunc)on_file_print, this); 
	obj = G_OBJECT(glade_xml_get_widget(xml, "vrml"));
	g_signal_connect(obj, "activate", (GtkSignalFunc)on_export_vrml, this);
	obj = G_OBJECT(glade_xml_get_widget(xml, "jpeg"));
#ifdef HAVE_LIBJPEG
	g_signal_connect(obj, "activate", (GtkSignalFunc)on_export_jpeg, this); 
#else
	gtk_widget_hide(GTK_WIDGET(obj));
#endif
	obj = G_OBJECT(glade_xml_get_widget(xml, "png"));
#ifdef HAVE_PNG
	g_signal_connect(obj, "activate", (GtkSignalFunc)on_export_png, this); 
#else
	gtk_widget_hide(GTK_WIDGET(obj));
#endif
	obj = G_OBJECT(glade_xml_get_widget(xml, "print_btn"));
	g_signal_connect(obj, "clicked", (GtkSignalFunc)on_file_print, this); 
	obj = G_OBJECT(glade_xml_get_widget(xml, "lattice"));
	g_signal_connect(obj, "activate", (GtkSignalFunc)on_lattice, this); 
	obj = G_OBJECT(glade_xml_get_widget(xml, "atoms"));
	g_signal_connect(obj, "activate", (GtkSignalFunc)on_atoms, this); 
	obj = G_OBJECT(glade_xml_get_widget(xml, "bonds"));
	g_signal_connect(obj, "activate", (GtkSignalFunc)on_bonds, this); 
	obj = G_OBJECT(glade_xml_get_widget(xml, "size"));
	g_signal_connect(obj, "activate", (GtkSignalFunc)on_size, this); 
	obj = G_OBJECT(glade_xml_get_widget(xml, "cleavages"));
	g_signal_connect(obj, "activate", (GtkSignalFunc)on_cleavages, this); 
	obj = G_OBJECT(glade_xml_get_widget(xml, "view_settings"));
	g_signal_connect(obj, "activate", (GtkSignalFunc)on_view_settings, this); 
	obj = G_OBJECT(glade_xml_get_widget(xml, "preferences"));
	g_signal_connect(obj, "activate", (GtkSignalFunc)on_prefs, this); 
	obj = G_OBJECT(glade_xml_get_widget(xml, "new_view"));
	g_signal_connect(obj, "activate", (GtkSignalFunc)on_view_new, this); 
	obj = G_OBJECT(glade_xml_get_widget(xml, "close_view"));
	g_signal_connect(obj, "activate", (GtkSignalFunc)on_view_close, this); 
	obj = G_OBJECT(glade_xml_get_widget(xml, "about"));
	g_signal_connect(obj, "activate", (GtkSignalFunc)on_about, NULL); 
	obj = G_OBJECT(glade_xml_get_widget(xml, "manual"));
	g_signal_connect(obj, "activate", (GtkSignalFunc)on_help, NULL); 
	UpdateConfig();
}

gcApplication::~gcApplication()
{
}

void gcApplication::OnFileNew()
{
	gcDocument* pDoc = GetNewDocument();
	gchar tmp[32];
	g_snprintf(tmp, sizeof(tmp), _("Untitled%d"), nNewDocs++);
	pDoc->SetTitle(tmp);
	gcView* pView = (gcView*)pDoc->GetView();
	AddView(pView);
	m_bFileOpening = false;
}

void gcApplication::OnFileOpen()
{
	gcView* pView;
	gcDocument* pDoc;
	if (m_Views.empty())
	{
		OnFileNew();
		m_bFileOpening = true;
	}
	pView = m_Views.back();
	gcFileSel* FileSel = new gcFileSel(_("Load a model..."), (void (*)(const    gchar*, gcView*))do_load, false, ".gcrystal", pView);
}

void gcApplication::OnFileSave()
{
	if (!m_pActiveView) return;
	gcDocument* pDoc = m_pActiveView->GetDocument();
	if (!pDoc) return;
	if (pDoc->GetFileName()) pDoc->Save();
	else OnFileSaveAs();
}

void gcApplication::OnFileSaveAs()
{
	if (!m_pActiveView) return;
	gcDocument* pDoc = m_pActiveView->GetDocument();
	if (!pDoc) return;
	gcFileSel* FileSel = new gcFileSel(_("Save model as..."), do_save_as, true, ".gcrystal", m_pActiveView);
}

bool gcApplication::OnFileClose()
{
	if (m_pActiveView == NULL)
	{
		if (m_Views.empty() && (Apps.size() > 1))
		{
			Apps.remove(this);
			gtk_widget_destroy(m_App);
			return true;
		}
		else return false;
	}
	if (m_pActiveView->IsLocked()) return false;
	gcDocument* pDoc = m_pActiveView->GetDocument();
	if (!pDoc->VerifySaved()) return false;
	std::list<gcApplication*>::iterator i;
	std::list<gcView*>::iterator j;
	gcView *pView;
	bool bEnded;
	do
	{
		bEnded = true;
		for (i = Apps.begin(); i != Apps.end(); i++)
		{
			if (*i == this) continue;
			for (j = (*i)->m_Views.begin(); j != (*i)->m_Views.end(); j++)
				if ((*j)->GetDocument() == pDoc)
				{
					(*i)->OnViewClose(*j);
					bEnded = false;
					break;
				}
			if (!bEnded) break;
		}
	}
	while (!bEnded);
	OnViewClose(m_pActiveView);
	return true;
}

void gcApplication::OnSelectView(GtkWidget* widget)
{
	gint i;
	i = gtk_notebook_page_num(m_Notebook, widget);
	if(i != gtk_notebook_get_current_page(m_Notebook))
		gtk_notebook_set_current_page(m_Notebook, i);
}

void gcApplication::OnCrystalDefine(int page)
{
	gcDocument* pDoc = m_pActiveView->GetDocument();
	if (pDoc) pDoc->Define(page);
}

void gcApplication::OnViewSettings()
{
	if (!m_pActiveView) return;
	gcViewSettingsDlg* pDlg = new gcViewSettingsDlg(m_pActiveView);
}

void gcApplication::OnFilePrint()
{
	GnomePrintConfig* config = gnome_print_config_default();
	GnomePrintContext *pc;// = gnome_print_context_new(config);
	GnomePrintJob *gpj = gnome_print_job_new(config);
	int do_preview, copies = 1, collate = 0;
	GnomePrintDialog *gpd;
	gpd = GNOME_PRINT_DIALOG (gnome_print_dialog_new(gpj, (const guchar*)"Print test", GNOME_PRINT_DIALOG_COPIES));
	gnome_print_dialog_set_copies(gpd, copies, collate);
	switch (gtk_dialog_run(GTK_DIALOG(gpd)))
	{
	case GNOME_PRINT_DIALOG_RESPONSE_PRINT:
		do_preview = 0;
		break;
	case GNOME_PRINT_DIALOG_RESPONSE_PREVIEW:
		do_preview = 1;
		break;
	case GNOME_PRINT_DIALOG_RESPONSE_CANCEL:
		gtk_widget_destroy (GTK_WIDGET(gpd));
		return;
	}
//	GnomePrintConfig *gpc = gnome_print_dialog_get_config (GNOME_PRINT_DIALOG (gpd));
	gtk_widget_destroy(GTK_WIDGET(gpd));
//	gpm = gnome_print_master_new_from_config (gpc);
	pc = gnome_print_job_get_context (gpj);
	gnome_print_beginpage(pc, (const guchar*)"");
	gdouble width, height;
	gnome_print_job_get_page_size_from_config(config, &width, &height);//FIXME: should work
printf("paper size: width:%g height:%g\n",width,height);
	width = 595.2755906;//suppose A4
	height = 841.8897638;
	m_pActiveView->Print(pc, width, height);
	gnome_print_showpage(pc);
	g_object_unref(pc);
	gnome_print_job_close(gpj);
	if (do_preview)
	{
		gtk_widget_show (gnome_print_job_preview_new (gpj, (const guchar*)_("Preview")));
	}
	else
	{
		gnome_print_job_print(gpj);
	}
	g_object_unref (gpj);
	gnome_print_config_unref(config);
}

#ifdef HAVE_LIBJPEG
void gcApplication::OnExportJPEG()
{
	if (!m_pActiveView) return;
	gcFileSel* FileSel = new gcFileSel(_("Export model as a Jpeg file..."), do_export_jpeg, true, ".jpg", m_pActiveView);
}
#endif

#ifdef HAVE_PNG
void gcApplication::OnExportPNG()
{
	if (!m_pActiveView) return;
	gcFileSel* FileSel = new gcFileSel(_("Export model as a PNG file..."), do_export_png, true, ".png", m_pActiveView);
}
#endif

void gcApplication::OnExportVRML()
{
	if (!m_pActiveView) return;
	gcFileSel* FileSel = new gcFileSel(_("Export model as a VRML file..."), do_export_vrml, true, ".wrl", m_pActiveView);
}

void gcApplication::OnViewNew()
{
	if (!m_pActiveView) return;
	gcView* pView = new gcView(m_pActiveView);
	gcDocument* pDoc = m_pActiveView->GetDocument();
	//Try to add the new view to an existing gcApplication
	gcApplication* pApp= NULL;
	std::list<gcApplication*>::iterator i;
	for (i = Apps.begin(); i!= Apps.end(); i++)
	{
		if (*i == this) continue;
		if (!(*i)->HasDocument(pDoc))
		{
			pApp = *i;
			break;
		}
	}
	if (!pApp)
	{
		pApp = new gcApplication();
		Apps.push_back(pApp);
	}
	if (pApp)
	{
		pApp->AddView(pView);
		pDoc->SetDirty();
		gtk_label_set_text(pView->GetLabel(), pDoc->GetTitle());
		GtkLabel *pLabel = pView->GetMenuLabel();
		if (pLabel) gtk_label_set_text(pLabel, pDoc->GetTitle());
	}
	else delete pView;
}

bool gcApplication::OnViewClose(gcView* pView)
{
	if (pView == NULL) pView = m_pActiveView;
	if (pView == NULL) return false;
	gcDocument* pDoc = pView->GetDocument();
	gint i = 0;
	GtkWidget* w;
	if (pView == m_pActiveView) i = gtk_notebook_get_current_page(m_Notebook);
	else do
	{
		w = gtk_notebook_get_nth_page(m_Notebook, i);
		if (w == NULL) return false; //should not occur
		if (pView == ((gcView*) g_object_get_data(G_OBJECT(w), "view"))) break;
		else i++;
	}
	while (w);
	if (!pDoc->RemoveView(pView)) return false;
	m_Views.remove(pView);
	gtk_notebook_remove_page(m_Notebook, i);
	delete pView;
	if (pView == m_pActiveView)
	{
		i = gtk_notebook_get_current_page(m_Notebook);
		w = gtk_notebook_get_nth_page(m_Notebook, i);
		m_pActiveView = (w) ? (gcView*) g_object_get_data(G_OBJECT(w), "view") : NULL;
	}
	if (m_Views.empty() && (Apps.size() > 1))
	{
		Apps.remove(this);
		gtk_widget_destroy(m_App);
		delete this;
	}
	return true;
}

bool gcApplication::HasDocument(gcDocument* pDoc)
{
	std::list<gcView*>::iterator i;
	for (i = m_Views.begin(); i != m_Views.end(); i++)
		if (pDoc == (*i)->GetDocument()) return true;
	return false;
}

void gcApplication::AddView(gcView* pView)
{
	pView->SetApp(this);
	GtkWidget* widget = pView->CreateNewWidget();
	m_Views.push_back(pView);
	m_pActiveView = pView;
	gchar* tmp = pView->GetDocument()->GetTitle();
	GtkWidget *label = gtk_label_new(tmp);
	pView->SetLabel(GTK_LABEL(label));
	GtkWidget* item = gtk_menu_item_new_with_label(tmp);
	pView->SetMenu(GTK_MENU_ITEM(item));
	g_object_set_data(G_OBJECT(item), "page", widget);
	g_object_set_data(G_OBJECT(widget), "menu", item);
	g_object_set_data(G_OBJECT(widget), "view", pView);
	gtk_widget_show(item);
	gtk_menu_shell_append((GtkMenuShell*)m_WindowsMenu, item);
	gtk_notebook_append_page(m_Notebook, widget, label);
	gtk_notebook_set_current_page(m_Notebook, g_list_length(m_Notebook->children) - 1);
	g_signal_connect(G_OBJECT(item), "activate", (GtkSignalFunc)on_select, this); 
}

bool gcApplication::LoadFile(char* filename)
{
	return do_load(filename, m_pActiveView);
}

void gcApplication::UpdateConfig()
{
	if (TabPos == 0) gtk_notebook_set_show_tabs(m_Notebook, false);
	else
	{
		gtk_notebook_set_show_tabs(m_Notebook, true);
		gtk_notebook_set_tab_pos(m_Notebook, (GtkPositionType)(TabPos - 1));
	}
}

gcView* gcApplication::GetDocView(const char* filename)
{
	gcDocument* pDoc;
	gcView *pView = NULL;
	std::list<gcView*>::iterator i;
	for (i = m_Views.begin(); i != m_Views.end(); i++)
	{
		pDoc = (*i)->GetDocument();
		if (!pDoc->GetFileName()) continue;
		if (!strcmp(pDoc->GetFileName(), filename))
		{
			pView = *i;
			break;
		}
	}
	if (!pView) //search doc in other apps
	{
		std::list<gcApplication*>::iterator j;
		for (j = Apps.begin(); j != Apps.end() && !pView; j++)
		{
			if (*j == this) continue;
			for (i = (*j)->m_Views.begin(); i != (*j)->m_Views.end(); i++)
			{
				pDoc = (*i)->GetDocument();
				if (!pDoc->GetFileName()) continue;
				if (!strcmp(pDoc->GetFileName(), filename))
				{
					pView = *i;
					GdkWindow *win = GTK_WIDGET((*j)->m_App)->window;
					gdk_window_hide(win);
					gdk_window_show(win);
					gdk_window_raise(win);
					break;
				}
			}
		}
	}
	if (pView) return pView;
	if (m_bFileOpening)
	{
		pView = m_Views.back();
		pDoc = pView->GetDocument();
		if (!pDoc->IsEmpty() || pDoc->IsDirty()) pView = NULL;
	}
	if (!pView)
	{
		OnFileNew();
		pView = m_Views.back();
	}
	nNewDocs--;
	return pView;
}

gcView* gcApplication::GetView(gcDocument* pDoc)
{
	std::list<gcView*>::iterator i;
	for (i = m_Views.begin(); i != m_Views.end(); i++)
		if (pDoc == (*i)->GetDocument()) return *i;
	return NULL;
}

void gcApplication::SelectView(gcView* pView)
{
	GtkWidget *w;
	guint i = 0;
	do
	{
		w = gtk_notebook_get_nth_page(m_Notebook, i);
		if (w == NULL) return; //should not occur
		if (pView == ((gcView*) g_object_get_data(G_OBJECT(w), "view"))) break;
		else i++;
	}
	while (w);
	gtk_notebook_set_current_page(m_Notebook, i);
}

void gcApplication::OnChangePage(int i)
{
	GtkWidget *w = gtk_notebook_get_nth_page(m_Notebook, i);
	m_pActiveView = (gcView*)g_object_get_data(G_OBJECT(w), "view");
}
