// -*- C++ -*-

/* 
 * Gnome Crystal
 * latticedlg.cc 
 *
 * Copyright (C) 2000-2002
 *
 * Developed by Jean Brfort <jean.brefort@ac-dijon.fr>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "../config.h"
#include <gnome.h>
#include <glade/glade.h>
#include <stdlib.h>
#include <stdio.h>
#include "element.h"
#include "latticedlg.h"
#include "mendeleiev.h"

extern "C"
{
	static void on_atom_add(GtkWidget *widget, void * data)
	{
		gcLatticeDlg* pBox = (gcLatticeDlg*)data;
		pBox->AtomAdd();
	}

	static void on_atom_delete(GtkWidget *widget, void * data)
	{
		gcLatticeDlg* pBox = (gcLatticeDlg*)data;
		pBox->AtomDelete();
	}

	static void on_atom_modify(GtkWidget *widget, void * data)
	{
		gcLatticeDlg* pBox = (gcLatticeDlg*)data;
		pBox->AtomModify();
	}

	static void on_atom_delete_all(GtkWidget *widget, void * data)
	{
		gcLatticeDlg* pBox = (gcLatticeDlg*)data;
		pBox->AtomDeleteAll();
	}

	static void on_atom_select(GtkWidget *widget, void * data)
	{
		gcLatticeDlg* pBox = (gcLatticeDlg*)data;
		pBox->AtomSelect();
	}

	static void on_bond_add(GtkWidget *widget, void * data)
	{
		gcLatticeDlg* pBox = (gcLatticeDlg*)data;
		pBox->BondAdd();
	}

	static void on_bond_delete(GtkWidget *widget, void * data)
	{
		gcLatticeDlg* pBox = (gcLatticeDlg*)data;
		pBox->BondDelete();
	}

	static void on_bond_modify(GtkWidget *widget, void * data)
	{
		gcLatticeDlg* pBox = (gcLatticeDlg*)data;
		pBox->BondModify();
	}

	static void on_bond_delete_all(GtkWidget *widget, void * data)
	{
		gcLatticeDlg* pBox = (gcLatticeDlg*)data;
		pBox->BondDeleteAll();
	}

	static void on_bond_select(GtkWidget *widget, void * data)
	{
		gcLatticeDlg* pBox = (gcLatticeDlg*)data;
		pBox->BondSelect();
	}

	static void on_bond_type_changed(GtkWidget *widget, void * data)
	{
		gcLatticeDlg* pBox = (gcLatticeDlg*)data;
		pBox->BondTypeChanged();
	}

	static void on_cleavage_add(GtkWidget *widget, void * data)
	{
		gcLatticeDlg* pBox = (gcLatticeDlg*)data;
		pBox->CleavageAdd();
	}

	static void on_cleavage_delete(GtkWidget *widget, void * data)
	{
		gcLatticeDlg* pBox = (gcLatticeDlg*)data;
		pBox->CleavageDelete();
	}

	static void on_cleavage_modify(GtkWidget *widget, void * data)
	{
		gcLatticeDlg* pBox = (gcLatticeDlg*)data;
		pBox->CleavageModify();
	}

	static void on_cleavage_delete_all(GtkWidget *widget, void * data)
	{
		gcLatticeDlg* pBox = (gcLatticeDlg*)data;
		pBox->CleavageDeleteAll();
	}

	static void on_cleavage_select(GtkWidget *widget, void * data)
	{
		gcLatticeDlg* pBox = (gcLatticeDlg*)data;
		pBox->CleavageSelect();
	}

	static void on_lattice_select(GtkWidget *widget, void * data)
	{
		gcLatticeDlg* pBox = (gcLatticeDlg*)data;
		pBox->LatticeSelect(widget);
	}
	
	static void on_element(GtkWidget *widget, void * data)
	{
		gcLatticeDlg* pBox = (gcLatticeDlg*)data;
		pBox->OnElement();
	}
}

gcLatticeDlg::gcLatticeDlg(gcDocument* pDoc, unsigned nPage): gcDialog(PREFIX"/share/gcrystal/glade/lattice.glade", "lattice", "crystal.html")
{
	m_pDoc = pDoc;
	m_pDoc->NotifyDialog(this);
	m_pAtoms = pDoc->GetAtomList();
	m_pBonds = pDoc->GetBondList();
	m_pCleavages = pDoc->GetCleavageList();

	mendeleiev = NULL;
	
//Connect signals to handlers
	GtkWidget* button = glade_xml_get_widget(xml, "atom_add");
	gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(on_atom_add), this);
	AtomDeleteBtn = (GtkButton*) glade_xml_get_widget(xml, "atom_delete");
	gtk_widget_set_sensitive(GTK_WIDGET(AtomDeleteBtn),0);
	gtk_signal_connect(GTK_OBJECT(AtomDeleteBtn), "clicked", GTK_SIGNAL_FUNC(on_atom_delete), this);
	AtomModifyBtn = (GtkButton*) glade_xml_get_widget(xml, "atom_modify");
	gtk_widget_set_sensitive(GTK_WIDGET(AtomModifyBtn),0);
	gtk_signal_connect(GTK_OBJECT(AtomModifyBtn), "clicked", GTK_SIGNAL_FUNC(on_atom_modify), this);
	AtomDeleteAllBtn = (GtkButton*) glade_xml_get_widget(xml, "atom_delete_all");
	gtk_signal_connect(GTK_OBJECT(AtomDeleteAllBtn), "clicked", GTK_SIGNAL_FUNC(on_atom_delete_all), this);
	button = glade_xml_get_widget(xml, "bond_add");
	gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(on_bond_add), this);
	BondDeleteBtn = (GtkButton*) glade_xml_get_widget(xml, "bond_delete");
	gtk_widget_set_sensitive(GTK_WIDGET(BondDeleteBtn),0);
	gtk_signal_connect(GTK_OBJECT(BondDeleteBtn), "clicked", GTK_SIGNAL_FUNC(on_bond_delete), this);
	BondModifyBtn = (GtkButton*) glade_xml_get_widget(xml, "bond_modify");
	gtk_widget_set_sensitive(GTK_WIDGET(BondModifyBtn),0);
	gtk_signal_connect(GTK_OBJECT(BondModifyBtn), "clicked", GTK_SIGNAL_FUNC(on_bond_modify), this);
	BondDeleteAllBtn = (GtkButton*) glade_xml_get_widget(xml, "bond_delete_all");
	gtk_signal_connect(GTK_OBJECT(BondDeleteAllBtn), "clicked", GTK_SIGNAL_FUNC(on_bond_delete_all), this);
	Btn0 = (GtkToggleButton*)glade_xml_get_widget(xml, "edges");
	gtk_signal_connect(GTK_OBJECT(Btn0), "clicked", GTK_SIGNAL_FUNC(on_bond_type_changed), this);
	Btn1 = (GtkToggleButton*)glade_xml_get_widget(xml, "diagonals");
	gtk_signal_connect(GTK_OBJECT(Btn1), "clicked", GTK_SIGNAL_FUNC(on_bond_type_changed), this);
	Btn2 = (GtkToggleButton*)glade_xml_get_widget(xml, "medians");
	gtk_signal_connect(GTK_OBJECT(Btn2), "clicked", GTK_SIGNAL_FUNC(on_bond_type_changed), this);
	Btn3 = (GtkToggleButton*)glade_xml_get_widget(xml, "bonds_and_lines");
	gtk_signal_connect(GTK_OBJECT(Btn3), "clicked", GTK_SIGNAL_FUNC(on_bond_type_changed), this);
	Btn4 = (GtkToggleButton*)glade_xml_get_widget(xml, "unique_lines");
	gtk_signal_connect(GTK_OBJECT(Btn4), "clicked", GTK_SIGNAL_FUNC(on_bond_type_changed), this);
	gtk_toggle_button_set_state(Btn0, 1);
	button = glade_xml_get_widget(xml, "cleavage_add");
	gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(on_cleavage_add), this);
	CleavageDeleteBtn = (GtkButton*) glade_xml_get_widget(xml, "cleavage_delete");
	gtk_widget_set_sensitive(GTK_WIDGET(CleavageDeleteBtn),0);
	gtk_signal_connect(GTK_OBJECT(CleavageDeleteBtn), "clicked", GTK_SIGNAL_FUNC(on_cleavage_delete), this);
	CleavageModifyBtn = (GtkButton*) glade_xml_get_widget(xml, "cleavage_modify");
	gtk_widget_set_sensitive(GTK_WIDGET(CleavageModifyBtn),0);
	gtk_signal_connect(GTK_OBJECT(CleavageModifyBtn), "clicked", GTK_SIGNAL_FUNC(on_cleavage_modify), this);
	CleavageDeleteAllBtn = (GtkButton*) glade_xml_get_widget(xml, "cleavage_delete_all");
	gtk_signal_connect(GTK_OBJECT(CleavageDeleteAllBtn), "clicked", GTK_SIGNAL_FUNC(on_cleavage_delete_all), this);
	ElementBtn = (GtkButton*) glade_xml_get_widget(xml, "element");
	gtk_signal_connect(GTK_OBJECT(ElementBtn), "clicked", GTK_SIGNAL_FUNC(on_element), this);
	FixedBtn = GTK_TOGGLE_BUTTON(glade_xml_get_widget(xml, "sizefixed"));
	
//Get pointers to useful widgets
//Set notebook current page (don't store)
	gtk_notebook_set_page(notebook, nPage);
	
//Cell page
	LatticeMenu = (GtkOptionMenu*) glade_xml_get_widget(xml, "lattice_type");
	GtkMenu* menu = GTK_MENU(LatticeMenu->menu);
	GList* list = menu->menu_shell.children;
	while (list)
	{
		gtk_signal_connect(GTK_OBJECT(list->data), "activate", GTK_SIGNAL_FUNC(on_lattice_select), this);
		list = g_list_next(list);
	}
	A = (GtkEntry*) glade_xml_get_widget(xml, "a");
	B = (GtkEntry*) glade_xml_get_widget(xml, "b");
	C = (GtkEntry*) glade_xml_get_widget(xml, "c");
	Alpha = (GtkEntry*) glade_xml_get_widget(xml, "alpha");
	Beta = (GtkEntry*) glade_xml_get_widget(xml, "beta");
	Gamma = (GtkEntry*) glade_xml_get_widget(xml, "gamma");

//Atoms page
	AtomX = (GtkEntry*) glade_xml_get_widget(xml, "atomx");
	AtomY = (GtkEntry*) glade_xml_get_widget(xml, "atomy");
	AtomZ = (GtkEntry*) glade_xml_get_widget(xml, "atomz");
	AtomR = (GtkEntry*) glade_xml_get_widget(xml, "atomr");
	AtomColor = (GnomeColorPicker*) glade_xml_get_widget(xml, "atomcolor");
	AtomList = (GtkList*) glade_xml_get_widget(xml, "atomlist");
	gtk_signal_connect(GTK_OBJECT(AtomList), "selection_changed", GTK_SIGNAL_FUNC(on_atom_select), this);

//Bonds page
	BondX1 = (GtkEntry*) glade_xml_get_widget(xml, "bondx1");
	BondY1 = (GtkEntry*) glade_xml_get_widget(xml, "bondy1");
	BondZ1 = (GtkEntry*) glade_xml_get_widget(xml, "bondz1");
	BondX2 = (GtkEntry*) glade_xml_get_widget(xml, "bondx2");
	BondY2 = (GtkEntry*) glade_xml_get_widget(xml, "bondy2");
	BondZ2 = (GtkEntry*) glade_xml_get_widget(xml, "bondz2");
	BondR = (GtkEntry*) glade_xml_get_widget(xml, "bondr");
	BondColor = (GnomeColorPicker*) glade_xml_get_widget(xml, "bondcolor");
	BondList = (GtkList*) glade_xml_get_widget(xml, "bondlist");
	gtk_signal_connect(GTK_OBJECT(BondList), "selection_changed", GTK_SIGNAL_FUNC(on_bond_select), this);

//Size page
	MinX = (GtkEntry*) glade_xml_get_widget(xml, "xmin");
	MaxX = (GtkEntry*) glade_xml_get_widget(xml, "xmax");
	MinY = (GtkEntry*) glade_xml_get_widget(xml, "ymin");
	MaxY = (GtkEntry*) glade_xml_get_widget(xml, "ymax");
	MinZ = (GtkEntry*) glade_xml_get_widget(xml, "zmin");
	MaxZ = (GtkEntry*) glade_xml_get_widget(xml, "zmax");

//Cleavages page
	Btnh = GTK_SPIN_BUTTON(glade_xml_get_widget(xml, "h"));
	Btnk = GTK_SPIN_BUTTON(glade_xml_get_widget(xml, "k"));
	Btnl = GTK_SPIN_BUTTON(glade_xml_get_widget(xml, "l"));
	Btnn = GTK_SPIN_BUTTON(glade_xml_get_widget(xml, "planes"));
	CleavageList = GTK_LIST(glade_xml_get_widget(xml, "cleavagelist"));
	gtk_signal_connect(GTK_OBJECT(CleavageList), "selection_changed", GTK_SIGNAL_FUNC(on_cleavage_select), this);

//Initialize all values
//Cell page
	m_pDoc->GetCell(&m_Lattice, &m_a, &m_b, &m_c, &m_alpha, &m_beta, &m_gamma);
	gtk_option_menu_set_history(LatticeMenu, m_Lattice);
	snprintf(m_buf, sizeof(m_buf), "%g", m_a);
	gtk_entry_set_text(A, m_buf);
	snprintf(m_buf, sizeof(m_buf), "%g", m_b);
	gtk_entry_set_text(B, m_buf);
	snprintf(m_buf, sizeof(m_buf), "%g", m_c);
	gtk_entry_set_text(C, m_buf);
	snprintf(m_buf, sizeof(m_buf), "%g", m_alpha);
	gtk_entry_set_text(Alpha, m_buf);
	snprintf(m_buf, sizeof(m_buf), "%g", m_beta);
	gtk_entry_set_text(Beta, m_buf);
	snprintf(m_buf, sizeof(m_buf), "%g", m_gamma);
	gtk_entry_set_text(Gamma, m_buf);
	UpdateCell();

//Atoms page
	m_Z = 0;
	gcAtomList::iterator i;
	GList *alist = NULL;
	if (m_pAtoms->empty()) gtk_widget_set_sensitive(GTK_WIDGET(AtomDeleteAllBtn),0);
	else for (i = m_pAtoms->begin(); i != m_pAtoms->end(); i++)
	{
		m_pAtom = new gcAtom(**i);
		if (m_pAtom->Z()) snprintf(m_buf, sizeof(m_buf), "%s x=%g,y=%g,z=%g", m_pAtom->Symbol(), m_pAtom->x(), m_pAtom->y(), m_pAtom->z());
		else snprintf(m_buf, sizeof(m_buf), "x=%g,y=%g,z=%g", m_pAtom->x(), m_pAtom->y(), m_pAtom->z());
		alist = g_list_append(alist, gtk_list_item_new_with_label(m_buf));
		m_mAtoms[m_buf] = m_pAtom;
	}
	gtk_list_append_items(AtomList, alist);

//Bonds page
	m_BondType = edges;
	gtk_toggle_button_set_state(Btn0, 1);
	gtk_widget_set_sensitive(GTK_WIDGET(BondX1),0);
	gtk_widget_set_sensitive(GTK_WIDGET(BondY1),0);
	gtk_widget_set_sensitive(GTK_WIDGET(BondZ1),0);
	gtk_widget_set_sensitive(GTK_WIDGET(BondX2),0);
	gtk_widget_set_sensitive(GTK_WIDGET(BondY2),0);
	gtk_widget_set_sensitive(GTK_WIDGET(BondZ2),0);
	gcBondList::iterator j;
	alist = NULL;
	if (m_pBonds->empty()) gtk_widget_set_sensitive(GTK_WIDGET(BondDeleteAllBtn),0);
	else for (j = m_pBonds->begin(); j != m_pBonds->end(); j++)
	{
		m_pBond = new gcBond(**j);
		switch(m_pBond->Type())
		{
		case edges: strncpy(m_buf, _("Cell edges"), sizeof(m_buf)); break;
		case diagonals: strncpy(m_buf, _("Diagonals"), sizeof(m_buf)); break;
		case medians: strncpy(m_buf, _("Medians"), sizeof(m_buf)); break;
		default:
			snprintf(m_buf, sizeof(m_buf), _("Begin:%g,%g,%g end:%g,%g,%g"), m_pBond->X1(), m_pBond->Y1(), m_pBond->Z1(), m_pBond->X2(), m_pBond->Y2(), m_pBond->Z2());
		}
		alist = g_list_append(alist, gtk_list_item_new_with_label(m_buf));
		m_mBonds[m_buf] = m_pBond;
	}
	gtk_list_append_items(BondList, alist);
	
//Size page
	gdouble x0, y0, z0, x1, y1, z1;
	m_pDoc->GetSize(&x0, &x1, &y0, &y1, &z0, &z1);
	snprintf(m_buf, sizeof(m_buf), "%g", x0);
	gtk_entry_set_text(MinX, m_buf);
	snprintf(m_buf, sizeof(m_buf), "%g", x1);
	gtk_entry_set_text(MaxX, m_buf);
	snprintf(m_buf, sizeof(m_buf), "%g", y0);
	gtk_entry_set_text(MinY, m_buf);
	snprintf(m_buf, sizeof(m_buf), "%g", y1);
	gtk_entry_set_text(MaxY, m_buf);
	snprintf(m_buf, sizeof(m_buf), "%g", z0);
	gtk_entry_set_text(MinZ, m_buf);
	snprintf(m_buf, sizeof(m_buf), "%g", z1);
	gtk_entry_set_text(MaxZ, m_buf);

//Cleavages page
	gcCleavageList::iterator k;
	int mh, ml, mk;
	alist = NULL;
	if (m_pCleavages->empty()) gtk_widget_set_sensitive(GTK_WIDGET(CleavageDeleteAllBtn),0);
	else for (k = m_pCleavages->begin(); k != m_pCleavages->end(); k++)
	{
		m_pCleavage = new gcCleavage(**k);
		mh = m_pCleavage->h();
		mk = m_pCleavage->k();
		ml = m_pCleavage->l();
		snprintf(m_buf, sizeof(m_buf), _("h:%d, k:%d, l:%d"), mh, mk, ml);
		alist = g_list_append(alist, gtk_list_item_new_with_label(m_buf));
		m_mCleavages[m_buf] = m_pCleavage;
	}
	gtk_list_append_items(CleavageList, alist);
	
	gtk_widget_show_all(GTK_WIDGET(dialog));
}

gcLatticeDlg::~gcLatticeDlg()
{
	m_pDoc->RemoveDialog(this);
	if (mendeleiev) ((gcMendeleievDlg*)mendeleiev)->Destroy();
	//delete atoms
	std::map<std::string,gcAtom*>::iterator i ;
	for (i = m_mAtoms.begin() ; i != m_mAtoms.end() ; i++) delete (*i).second;
	m_mAtoms.clear();
	//delete bonds
	std::map<std::string,gcBond*>::iterator j ;
	for (j = m_mBonds.begin() ; j != m_mBonds.end() ; j++) delete (*j).second;
	m_mBonds.clear();
	//delete cleavages
	std::map<std::string,gcCleavage*>::iterator k ;
	for (k = m_mCleavages.begin() ; k != m_mCleavages.end() ; k++) delete (*k).second;
	m_mCleavages.clear();
}

bool gcLatticeDlg::Apply()
{
	gdouble x0, y0, z0, x1, y1, z1, x;
//Unit cell and size are treated first because the method must fail if an entry is incorrect
	//Unit cell
	switch(m_Lattice)
	{
	case cubic:
	case body_centered_cubic:
	case face_centered_cubic:
		if (!GetNumber(A, &m_a, gccMin, 0))
		{
			gtk_notebook_set_page(notebook, 0);
			return false;
		}
		m_alpha = m_beta = m_gamma  = 90;
		m_b = m_c = m_a;
		break;
	case hexagonal:
		if ((!GetNumber(A, &m_a, gccMin, 0)) ||
			(!GetNumber(C, &m_c, gccMin, 0)))
		{
			gtk_notebook_set_page(notebook, 0);
			return false;
		}
		m_alpha = m_beta = 90;
		m_gamma  = 120;
		m_b = m_a;
		break;
	case tetragonal:
	case body_centered_tetragonal:
		if ((!GetNumber(A, &m_a, gccMin, 0)) ||
			(!GetNumber(C, &m_c, gccMin, 0)))
		{
			gtk_notebook_set_page(notebook, 0);
			return false;
		}
		m_alpha = m_beta = m_gamma  = 90;
		m_b = m_a;
		break;
	case orthorhombic:
	case base_centered_orthorhombic:
	case body_centered_orthorhombic:
	case face_centered_orthorhombic:
		if ((!GetNumber(A, &m_a, gccMin, 0)) ||
			(!GetNumber(B, &m_b, gccMin, 0)) ||
			(!GetNumber(C, &m_c, gccMin, 0)))
		{
			gtk_notebook_set_page(notebook, 0);
			return false;
		}
		m_alpha = m_beta = m_gamma  = 90;
		break;
	case rhombohedral:
		if ((!GetNumber(A, &m_a, gccMin, 0)) ||
			(!GetNumber(Alpha, &m_alpha, gccMinMax, 0, 180)))
		{
			gtk_notebook_set_page(notebook, 0);
			return false;
		}
		m_beta = m_gamma = m_alpha;
		m_b = m_c = m_a;
		break;
	case monoclinic:
	case base_centered_monoclinic:
		if ((!GetNumber(A, &m_a, gccMin, 0)) ||
			(!GetNumber(B, &m_b, gccMin, 0)) ||
			(!GetNumber(C, &m_c, gccMin, 0)) ||
			(!GetNumber(Beta, &m_beta, gccMinMax, 0, 180)))
		{
			gtk_notebook_set_page(notebook, 0);
			return false;
		}
		m_alpha = m_gamma  = 90;
		break;
	case triclinic:
		if ((!GetNumber(A, &m_a, gccMin, 0)) ||
			(!GetNumber(B, &m_b, gccMin, 0)) ||
			(!GetNumber(C, &m_c, gccMin, 0)) ||
			(!GetNumber(Alpha, &m_alpha, gccMinMax, 0, 180)) ||
			(!GetNumber(Beta, &m_beta, gccMinMax, 0, 180)) ||
			(!GetNumber(Gamma, &m_gamma, gccMinMax, 0, 180)))
		{
			gtk_notebook_set_page(notebook, 0);
			return false;
		}
		break;
	}
	if (m_alpha + m_beta + m_gamma >= 360)
	{
		strncpy(m_buf, _("The sum of the three angles must be less than 360"), sizeof(m_buf));
		GnomeDialog* box = (GnomeDialog*)gnome_error_dialog_parented(m_buf, GTK_WINDOW(dialog));
		gnome_dialog_run(box);
	}
	
//Size
	if ((!GetNumber(MinX, &x0)) ||
		(!GetNumber(MaxX, &x1)) ||
		(!GetNumber(MinY, &y0)) ||
		(!GetNumber(MaxY, &y1)) ||
		(!GetNumber(MinZ, &z0)) ||
		(!GetNumber(MaxZ, &z1)))
	{
		gtk_notebook_set_page(notebook, 3);
		return false;
	}
	if (x0 > x1) {x = x0; x0 = x1; x1 = x;}
	if (y0 > y1) {x = y0; y0 = y1; y1 = x;}
	if (z0 > z1) {x = z0; z0 = z1; z1 = x;}
	m_pDoc->SetSize(x0, x1, y0, y1, z0, z1);
	m_pDoc->SetCell(m_Lattice, m_a, m_b, m_c, m_alpha, m_beta, m_gamma);

//Atoms
	//First, delete old atoms
	while (!m_pAtoms->empty())
	{
		delete m_pAtoms->front();
		m_pAtoms->pop_front();
	}
	//Add new atoms from the map
	std::map<std::string,gcAtom*>::iterator i ;
	for (i = m_mAtoms.begin() ; i != m_mAtoms.end() ; i++)
	{
		m_pAtom = new gcAtom(*((*i).second));//create a new atom to avoid conflicts
		m_pAtoms->push_back(m_pAtom);
	}
	
	//Bonds
	//First, delete old bonds
	while (!m_pBonds->empty())
	{
		delete m_pBonds->front();
		m_pBonds->pop_front();
	}
	//Add new bonds from the map
	std::map<std::string,gcBond*>::iterator j ;
	for (j = m_mBonds.begin() ; j != m_mBonds.end() ; j++)
	{
		m_pBond = new gcBond(*((*j).second));//create a new bond to avoid conflicts
		m_pBonds->push_back(m_pBond);
	}
	
	//Cleavages
	//First, delete old cleavages
	while (!m_pCleavages->empty())
	{
		delete m_pCleavages->front();
		m_pCleavages->pop_front();
	}
	m_pDoc->SetFixedSize(gtk_toggle_button_get_active(FixedBtn));
	//Add new cleavages from the map
	std::map<std::string,gcCleavage*>::iterator k ;
	for (k = m_mCleavages.begin() ; k != m_mCleavages.end() ; k++)
	{
		m_pCleavage = new gcCleavage(*((*k).second));//create a new cleavage to avoid conflicts
		m_pCleavages->push_back(m_pCleavage);
	}
	
	m_pDoc->Update();
	m_pDoc->SetDirty();
	return true;
}

void gcLatticeDlg::AtomAdd()
{
	if ((m_pAtom = GetNewAtom()) == NULL) return;
	if (m_mAtoms[m_buf] != NULL)
	{
		delete m_mAtoms[m_buf];
		m_mAtoms[m_buf] = m_pAtom;
	}
	else
	{
		m_mAtoms[m_buf] = m_pAtom;
		GtkListItem* item = GTK_LIST_ITEM(gtk_list_item_new_with_label(m_buf));
		GList* ilist = g_list_append(NULL, GTK_WIDGET(item));
		gtk_list_append_items(AtomList, ilist);
		gtk_list_select_item(AtomList, gtk_list_child_position(AtomList, GTK_WIDGET(item)));
		gtk_widget_set_sensitive(GTK_WIDGET(AtomDeleteBtn),1);
		gtk_widget_set_sensitive(GTK_WIDGET(AtomModifyBtn),1);
		gtk_widget_set_sensitive(GTK_WIDGET(AtomDeleteAllBtn),1);
		gtk_widget_show_all(GTK_WIDGET(dialog));
	}
}

void gcLatticeDlg::AtomDelete()
{
	unsigned pos;
	char*tmp = GetListText(AtomList, &pos);
	if (!tmp) return;
	delete m_mAtoms[tmp];
	m_mAtoms.erase(tmp);
	gtk_list_clear_items(AtomList, pos, pos + 1);
}

void gcLatticeDlg::AtomModify()
{
	unsigned pos;
	char*tmp = GetListText(AtomList, &pos);
	if ((!tmp) || ((m_pAtom = GetNewAtom()) == NULL)) return;
	delete m_mAtoms[tmp];
	m_mAtoms.erase(tmp);
	m_mAtoms[m_buf] = m_pAtom;
	gtk_list_clear_items(AtomList, pos, pos + 1);
	GtkListItem* item = GTK_LIST_ITEM(gtk_list_item_new_with_label(m_buf));
	GList* ilist = g_list_append(NULL, GTK_WIDGET(item));
	gtk_list_insert_items(AtomList, ilist, pos);
	gtk_list_select_item(AtomList, gtk_list_child_position(AtomList, GTK_WIDGET(item)));
	gtk_widget_show_all(GTK_WIDGET(dialog));
}

void gcLatticeDlg::AtomDeleteAll()
{
	int n = m_mAtoms.size();
	std::map<std::string,gcAtom*>::iterator i ;
	for (i = m_mAtoms.begin() ; i != m_mAtoms.end() ; i++) delete (*i).second;
	m_mAtoms.clear();
	gtk_list_clear_items(AtomList, 0, n);
	gtk_widget_set_sensitive(GTK_WIDGET(AtomDeleteAllBtn),0);
}

void gcLatticeDlg::AtomSelect()
{
	char* tmp = GetListText(AtomList);
	if (!tmp)
	{
		gtk_widget_set_sensitive(GTK_WIDGET(AtomDeleteBtn),0);
		gtk_widget_set_sensitive(GTK_WIDGET(AtomModifyBtn),0);
		if (m_mAtoms.empty()) gtk_widget_set_sensitive(GTK_WIDGET(AtomDeleteAllBtn),0);
		return;
	}
	gtk_widget_set_sensitive(GTK_WIDGET(AtomDeleteBtn),1);
	gtk_widget_set_sensitive(GTK_WIDGET(AtomModifyBtn),1);
	m_pAtom = m_mAtoms[tmp];
	snprintf(m_buf, sizeof(m_buf), "%g", m_pAtom->x());
	gtk_entry_set_text(AtomX, m_buf);
	snprintf(m_buf, sizeof(m_buf), "%g", m_pAtom->y());
	gtk_entry_set_text(AtomY, m_buf);
	snprintf(m_buf, sizeof(m_buf), "%g", m_pAtom->z());
	gtk_entry_set_text(AtomZ, m_buf);
	snprintf(m_buf, sizeof(m_buf), "%g", m_pAtom->r());
	gtk_entry_set_text(AtomR, m_buf);
	m_pAtom->GetColor(&red, &green, &blue, &alpha);
	gnome_color_picker_set_d(AtomColor, red, green, blue, alpha);
	GtkLabel* label = GTK_LABEL(GTK_BIN(ElementBtn)->child);
	gtk_label_set(label,m_pAtom->Symbol());
	m_Z = m_pAtom->Z();
}

gcAtom* gcLatticeDlg::GetNewAtom()
{
	if (!GetNumber(AtomX, &x, gccMinEqMax, 0, 1)) return NULL;
	if (!GetNumber(AtomY, &y, gccMinEqMax, 0, 1)) return NULL;
	if (!GetNumber(AtomZ, &z, gccMinEqMax, 0, 1)) return NULL;
	if (!GetNumber(AtomR, &r, gccMin, 0)) return NULL;
	if (r <= 0) return NULL;
	gnome_color_picker_get_d(AtomColor, &red, &green, &blue, &alpha);
	gcAtom* pAtom = new gcAtom(m_Z, x, y, z, r, red, green, blue, alpha);
	if (m_Z) snprintf(m_buf, sizeof(m_buf), "%s x=%g,y=%g,z=%g", pAtom->Symbol(), x, y, z);
	else snprintf(m_buf, sizeof(m_buf), "x=%g,y=%g,z=%g", x, y, z);
	return pAtom;
}

char* gcLatticeDlg::GetListText(GtkList* plist, unsigned * pos)
{
	if (plist->selection == NULL) return NULL;
	GtkContainer* item = (GtkContainer*) plist->selection->data;
	if (item == NULL) return NULL;
	GList* ItemList = gtk_container_children(GTK_CONTAINER(item));
	GtkLabel* label = (GtkLabel*)ItemList->data;
	if (pos) *pos = gtk_list_child_position(plist, GTK_WIDGET(item));
	char* tmp;
	gtk_label_get(label, &tmp);
	return tmp;
}

void gcLatticeDlg::LatticeSelect(GtkWidget *widget)
{
	GtkMenu* menu = GTK_MENU(LatticeMenu->menu);
	GList* list = menu->menu_shell.children;
	gcLattices i = cubic;
	while (list)
	{
		if (widget == GTK_WIDGET(list->data)) break;
		list = g_list_next(list);
		i = gcLattices(i + 1);
	}
	if (!list) return;//that should not happen!
	m_Lattice = i;
	UpdateCell();
}

void gcLatticeDlg::UpdateCell()
{
	switch(m_Lattice)
	{
	case cubic:
	case body_centered_cubic:
	case face_centered_cubic:
		gtk_entry_set_text(Alpha, "90");
		gtk_entry_set_text(Beta, "90");
		gtk_entry_set_text(Gamma, "90");
		gtk_widget_set_sensitive(GTK_WIDGET(Alpha),0);
		gtk_widget_set_sensitive(GTK_WIDGET(Beta),0);
		gtk_widget_set_sensitive(GTK_WIDGET(Gamma),0);
		gtk_widget_set_sensitive(GTK_WIDGET(B),0);
		gtk_widget_set_sensitive(GTK_WIDGET(C),0);
		break;
	case hexagonal:
		gtk_entry_set_text(Alpha, "90");
		gtk_entry_set_text(Beta, "90");
		gtk_entry_set_text(Gamma, "120");
		gtk_widget_set_sensitive(GTK_WIDGET(Alpha),0);
		gtk_widget_set_sensitive(GTK_WIDGET(Beta),0);
		gtk_widget_set_sensitive(GTK_WIDGET(Gamma),0);
		gtk_widget_set_sensitive(GTK_WIDGET(B),0);
		gtk_widget_set_sensitive(GTK_WIDGET(C),1);
		break;
	case tetragonal:
	case body_centered_tetragonal:
		gtk_entry_set_text(Alpha, "90");
		gtk_entry_set_text(Beta, "90");
		gtk_entry_set_text(Gamma, "90");
		gtk_widget_set_sensitive(GTK_WIDGET(Alpha),0);
		gtk_widget_set_sensitive(GTK_WIDGET(Beta),0);
		gtk_widget_set_sensitive(GTK_WIDGET(Gamma),0);
		gtk_widget_set_sensitive(GTK_WIDGET(B),0);
		gtk_widget_set_sensitive(GTK_WIDGET(C),1);
		break;
	case orthorhombic:
	case base_centered_orthorhombic:
	case body_centered_orthorhombic:
	case face_centered_orthorhombic:
		gtk_entry_set_text(Alpha, "90");
		gtk_entry_set_text(Beta, "90");
		gtk_entry_set_text(Gamma, "90");
		gtk_widget_set_sensitive(GTK_WIDGET(Alpha),0);
		gtk_widget_set_sensitive(GTK_WIDGET(Beta),0);
		gtk_widget_set_sensitive(GTK_WIDGET(Gamma),0);
		gtk_widget_set_sensitive(GTK_WIDGET(B),1);
		gtk_widget_set_sensitive(GTK_WIDGET(C),1);
		break;
	case rhombohedral:
		gtk_widget_set_sensitive(GTK_WIDGET(Alpha),1);
		gtk_widget_set_sensitive(GTK_WIDGET(Beta),0);
		gtk_widget_set_sensitive(GTK_WIDGET(Gamma),0);
		gtk_widget_set_sensitive(GTK_WIDGET(B),0);
		gtk_widget_set_sensitive(GTK_WIDGET(C),0);
		break;
	case monoclinic:
	case base_centered_monoclinic:
		gtk_entry_set_text(Alpha, "90");
		gtk_entry_set_text(Gamma, "90");
		gtk_widget_set_sensitive(GTK_WIDGET(Alpha),0);
		gtk_widget_set_sensitive(GTK_WIDGET(Beta),1);
		gtk_widget_set_sensitive(GTK_WIDGET(Gamma),0);
		gtk_widget_set_sensitive(GTK_WIDGET(B),1);
		gtk_widget_set_sensitive(GTK_WIDGET(C),1);
		break;
	case triclinic:
		gtk_widget_set_sensitive(GTK_WIDGET(Alpha),1);
		gtk_widget_set_sensitive(GTK_WIDGET(Beta),1);
		gtk_widget_set_sensitive(GTK_WIDGET(Gamma),1);
		gtk_widget_set_sensitive(GTK_WIDGET(B),1);
		gtk_widget_set_sensitive(GTK_WIDGET(C),1);
		break;
	}
}

void gcLatticeDlg::OnElement()
{
	mendeleiev = new gcMendeleievDlg(this, m_Z);
}

void gcLatticeDlg::SetElement(int Z)
{
	m_Z = Z;
	GtkLabel* label = GTK_LABEL(GTK_BIN(ElementBtn)->child);
	if (Z) gtk_label_set(label, Elt[Z]->Symbol());
	else gtk_label_set(label, "");
}

void gcLatticeDlg::BondAdd()
{
	if ((m_pBond = GetNewBond()) == NULL) return;
	if (m_mBonds[m_buf] != NULL)
	{
		delete m_mBonds[m_buf];
		m_mBonds[m_buf] = m_pBond;
	}
	else
	{
		m_mBonds[m_buf] = m_pBond;
		GtkListItem* item = GTK_LIST_ITEM(gtk_list_item_new_with_label(m_buf));
		GList* ilist = g_list_append(NULL, GTK_WIDGET(item));
		gtk_list_append_items(BondList, ilist);
		gtk_list_select_item(BondList, gtk_list_child_position(BondList, GTK_WIDGET(item)));
		gtk_widget_set_sensitive(GTK_WIDGET(BondDeleteBtn),1);
		gtk_widget_set_sensitive(GTK_WIDGET(BondModifyBtn),1);
		gtk_widget_set_sensitive(GTK_WIDGET(BondDeleteAllBtn),1);
		gtk_widget_show_all(GTK_WIDGET(dialog));
	}
}

void gcLatticeDlg::BondDelete()
{
	unsigned pos;
	char*tmp = GetListText(BondList, &pos);
	if (!tmp) return;
	delete m_mBonds[tmp];
	m_mBonds.erase(tmp);
	gtk_list_clear_items(BondList, pos, pos + 1);
}

void gcLatticeDlg::BondModify()
{
	unsigned pos;
	char*tmp = GetListText(BondList, &pos);
	if ((!tmp) || ((m_pBond = GetNewBond()) == NULL)) return;
	delete m_mBonds[tmp];
	m_mBonds.erase(tmp);
	m_mBonds[m_buf] = m_pBond;
	gtk_list_clear_items(BondList, pos, pos + 1);
	GtkListItem* item = GTK_LIST_ITEM(gtk_list_item_new_with_label(m_buf));
	GList* ilist = g_list_append(NULL, GTK_WIDGET(item));
	gtk_list_insert_items(BondList, ilist, pos);
	gtk_list_select_item(BondList, gtk_list_child_position(BondList, GTK_WIDGET(item)));
	gtk_widget_show_all(GTK_WIDGET(dialog));
}

void gcLatticeDlg::BondDeleteAll()
{
	int n = m_mBonds.size();
	std::map<std::string,gcBond*>::iterator i ;
	for (i = m_mBonds.begin() ; i != m_mBonds.end() ; i++) delete (*i).second;
	m_mBonds.clear();
	gtk_list_clear_items(BondList, 0, n);
	gtk_widget_set_sensitive(GTK_WIDGET(BondDeleteAllBtn),0);
}

void gcLatticeDlg::BondSelect()
{
	char* tmp = GetListText(BondList);
	if (!tmp)
	{
		gtk_widget_set_sensitive(GTK_WIDGET(BondDeleteBtn),0);
		gtk_widget_set_sensitive(GTK_WIDGET(BondModifyBtn),0);
		if (m_mBonds.empty()) gtk_widget_set_sensitive(GTK_WIDGET(BondDeleteAllBtn),0);
		return;
	}
	gtk_widget_set_sensitive(GTK_WIDGET(BondDeleteBtn),1);
	gtk_widget_set_sensitive(GTK_WIDGET(BondModifyBtn),1);
	m_pBond = m_mBonds[tmp];
	switch (m_pBond->Type())
	{
	case edges: gtk_toggle_button_set_state(Btn0, 1); break;
	case diagonals: gtk_toggle_button_set_state(Btn1, 1); break;
	case medians: gtk_toggle_button_set_state(Btn2, 1); break;
	case bond: gtk_toggle_button_set_state(Btn3, 1); break;
	case unique: gtk_toggle_button_set_state(Btn4, 1); break;
	}
	if (m_pBond->Type() > medians)
	{
		snprintf(m_buf, sizeof(m_buf), "%g", m_pBond->X1());
		gtk_entry_set_text(BondX1, m_buf);
		snprintf(m_buf, sizeof(m_buf), "%g", m_pBond->Y1());
		gtk_entry_set_text(BondY1, m_buf);
		snprintf(m_buf, sizeof(m_buf), "%g", m_pBond->Z1());
		gtk_entry_set_text(BondZ1, m_buf);
		snprintf(m_buf, sizeof(m_buf), "%g", m_pBond->X2());
		gtk_entry_set_text(BondX2, m_buf);
		snprintf(m_buf, sizeof(m_buf), "%g", m_pBond->Y2());
		gtk_entry_set_text(BondY2, m_buf);
		snprintf(m_buf, sizeof(m_buf), "%g", m_pBond->Z2());
		gtk_entry_set_text(BondZ2, m_buf);	}
	snprintf(m_buf, sizeof(m_buf), "%g", m_pBond->GetRadius());
	gtk_entry_set_text(BondR, m_buf);
	m_pBond->GetColor(&red, &green, &blue, &alpha);
	gnome_color_picker_set_d(BondColor, red, green, blue, alpha);
}

void gcLatticeDlg::BondTypeChanged()
{
	gcBondType nType;
	if (Btn0->active) nType = edges;
	else if (Btn1->active) nType = diagonals;
	else if (Btn2->active) nType = medians;
	else if (Btn3->active) nType = bond;
	else if (Btn4->active) nType = unique;
	if (nType == m_BondType) return;
	if (nType > medians)
	{
		gtk_widget_set_sensitive(GTK_WIDGET(BondX1),1);
		gtk_widget_set_sensitive(GTK_WIDGET(BondY1),1);
		gtk_widget_set_sensitive(GTK_WIDGET(BondZ1),1);
		gtk_widget_set_sensitive(GTK_WIDGET(BondX2),1);
		gtk_widget_set_sensitive(GTK_WIDGET(BondY2),1);
		gtk_widget_set_sensitive(GTK_WIDGET(BondZ2),1);
	}
	else
	{
		gtk_widget_set_sensitive(GTK_WIDGET(BondX1),0);
		gtk_widget_set_sensitive(GTK_WIDGET(BondY1),0);
		gtk_widget_set_sensitive(GTK_WIDGET(BondZ1),0);
		gtk_widget_set_sensitive(GTK_WIDGET(BondX2),0);
		gtk_widget_set_sensitive(GTK_WIDGET(BondY2),0);
		gtk_widget_set_sensitive(GTK_WIDGET(BondZ2),0);
	}
	m_BondType = nType;
}

void gcLatticeDlg::CleavageAdd()
{
	if ((m_pCleavage = GetNewCleavage()) == NULL) return;
	if (m_mCleavages[m_buf] != NULL)
	{
		delete m_mCleavages[m_buf];
		m_mCleavages[m_buf] = m_pCleavage;
	}
	else
	{
		m_mCleavages[m_buf] = m_pCleavage;
		GtkListItem* item = GTK_LIST_ITEM(gtk_list_item_new_with_label(m_buf));
		GList* ilist = g_list_append(NULL, GTK_WIDGET(item));
		gtk_list_append_items(CleavageList, ilist);
		gtk_list_select_item(CleavageList, gtk_list_child_position(CleavageList, GTK_WIDGET(item)));
		gtk_widget_set_sensitive(GTK_WIDGET(CleavageDeleteBtn),1);
		gtk_widget_set_sensitive(GTK_WIDGET(CleavageModifyBtn),1);
		gtk_widget_set_sensitive(GTK_WIDGET(CleavageDeleteAllBtn),1);
		gtk_widget_show_all(GTK_WIDGET(dialog));
	}
}

void gcLatticeDlg::CleavageDelete()
{
	unsigned pos;
	char*tmp = GetListText(CleavageList, &pos);
	if (!tmp) return;
	delete m_mCleavages[tmp];
	m_mCleavages.erase(tmp);
	gtk_list_clear_items(CleavageList, pos, pos + 1);
}

void gcLatticeDlg::CleavageModify()
{
	unsigned pos;
	char*tmp = GetListText(CleavageList, &pos);
	if ((!tmp) || ((m_pCleavage = GetNewCleavage()) == NULL)) return;
	delete m_mCleavages[tmp];
	m_mCleavages.erase(tmp);
	m_mCleavages[m_buf] = m_pCleavage;
	gtk_list_clear_items(CleavageList, pos, pos + 1);
	GtkListItem* item = GTK_LIST_ITEM(gtk_list_item_new_with_label(m_buf));
	GList* ilist = g_list_append(NULL, GTK_WIDGET(item));
	gtk_list_insert_items(CleavageList, ilist, pos);
	gtk_list_select_item(CleavageList, gtk_list_child_position(CleavageList, GTK_WIDGET(item)));
	gtk_widget_show_all(GTK_WIDGET(dialog));
}

void gcLatticeDlg::CleavageDeleteAll()
{
	int n = m_mCleavages.size();
	std::map<std::string,gcCleavage*>::iterator i ;
	for (i = m_mCleavages.begin() ; i != m_mCleavages.end() ; i++) delete (*i).second;
	m_mCleavages.clear();
	gtk_list_clear_items(CleavageList, 0, n);
	gtk_widget_set_sensitive(GTK_WIDGET(CleavageDeleteAllBtn),0);
}

void gcLatticeDlg::CleavageSelect()
{
	char* tmp = GetListText(CleavageList);
	if (!tmp)
	{
		gtk_widget_set_sensitive(GTK_WIDGET(CleavageDeleteBtn),0);
		gtk_widget_set_sensitive(GTK_WIDGET(CleavageModifyBtn),0);
		if (m_mCleavages.empty()) gtk_widget_set_sensitive(GTK_WIDGET(CleavageDeleteAllBtn),0);
		return;
	}
	gtk_widget_set_sensitive(GTK_WIDGET(CleavageDeleteBtn),1);
	gtk_widget_set_sensitive(GTK_WIDGET(CleavageModifyBtn),1);
	m_pCleavage = m_mCleavages[tmp];
	int n = m_pCleavage->h();
	gtk_spin_button_set_value(Btnh, n);
	n = m_pCleavage->k();
	gtk_spin_button_set_value(Btnk, n);
	n = m_pCleavage->l();
	gtk_spin_button_set_value(Btnl, n);
	n = m_pCleavage->Planes();
	gtk_spin_button_set_value(Btnn, n);
}

gcBond* gcLatticeDlg::GetNewBond()
{
	if (m_BondType > medians)
	{
		if (!GetNumber(BondX1, &x)) return NULL;
		if (!GetNumber(BondY1, &y)) return NULL;
		if (!GetNumber(BondZ1, &z)) return NULL;
		if (!GetNumber(BondX2, &x1)) return NULL;
		if (!GetNumber(BondY2, &y1)) return NULL;
		if (!GetNumber(BondZ2, &z1)) return NULL;
	}
	if (!GetNumber(BondR, &r, gccMin, 0)) return NULL;
	if (r <= 0) return NULL;
	gnome_color_picker_get_d(BondColor, &red, &green, &blue, &alpha);
	gcBond* pBond = new gcBond(m_BondType, x, y, z, x1, y1, z1, r, red, green, blue, alpha);
	switch(pBond->Type())
	{
	case edges: strncpy(m_buf, _("Cell edges"), sizeof(m_buf)); break;
	case diagonals: strncpy(m_buf, _("Diagonals"), sizeof(m_buf)); break;
	case medians: strncpy(m_buf, _("Medians"), sizeof(m_buf)); break;
	default:
		snprintf(m_buf, sizeof(m_buf), _("Begin:%g,%g,%g end:%g,%g,%g"), pBond->X1(), pBond->Y1(), pBond->Z1(), pBond->X2(), pBond->Y2(), pBond->Z2());
	}
	return pBond;
}

gcCleavage* gcLatticeDlg::GetNewCleavage()
{
	int h, k, l, n;
	h = gtk_spin_button_get_value_as_int(Btnh);
	k = gtk_spin_button_get_value_as_int(Btnk);
	l = gtk_spin_button_get_value_as_int(Btnl);
	n = gtk_spin_button_get_value_as_int(Btnn);
	if (!n || (!h && !k &&!l)) return NULL;
	gcCleavage* pCleavage = new gcCleavage();
	pCleavage->h() = h;
	pCleavage->k() = k;
	pCleavage->l() = l;
	pCleavage->Planes() = n;
	snprintf(m_buf, sizeof(m_buf), _("h:%d, k:%d, l:%d"), h, k, l);
	return pCleavage;
}
