/* AppletViewer.java -- a standalone viewer for Java applets
   Copyright (C) 2003, 2004, 2006  Thomas Fitzsimmons <fitzsim@redhat.com>

   This file is part of GCJ Applet Viewer.

   GCJ Applet Viewer is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   GCJ Applet Viewer is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with GCJ Applet Viewer; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

package gnu.gcjwebplugin;

import gnu.getopt.Getopt;
import gnu.getopt.LongOpt;
import java.applet.Applet;
import java.awt.Dimension;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.MissingResourceException;
import java.util.ResourceBundle;


class AppletViewer
{
  /**
   * The localized strings are kept in a separate file.
   */
  public static final ResourceBundle messages = ResourceBundle.getBundle
    ("gnu.gcjwebplugin.locale.MessagesBundle");

  private static HashMap classLoaderCache = new HashMap();
  
  private static ClassLoader getClassLoader(URL codebase, ArrayList archives)
  {
    // Should load class loader each time. It is possible that there
    // are more than one applet to be loaded with different archives.
    AppletClassLoader loader = new AppletClassLoader(codebase, archives);
    classLoaderCache.put(codebase, loader);

    return loader;
  }
  
  static Applet createApplet(AppletTag tag)
  {
    Applet applet = null;

    try
      {
        ClassLoader loader = getClassLoader(tag.prependCodeBase(""),
                                            tag.getArchives());
        String code = tag.getCode();

        if (code.endsWith(".class"))
          code = code.substring(0, code.length() - 6).replace('/', '.');

        Class c = loader.loadClass(code);
        applet = (Applet) c.newInstance();
      }
    catch (Exception e)
      {
        e.printStackTrace();
      }

    if (applet == null)
      applet = new ErrorApplet("Error loading applet");

    return applet;
  }

  static void printVersion()
  {
    System.out.println("gcjappletviewer (GCJ Applet Viewer) "
                       + Configuration.VERSION);
    System.exit(0);
  }

  static void printHelpMessage(int status)
  {
    try
      {
	for (int count = 0; ; count++)
	  System.out.println
	    (AppletViewer.messages.getString("gcjwebplugin.help." + count));
      }
    catch (MissingResourceException e)
      {
	// Do nothing here.
      }
    
    System.exit(status);
  }

  protected static boolean verbose;

  /**
   * The main method starting the applet viewer.
   *
   * @param args the arguments given on the command line.
   *
   * @exception IOException if an error occurs.
   */
  public static void main(String[] args) throws IOException
  {
    if (args.length == 0)
      {
	System.err.println(AppletViewer.messages.getString
			   ("gcjwebplugin.no_input_files"));
	printHelpMessage(1);
      }

    List argsList = new LinkedList(Arrays.asList(args));

    Iterator argsIter = argsList.iterator();

    // Check for the -encoding option, for appletviewer compatibility.
    // The -encoding option can be given multiple times on the command
    // line.  The last occurence wins.

    // FIXME: We should probably be using
    // java.nio.charset.CharsetDecoder to handle the encoding.  What
    // is the status of Classpath's implementation?
    argsIter = argsList.iterator();
    while (argsIter.hasNext())
      {
	String currentArg = (String) argsIter.next();
	if (currentArg.equals("--"))
	  break;
	else if (currentArg.equals("-encoding"))
	  {
	    if (! argsIter.hasNext())
	      {
		System.err.println("gcjappletviewer: option '-encoding' requires an argument");
		printHelpMessage(1);
	      }

	    // Remove the -encoding option.
	    argsIter.remove();
	    // Remove the argument to -encoding.
	    argsIter.remove();
	  }
      }

    args = (String[]) argsList.toArray(new String[argsList.size()]);

    LongOpt[] longOptions =
      new LongOpt[]
      {
        new LongOpt("code", LongOpt.REQUIRED_ARGUMENT, null, 0),
        new LongOpt("codebase", LongOpt.REQUIRED_ARGUMENT, null, 0),
        new LongOpt("archive", LongOpt.REQUIRED_ARGUMENT, null, 0),
        new LongOpt("param", LongOpt.REQUIRED_ARGUMENT, null, 0),
        new LongOpt("width", LongOpt.REQUIRED_ARGUMENT, null, 0),
        new LongOpt("height", LongOpt.REQUIRED_ARGUMENT, null, 0),
        new LongOpt("plugin", LongOpt.REQUIRED_ARGUMENT, null, 0),
        new LongOpt("version", LongOpt.NO_ARGUMENT, null, 0),
        new LongOpt("help", LongOpt.NO_ARGUMENT, null, 0),
        new LongOpt("verbose", LongOpt.NO_ARGUMENT, null, 0),
      };

    Getopt opts = new Getopt("gcjappletviewer", args, "", longOptions);

    int optionValue;
    int optionIndex;

    String code = "";
    String codebase = "";
    String archives = "";
    List parameters = new ArrayList();
    Dimension dimensions = new Dimension(-1, -1);
    String pipeInName = "";
    String pipeOutName = "";

    boolean pluginMode = false;

    // This loop parses the command line, placing all option arguments
    // at the start of args, all non-options at the end, and setting
    // optind to the index of the first non-option argument.
    while ((optionValue = opts.getopt()) != -1)
      {
	if (optionValue == 0)
	  {
	    optionIndex = opts.getLongind();

            // --code
	    if (optionIndex == 0)
              code = opts.getOptarg();
            // --codebase
	    if (optionIndex == 1)
	      codebase = opts.getOptarg();
            // --archive
            if (optionIndex == 2)
	      archives = opts.getOptarg();
            // --param
	    else if (optionIndex == 3)
	      parameters.add(opts.getOptarg());
            // --width
	    else if (optionIndex == 4)
              dimensions.width =
                Math.max(1, Integer.parseInt(opts.getOptarg()));
            // --height
	    else if (optionIndex == 5)
              dimensions.height =
                Math.max(1, Integer.parseInt(opts.getOptarg()));
            // --plugin
	    else if (optionIndex == 6)
	      {
		pluginMode = true;
		String tmp = opts.getOptarg();
		int pos = tmp.indexOf(',');
		pipeInName = tmp.substring(0, pos);
		pipeOutName = tmp.substring(pos + 1);
		break;
	      }
            // --version
	    else if (optionIndex == 7)
	      printVersion();
            // --help
	    else if (optionIndex == 8)
	      printHelpMessage(0);
            // --verbose
            else if (optionIndex == 9)
              {
                verbose = true;
              }
	  }
      }

    if (dimensions.height < 0)
      dimensions.height = 200;

    if (dimensions.width < 0)
      dimensions.width = (int) (1.6 * dimensions.height);

    //System.setSecurityManager(new AppletSecurityManager(pluginMode));

    // Print arguments.
    printArguments(args);

    if (pluginMode)
      {
	InputStream in;
	OutputStream out;

        in = new FileInputStream(pipeInName);
        out = new FileOutputStream(pipeOutName);

	PluginAppletViewer.start(in, out);
      }
    else
      {
        if (code.equals(""))
          {
            // The --code option wasn't given and there are no URL
            // arguments so we have nothing to work with.
            if (opts.getOptind() == args.length)
              {
                System.err.println(AppletViewer.messages.getString("gcjappletviewer.no_input_files"));
                printHelpMessage(1);
              }
            // Create a standalone appletviewer from a list of URLs.
            String[] urls = new String[args.length - opts.getOptind()];
            System.arraycopy(args, opts.getOptind(), urls, 0, args.length - opts.getOptind());
            new StandaloneAppletViewer(urls);
          }
        else
          {
            // Create a standalone appletviewer from the --code
            // option.
            new StandaloneAppletViewer(code, codebase, archives, parameters, dimensions);
          }
      }
  }

  static void printArguments(String[] args)
  {
    if (verbose)
      {
        System.out.println("raw arguments:");

        for (int i = 0; i < args.length; i++)
          System.out.println(" " + args[i]);
      }
  }
}
