// -*- C++ -*-

/* 
 * GChemPaint library
 * theme.cc
 *
 * Copyright (C) 2002-2005 Jean Bréfort <jean.brefort@normalesup.org>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301
 * USA
 */

#include "gchempaint-config.h"
#include "theme.h"
#include "settings.h"
#include <gcu/macros.h>
#include <glib/gi18n-lib.h>

double DefaultBondLength = 140.;
double DefaultBondAngle = 120.;
double DefaultBondDist = 5.;
double DefaultBondWidth = 1.0;
double DefaultArrowLength = 200;
double DefaultArrowHeadA = 6.0;
double DefaultArrowHeadB = 8.0;
double DefaultArrowHeadC = 4.0;
double DefaultArrowDist = 5.0;
double DefaultArrowWidth = 1.0;
double DefaultHashWidth = 1.0;
double DefaultHashDist = 2.0;
double DefaultStereoBondWidth = 5.0;
double DefaultZoomFactor = 0.25;
double DefaultPadding = 2.0;
double DefaultArrowPadding = 16.0;
double DefaultStoichiometryPadding = 1.;
double DefaultObjectPadding = 16.0;
double DefaultSignPadding = 8.0;
double DefaultChargeSignHeight = 12.;
double DefaultChargeSignSize = 9.;
double DefaultChargeYAlign = 10.;
gchar *DefaultFontFamily = NULL;
PangoStyle DefaultFontStyle = PANGO_STYLE_NORMAL;
PangoWeight DefaultFontWeight = PANGO_WEIGHT_NORMAL;
PangoVariant DefaultFontVariant = PANGO_VARIANT_NORMAL;
PangoStretch DefaultFontStretch = PANGO_STRETCH_NORMAL;
gint DefaultFontSize = 12 * PANGO_SCALE;
gchar *DefaultTextFontFamily = NULL;
PangoStyle DefaultTextFontStyle = PANGO_STYLE_NORMAL;
PangoWeight DefaultTextFontWeight = PANGO_WEIGHT_NORMAL;
PangoVariant DefaultTextFontVariant = PANGO_VARIANT_NORMAL;
PangoStretch DefaultTextFontStretch = PANGO_STRETCH_NORMAL;
gint DefaultTextFontSize = 12 * PANGO_SCALE;

gcpTheme::gcpTheme (char const *name, char const *local_name)
{
	m_ZoomFactor = DefaultZoomFactor;
	m_BondLength = DefaultBondLength;
	m_BondAngle = DefaultBondAngle;
	m_BondDist = DefaultBondDist;
	m_BondWidth = DefaultBondWidth;
	m_ArrowLength = DefaultArrowLength;
	m_HashWidth = DefaultHashWidth;
	m_HashDist = DefaultHashDist;
	m_StereoBondWidth = DefaultStereoBondWidth;
	m_Padding = DefaultPadding;
	m_ArrowHeadA = DefaultArrowHeadA;
	m_ArrowHeadB = DefaultArrowHeadB;
	m_ArrowHeadC = DefaultArrowHeadC;
	m_ArrowDist = DefaultArrowDist;
	m_ArrowPadding = DefaultArrowPadding;
	m_ArrowWidth = DefaultArrowWidth;
	m_StoichiometryPadding = DefaultStoichiometryPadding;
	m_ObjectPadding = DefaultObjectPadding;
	m_SignPadding = DefaultSignPadding;
	m_ChargeSignHeight = DefaultChargeSignHeight;
	m_ChargeSignSize = DefaultChargeSignSize;
	m_ChargeYAlign = DefaultChargeYAlign;
	m_FontFamily = DefaultFontFamily;
	m_FontStyle = DefaultFontStyle;
	m_FontWeight = DefaultFontWeight;
	m_FontVariant = DefaultFontVariant;
	m_FontStretch = DefaultFontStretch;
	m_FontSize = DefaultFontSize;
	m_TextFontFamily = DefaultTextFontFamily;
	m_TextFontStyle = DefaultTextFontStyle;
	m_TextFontWeight = DefaultTextFontWeight;
	m_TextFontVariant = DefaultTextFontVariant;
	m_TextFontStretch = DefaultTextFontStretch;
	m_TextFontSize = DefaultTextFontSize;
	m_Name = name;
	if (local_name)
		m_LocalName = local_name;
}

gcpTheme::~gcpTheme ()
{
}

gcpThemeManager ThemeManager;

static void on_config_changed (GConfClient *client, guint cnxn_id, GConfEntry *entry, gcpThemeManager *manager)
{
	manager->OnConfigChanged (client, cnxn_id, entry);
}

#define ROOTDIR "/apps/gchempaint/settings/"

// transform functions for gconf key values

static double inv (double x) {return 1 / x;}

static PangoStyle set_fontstyle (int val)
{
	switch (val) {
		case 0: return PANGO_STYLE_NORMAL;
		case 1: return PANGO_STYLE_OBLIQUE;
		case 2: return PANGO_STYLE_ITALIC;
		default: return PANGO_STYLE_NORMAL;
	}
}

static PangoWeight set_fontweight (int val)
{
	switch (val) {
		case 2: return PANGO_WEIGHT_ULTRALIGHT;
		case 3: return PANGO_WEIGHT_LIGHT;
		case 4: return PANGO_WEIGHT_NORMAL;
		case 6: return PANGO_WEIGHT_SEMIBOLD;
		case 7: return PANGO_WEIGHT_BOLD;
		case 8: return PANGO_WEIGHT_ULTRABOLD;
		case 9: return PANGO_WEIGHT_HEAVY;
		default: return PANGO_WEIGHT_NORMAL;
	}
}

static PangoVariant set_fontvariant (int val)
{
	switch (val) {
		case 0: return PANGO_VARIANT_NORMAL;
		case 1: return PANGO_VARIANT_SMALL_CAPS;
		default: return PANGO_VARIANT_NORMAL;
	}
}

static PangoStretch set_fontstretch (int val)
{
	switch (val) {
		case 0: return PANGO_STRETCH_ULTRA_CONDENSED;
		case 1: return PANGO_STRETCH_EXTRA_CONDENSED;
		case 2: return PANGO_STRETCH_CONDENSED;
		case 3: return PANGO_STRETCH_SEMI_CONDENSED;
		case 4: return PANGO_STRETCH_NORMAL;
		case 5: return PANGO_STRETCH_SEMI_EXPANDED;
		case 6: return PANGO_STRETCH_EXPANDED;
		case 7: return PANGO_STRETCH_EXTRA_EXPANDED;
		case 8: return PANGO_STRETCH_ULTRA_EXPANDED;
		default: return PANGO_STRETCH_NORMAL;
	}
}

static int set_fontsize (double val) {return (int) (val * PANGO_SCALE);}

gcpThemeManager::gcpThemeManager ()
{
	GError *error = NULL;
	m_ConfClient = gconf_client_get_default ();
	gconf_client_add_dir (m_ConfClient, "/apps/gchempaint/settings", GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);
	GCU_GCONF_GET (ROOTDIR"bond-length", float, DefaultBondLength, 140.)
	GCU_GCONF_GET (ROOTDIR"bond-angle", float, DefaultBondAngle, 120.)
	GCU_GCONF_GET (ROOTDIR"bond-dist", float, DefaultBondDist, 5.)
	GCU_GCONF_GET (ROOTDIR"bond-width", float, DefaultBondWidth, 1.0)
	GCU_GCONF_GET (ROOTDIR"arrow-length", float, DefaultArrowLength, 200)
	GCU_GCONF_GET (ROOTDIR"arrow-headA", float, DefaultArrowHeadA, 6.0)
	GCU_GCONF_GET (ROOTDIR"arrow-headB", float, DefaultArrowHeadB, 8.0)
	GCU_GCONF_GET (ROOTDIR"arrow-headC", float, DefaultArrowHeadC, 4.0)
	GCU_GCONF_GET (ROOTDIR"arrow-dist", float, DefaultArrowDist, 5.0)
	GCU_GCONF_GET (ROOTDIR"arrow-width", float, DefaultArrowWidth, 1.0)
	GCU_GCONF_GET (ROOTDIR"hash-width", float, DefaultHashWidth, 1.0)
	GCU_GCONF_GET (ROOTDIR"hash-dist", float, DefaultHashDist, 2.0)
	GCU_GCONF_GET (ROOTDIR"stereo-width", float, DefaultStereoBondWidth, 5.0)
	GCU_GCONF_GET_N_TRANSFORM (ROOTDIR"scale", float, DefaultZoomFactor, 0.25, inv)
	GCU_GCONF_GET (ROOTDIR"padding", float, DefaultPadding, 2.0)
	GCU_GCONF_GET (ROOTDIR"arrow-padding", float, DefaultArrowPadding, 16.0)
	GCU_GCONF_GET (ROOTDIR"stoichiometry-padding", float, DefaultStoichiometryPadding, 1.)
	GCU_GCONF_GET (ROOTDIR"object-padding", float, DefaultObjectPadding, 16.0)
	GCU_GCONF_GET (ROOTDIR"charge-sign-padding", float, DefaultSignPadding, 8.0)
	GCU_GCONF_GET (ROOTDIR"charge-sign-height", float, DefaultChargeSignHeight, 12.)
	GCU_GCONF_GET (ROOTDIR"charge-sign-size", float, DefaultChargeSignSize, 9.)
	GCU_GCONF_GET (ROOTDIR"charge-sign-yalign", float, DefaultChargeYAlign, 10.)
	GCU_GCONF_GET_STRING (ROOTDIR"font-family", DefaultFontFamily, "Bitstream Vera Sans")
	GCU_GCONF_GET_N_TRANSFORM (ROOTDIR"font-style", int, DefaultFontStyle, 0, set_fontstyle)
	GCU_GCONF_GET_N_TRANSFORM (ROOTDIR"font-weight", int, DefaultFontWeight, 4, set_fontweight)
	GCU_GCONF_GET_N_TRANSFORM (ROOTDIR"font-variant", int, DefaultFontVariant, 0, set_fontvariant)
	GCU_GCONF_GET_N_TRANSFORM (ROOTDIR"font-stretch", int, DefaultFontStretch, 4, set_fontstretch)
	GCU_GCONF_GET_N_TRANSFORM (ROOTDIR"font-size", float, DefaultFontSize, 12., set_fontsize)
	GCU_GCONF_GET_STRING (ROOTDIR"text-font-family", DefaultTextFontFamily, "Bitstream Vera Serif")
	GCU_GCONF_GET_N_TRANSFORM (ROOTDIR"text-font-style", int, DefaultTextFontStyle, 0, set_fontstyle)
	GCU_GCONF_GET_N_TRANSFORM (ROOTDIR"text-font-weight", int, DefaultTextFontWeight, 4, set_fontweight)
	GCU_GCONF_GET_N_TRANSFORM (ROOTDIR"text-font-variant", int, DefaultTextFontVariant, 0, set_fontvariant)
	GCU_GCONF_GET_N_TRANSFORM (ROOTDIR"text-font-stretch", int, DefaultTextFontStretch, 4, set_fontstretch)
	GCU_GCONF_GET_N_TRANSFORM (ROOTDIR"text-font-size", float, DefaultTextFontSize, 12., set_fontsize)
	m_NotificationId = gconf_client_notify_add (m_ConfClient, "/apps/gchempaint/settings", (GConfClientNotifyFunc) on_config_changed, this, NULL, NULL);
	// Build default theme from settings
	m_Themes["Default"] = new gcpTheme ("Default", _("Default"));
	m_Names.push_front (_("Default"));
}

gcpThemeManager::~gcpThemeManager ()
{
	gconf_client_notify_remove (m_ConfClient, m_NotificationId);
	gconf_client_remove_dir (m_ConfClient, "/apps/gchempaint/settings", NULL);
	g_object_unref (m_ConfClient);
}

gcpTheme *gcpThemeManager::GetTheme (char const *name)
{
	return m_Themes[_(name)];
}

gcpTheme *gcpThemeManager::GetTheme (string &name)
{
	return m_Themes[_(name.c_str ())];
}

list <string> const &gcpThemeManager::GetThemesNames ()
{
	return m_Names;
}

void gcpThemeManager::OnConfigChanged (GConfClient *client, guint cnxn_id, GConfEntry *entry)
{
	if (client != m_ConfClient)
		return;	// we might want an error message?
	if (cnxn_id != m_NotificationId)
		return;	// we might want an error message?
	gcpTheme *Theme = m_Themes[_("Default")];
	if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"bond-length")) {
		DefaultBondLength = gconf_value_get_float (gconf_entry_get_value (entry));
		Theme->m_BondLength = DefaultBondLength;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"bond-angle"))  {
		DefaultBondAngle = gconf_value_get_float (gconf_entry_get_value (entry));
		Theme->m_BondAngle = DefaultBondAngle;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"bond-dist"))  {
		DefaultBondDist = gconf_value_get_float (gconf_entry_get_value (entry));
		Theme->m_BondDist = DefaultBondDist;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"bond-width"))  {
		DefaultBondWidth = gconf_value_get_float (gconf_entry_get_value (entry));
		Theme->m_BondWidth = DefaultBondWidth;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"arrow-length"))  {
		DefaultArrowLength = gconf_value_get_float (gconf_entry_get_value (entry));
		Theme->m_ArrowLength = DefaultArrowLength;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"arrow-headA"))  {
		DefaultArrowHeadA = gconf_value_get_float (gconf_entry_get_value (entry));
		Theme->m_ArrowHeadA = DefaultArrowHeadA;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"arrow-headB"))  {
		DefaultArrowHeadB = gconf_value_get_float (gconf_entry_get_value (entry));
		Theme->m_ArrowHeadB = DefaultArrowHeadB;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"arrow-headC"))  {
		DefaultArrowHeadC = gconf_value_get_float (gconf_entry_get_value (entry));
		Theme->m_ArrowHeadC = DefaultArrowHeadC;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"arrow-dist"))  {
		DefaultArrowDist = gconf_value_get_float (gconf_entry_get_value (entry));
		Theme->m_ArrowDist = DefaultArrowDist;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"arrow-width"))  {
		DefaultArrowWidth = gconf_value_get_float (gconf_entry_get_value (entry));
		Theme->m_ArrowWidth = DefaultArrowWidth;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"hash-width"))  {
		DefaultHashWidth = gconf_value_get_float (gconf_entry_get_value (entry));
		Theme->m_HashWidth = DefaultHashWidth;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"hash-dist"))  {
		DefaultHashDist = gconf_value_get_float (gconf_entry_get_value (entry));
		Theme->m_HashDist = DefaultHashDist;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"stereo-width"))  {
		DefaultStereoBondWidth = gconf_value_get_float (gconf_entry_get_value (entry));
		Theme->m_StereoBondWidth = DefaultStereoBondWidth;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"scale"))  {
		double scale = gconf_value_get_float (gconf_entry_get_value (entry));
		if (scale > 1e-5) {
			DefaultZoomFactor = 1 / scale;
			Theme->m_ZoomFactor = DefaultZoomFactor;
		}
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"padding"))  {
		DefaultPadding = gconf_value_get_float (gconf_entry_get_value (entry));
		Theme->m_Padding = DefaultPadding;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"stoichiometry-padding"))  {
		DefaultStoichiometryPadding = gconf_value_get_float (gconf_entry_get_value (entry));
		Theme->m_StoichiometryPadding = DefaultStoichiometryPadding;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"object-padding"))  {
		DefaultObjectPadding = gconf_value_get_float (gconf_entry_get_value (entry));
		Theme->m_ObjectPadding = DefaultObjectPadding;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"charge-sign-padding"))  {
		DefaultSignPadding = gconf_value_get_float (gconf_entry_get_value (entry));
		Theme->m_SignPadding = DefaultSignPadding;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"charge-sign-height"))  {
		DefaultChargeSignHeight = gconf_value_get_float (gconf_entry_get_value (entry));
		Theme->m_ChargeSignHeight = DefaultChargeSignHeight;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"charge-sign-size"))  {
		DefaultChargeSignSize = gconf_value_get_float (gconf_entry_get_value (entry));
		Theme->m_ChargeSignSize = DefaultChargeSignSize;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"charge-sign-yalign"))  {
		DefaultChargeYAlign = gconf_value_get_float (gconf_entry_get_value (entry));
		Theme->m_ChargeYAlign = DefaultChargeYAlign;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"font-family"))  {
		char const *name = gconf_value_get_string (gconf_entry_get_value (entry));
		if (name) {
			if (DefaultFontFamily != NULL)
				g_free (DefaultFontFamily);
			DefaultFontFamily = g_strdup (name);
			Theme->m_FontFamily = DefaultFontFamily;
		}
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"font-style"))  {
		DefaultFontStyle = set_fontstyle (gconf_value_get_int (gconf_entry_get_value (entry)));
		Theme->m_FontStyle = DefaultFontStyle;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"font-weight"))  {
		DefaultFontWeight = set_fontweight (gconf_value_get_int (gconf_entry_get_value (entry)));
		Theme->m_FontWeight = DefaultFontWeight;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"font-variant"))  {
		DefaultFontVariant = set_fontvariant (gconf_value_get_int (gconf_entry_get_value (entry)));
		Theme->m_FontVariant = DefaultFontVariant;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"font-stretch"))  {
		DefaultFontStretch = set_fontstretch (gconf_value_get_int (gconf_entry_get_value (entry)));
		Theme->m_FontStretch = DefaultFontStretch;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"font-size"))  {
		DefaultFontSize = set_fontsize (gconf_value_get_float (gconf_entry_get_value (entry)));
		Theme->m_FontSize = DefaultFontSize;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"text-font-family"))  {
		char const *name = gconf_value_get_string (gconf_entry_get_value (entry));
		if (name) {
			if (DefaultTextFontFamily != NULL)
				g_free (DefaultTextFontFamily);
			DefaultTextFontFamily = g_strdup (name);
			Theme->m_TextFontFamily = DefaultTextFontFamily;
		}
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"text-font-style"))  {
		DefaultTextFontStyle = set_fontstyle (gconf_value_get_int (gconf_entry_get_value (entry)));
		Theme->m_TextFontStyle = DefaultTextFontStyle;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"text-font-weight"))  {
		DefaultTextFontWeight = set_fontweight (gconf_value_get_int (gconf_entry_get_value (entry)));
		Theme->m_TextFontWeight = DefaultTextFontWeight;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"text-font-variant"))  {
		DefaultTextFontVariant = set_fontvariant (gconf_value_get_int (gconf_entry_get_value (entry)));
		Theme->m_TextFontVariant = DefaultTextFontVariant;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"text-font-stretch"))  {
		DefaultTextFontStretch = set_fontstretch (gconf_value_get_int (gconf_entry_get_value (entry)));
		Theme->m_TextFontStretch = DefaultTextFontStretch;
	} else if (!strcmp (gconf_entry_get_key (entry),ROOTDIR"text-font-size"))  {
		DefaultTextFontSize = set_fontsize (gconf_value_get_float (gconf_entry_get_value (entry)));
		Theme->m_TextFontSize = DefaultTextFontSize;
	}
}

gcpTheme *gcpThemeManager::CreateNewTheme ()
{
	gchar *name = g_strdup (_("NewTheme1")), *local_name;
	int i = 2;
	while (m_Themes[name] != NULL) {
		g_free (name);
		name = g_strdup_printf ("NewTheme%d", i++);
	}
	local_name = g_strdup_printf (_("NewTheme%d"), i++);
	gcpTheme *pTheme = new gcpTheme (name, local_name);
	m_Themes[name] = pTheme;
	g_free (name);
	g_free (local_name);
	return pTheme;
}
