/* 
 * GChemPaint library
 * fontsel.c 
 *
 * Copyright (C) 2006 Jean Bréfort <jean.brefort@normalesup.org>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301
 * USA
 */

#include "gchempaint-config.h"
#include "fontsel.h"
#include <gtk/gtkbin.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtktable.h>
#include <gtk/gtktreeview.h>
#include <gtk/gtktreeselection.h>
#include <gtk/gtkcellrenderertext.h>
#include <gsf/gsf-impl-utils.h>
#include <map>
#include <string>

using namespace std;

typedef struct {
	GtkBin base;
	GtkListStore *FamilyList, *FaceList, *SizeList;
	GtkTreeView *FamilyTree, *FacesTree;
	guint FamilySignal, FaceSignal, SizeSignal;
	GtkTreeSelection *FamilySel, *FaceSel, *SizeSel;
	map<string, PangoFontFamily*> Families;
	map<string, PangoFontFace*> Faces;
	char const *FamilyName;
	PangoStyle Style;
	PangoWeight Weight;
	PangoStretch Stretch;
	PangoVariant Variant;
} GcpFontSel;

typedef struct {
	GtkBinClass base_class;
} GcpFontSelClass;

static void
gcp_font_sel_size_request (GtkWidget      *widget,
			GtkRequisition *requisition)
{
	GtkWidget *w = GTK_WIDGET (gtk_bin_get_child (GTK_BIN (widget)));
	if (w)
    	gtk_widget_size_request (w, requisition);
	else {
		requisition->width = 0;
		requisition->height = 0;
	}
}

static void
gcp_font_sel_size_allocate (GtkWidget     *widget,
			 GtkAllocation *allocation)
{
	GtkWidget *w = GTK_WIDGET (gtk_bin_get_child (GTK_BIN (widget)));
	if (w)
		gtk_widget_size_allocate (GTK_WIDGET (w), allocation);
}

static void
gcp_font_sel_class_init (GcpFontSelClass *klass)
{
	GObjectClass *object_class = (GObjectClass*) klass;
	GtkWidgetClass *widget_class = (GtkWidgetClass *) klass;

	widget_class->size_request = gcp_font_sel_size_request;
	widget_class->size_allocate = gcp_font_sel_size_allocate;
}

static void on_select_family (GtkTreeSelection *selection, GcpFontSel *fs)
{
//	tool->OnSelectFamily (selection);
	GtkTreeModel *model;
	GtkTreeIter iter, selected;
	char const *name;
	if (!gtk_tree_selection_get_selected (selection, &model, &iter))
		return;
	gtk_tree_model_get (model, &iter, 0, &fs->FamilyName, -1);
	PangoFontFamily *family = fs->Families[fs->FamilyName];
	PangoFontFace **faces;
	int i, besti, nb;
	g_signal_handler_block (fs->FaceSel, fs->FaceSignal);
	pango_font_family_list_faces (family, &faces, &nb);
	gtk_list_store_clear (fs->FaceList);
	map<string, PangoFontFace*>::iterator j, jend = fs->Faces.end ();
	for (j = fs->Faces.begin (); j != jend; j++) {
		g_object_unref ((*j).second);
	}
	fs->Faces.clear ();
	PangoFontDescription *desc;
	int distance, best;
	PangoStyle Style;
	PangoWeight Weight;
	PangoVariant Variant;
	PangoStretch Stretch;

	best = 32000; // This should be enough
	for (i = 0; i < nb; i++) {
		name = pango_font_face_get_face_name (faces[i]);
		desc = pango_font_face_describe (faces[i]);
		fs->Faces[name] = (PangoFontFace*) g_object_ref (faces[i]);
		gtk_list_store_append (fs->FaceList, &iter);
		gtk_list_store_set (fs->FaceList, &iter,
				  0, name,
				  -1);
		// Try to select the best available face
		Style = pango_font_description_get_style (desc);
		Weight = pango_font_description_get_weight (desc);
		Variant = pango_font_description_get_variant (desc);
		Stretch = pango_font_description_get_stretch (desc);
		distance = abs (Weight - fs->Weight)
						+ abs ((Style? Style + 2: 0) - (fs->Style? fs->Style + 2: 0)) * 1000
						+ abs (Variant - fs->Variant) * 10 + abs (Stretch - fs->Stretch);
		if (distance < best) {
			best = distance;
			selected = iter;
			besti = i;
		}
		// TODO: write this code
		pango_font_description_free (desc);
	}
	g_signal_handler_unblock (fs->FaceSel, fs->FaceSignal);
	GtkTreePath *path = gtk_tree_model_get_path (GTK_TREE_MODEL (fs->FaceList), &selected);
	if (path) {
		gtk_tree_selection_select_path (GTK_TREE_SELECTION (fs->FaceSel), path);
		gtk_tree_path_free (path);
	} else {
		//TODO: choose a face when default is not available
	}
}

static void on_select_face (GtkTreeSelection *selection, GcpFontSel *fs)
{
//	tool->OnSelectFace (selection);
puts("select face");
}

static void on_select_size (GtkTreeSelection *selection, GcpFontSel *fs)
{
	int size;
	GtkTreeModel *model;
	GtkTreeIter iter;
	gtk_tree_selection_get_selected (selection, &model, &iter);
	gtk_tree_model_get (model, &iter, 0, &size, -1);
//	tool->OnSelectSize (size * PANGO_SCALE);
}

static void on_size_activate (GtkEntry *entry, GcpFontSel *fs)
{
}

static void on_size_focus_out (GtkEntry *entry, GdkEventFocus *event, GcpFontSel *fs)
{
}

/* These are what we use as the standard font sizes, for the size list.
 */
static const guint16 font_sizes[] = {
  8, 9, 10, 11, 12, 13, 14, 16, 18, 20, 22, 24, 26, 28,
  32, 36, 40, 48, 56, 64, 72
};

static void
gcp_font_sel_init (GcpFontSel *fs)
{
	int i, nb;
	PangoFontFamily **families;
	GtkWidget *sc, *w = gtk_table_new (3, 3, FALSE);
	g_object_set (G_OBJECT (w), "border-width", 6, NULL);
	fs->Families = map<string, PangoFontFamily*>();
	fs->Faces = map<string, PangoFontFace*>();
	GtkTable *table = GTK_TABLE (w);
	gtk_table_set_col_spacings (table, 12);
	gtk_container_add (GTK_CONTAINER (fs), GTK_WIDGET (w));
	// Initialize faces list
	fs->FaceList = gtk_list_store_new (1, G_TYPE_STRING);
	gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (fs->FaceList), 0, GTK_SORT_ASCENDING);
	fs->FacesTree = GTK_TREE_VIEW (gtk_tree_view_new_with_model (GTK_TREE_MODEL (fs->FaceList)));
	gtk_tree_view_set_headers_visible (fs->FacesTree, false);
	gtk_table_attach (table, GTK_WIDGET (fs->FacesTree), 1, 2, 1, 3,
			(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			(GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
	GtkCellRenderer *renderer = gtk_cell_renderer_text_new ();
	GtkTreeViewColumn *column = gtk_tree_view_column_new_with_attributes (NULL, renderer, "text", 0, NULL);
	gtk_tree_view_append_column (fs->FacesTree, column);
	GtkTreeSelection *selection = gtk_tree_view_get_selection (fs->FacesTree);
	fs->FaceSel = selection;
	fs->FaceSignal = g_signal_connect (fs->FaceSel, "changed", G_CALLBACK (on_select_face), fs);
	// Initialize sizes list
	fs->SizeList = gtk_list_store_new (1, G_TYPE_INT);
	w = gtk_tree_view_new_with_model (GTK_TREE_MODEL (fs->SizeList));
	gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (w), false);
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (NULL, renderer, "text", 0, NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (w), column);
	GtkTreeIter iter, selected;
	for (i = 0; i < (int) G_N_ELEMENTS (font_sizes); i++) {
		gtk_list_store_append (fs->SizeList, &iter);
		gtk_list_store_set (fs->SizeList, &iter,
				  0, font_sizes[i],
				  -1);
	}
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (w));
	fs->SizeSel = selection;
	fs->SizeSignal = g_signal_connect (fs->SizeSel, "changed", G_CALLBACK (on_select_size), fs);
	sc = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sc), GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
	gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (sc), w);
	gtk_table_attach (table, sc, 2, 3, 2, 3,
			(GtkAttachOptions) (GTK_FILL),
			(GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
	w = gtk_entry_new ();
	gtk_table_attach (table, w, 2, 3, 1, 2,
			(GtkAttachOptions) (0),
			(GtkAttachOptions) 0, 0, 0);
	PangoContext *pc = gtk_widget_get_pango_context (w);
	PangoLayout *pl = pango_layout_new (pc);
	pango_layout_set_text (pl, "0000000", -1);
	PangoRectangle rect;
	pango_layout_get_extents (pl, NULL, &rect);
	g_object_unref (G_OBJECT (pl));
	gtk_widget_set_size_request (sc, -1, rect.height / PANGO_SCALE * 12);
	gtk_widget_set_size_request (w, rect.width / PANGO_SCALE, -1);
	pango_context_list_families (pc, &families, &nb);
	fs->FamilyList = gtk_list_store_new (1, G_TYPE_STRING);
	gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (fs->FamilyList), 0, GTK_SORT_ASCENDING);
	fs->FamilyTree =  GTK_TREE_VIEW (gtk_tree_view_new_with_model (GTK_TREE_MODEL (fs->FamilyList)));
	gtk_tree_view_set_headers_visible (fs->FamilyTree, false);
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (NULL, renderer, "text", 0, NULL);
	gtk_tree_view_append_column (fs->FamilyTree, column);
	string name;
	for (i = 0; i < nb; i++) {
		PangoFontFace **faces;
		int *sizes, n;
		pango_font_family_list_faces (families[i], &faces, &n);
		if (n <= 0)
			continue;
		pango_font_face_list_sizes (faces[0], &sizes, &n);
		if (n > 0) // Do not use bitmap fonts
			continue;
		name = pango_font_family_get_name (families[i]);
		fs->Families[name] = (PangoFontFamily*) g_object_ref (families[i]);
		gtk_list_store_append (fs->FamilyList, &iter);
		gtk_list_store_set (fs->FamilyList, &iter,
				  0, name.c_str (),
				  -1);
	}
	fs->FamilySel = gtk_tree_view_get_selection (fs->FamilyTree);
	gtk_tree_selection_set_mode (fs->FamilySel, GTK_SELECTION_BROWSE);
	fs->FamilySignal = g_signal_connect (G_OBJECT (fs->FamilySel), "changed", G_CALLBACK (on_select_family), fs);
	sc = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sc), GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
	gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (sc), GTK_WIDGET (fs->FamilyTree));
	gtk_table_attach (table, sc, 0, 1, 1, 3,
			(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			(GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
}

GSF_CLASS (GcpFontSel, gcp_font_sel,
	   gcp_font_sel_class_init, gcp_font_sel_init,
	   GTK_TYPE_BIN)
