/* libgcpcanvas/gcp-canvas-pango.c
 *
 * Copyright (c) 2005 Jean Bréfort <jean.brefort@normalesup.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA 02111-1307, USA.
 */

#include "gchempaint-config.h"
#include "gcp-canvas-pango.h"
#include "gnome-print-pango.h"
#include "gprintable.h"
#include <glib/gi18n-lib.h>
#include <cairo/cairo.h>

#include <libgnomecanvas/gnome-canvas.h>
#include <libgnomecanvas/gnome-canvas-util.h>

#include <math.h>

struct _GnomeCanvasPangoPrivate {
	PangoLayout *layout;
	/* Position at anchor */
	double x, y;
	/* Dimensions */
	double width, height; /* user set size */
	double _width, _height; /* size of the pango layout */
	GtkAnchorType anchor;
	guint rgba;
	char *color_name;
	gboolean editing, cursor_visible;
	guint blink_timeout;
	gint index;
};

enum {
	PROP_0,
	PROP_LAYOUT,
	PROP_X,
	PROP_Y,
	PROP_WIDTH,
	PROP_HEIGHT,
	PROP_ANCHOR,
	PROP_FILL_COLOR,
	PROP_EDITING,
};

static GnomeCanvasItemClass *parent_class;

static void gnome_canvas_pango_class_init (GnomeCanvasPangoClass *klass);
static void gnome_canvas_pango_init(GnomeCanvasPango *text);
static void gnome_canvas_pango_finalize(GObject *object);
static void gnome_canvas_pango_set_property(GObject *object, guint property_id,
						const GValue *value, GParamSpec *pspec);
static void gnome_canvas_pango_get_property(GObject *object, guint property_id,
						GValue *value, GParamSpec *pspec);
static void gnome_canvas_pango_update(GnomeCanvasItem *item, double *affine,
					  ArtSVP *clip_path, int flags);
static void gnome_canvas_pango_realize(GnomeCanvasItem *item);
static void gnome_canvas_pango_unrealize(GnomeCanvasItem *item);
static double gnome_canvas_pango_point(GnomeCanvasItem *item, 
					   double x, double y,
					   int cx, int cy, 
					   GnomeCanvasItem **actual_item);
static void gnome_canvas_pango_draw(GnomeCanvasItem *item, 
					GdkDrawable *drawable,
					int x, int y, int width, int height);
static void gnome_canvas_pango_render(GnomeCanvasItem *item,
					  GnomeCanvasBuf *buf);
static gint gnome_canvas_pango_event(GnomeCanvasItem *item, 
					 GdkEvent *event);
static void gnome_canvas_pango_get_bounds(GnomeCanvasItem *text, double *px1, double *py1,
	   double *px2, double *py2);
static void gnome_canvas_pango_print       (GPrintable *gprintable, GnomePrintContext *pc);
static void gnome_canvas_pango_export_svg   (GPrintable *gprintable, xmlDocPtr doc, xmlNodePtr node);

/*	static gint blink_cb(gpointer data);	*/

#define PREBLINK_TIME 300
#define CURSOR_ON_TIME 800
#define CURSOR_OFF_TIME 400

static gint
blink_cb (gpointer data)
{
	GnomeCanvasPango *text = GNOME_CANVAS_PANGO (data);

	if (text->_priv->cursor_visible)
		text->_priv->blink_timeout = g_timeout_add (
			CURSOR_OFF_TIME, blink_cb, text);
	else
		text->_priv->blink_timeout = g_timeout_add (
			CURSOR_ON_TIME, blink_cb, text);

	text->_priv->cursor_visible = !text->_priv->cursor_visible;
	gnome_canvas_item_request_update (GNOME_CANVAS_ITEM (text));
	/* Remove ourself */
	return FALSE;
}

static void
gnome_canvas_pango_print_init (GPrintableIface *iface)
{
	iface->print = gnome_canvas_pango_print;
	iface->export_svg = gnome_canvas_pango_export_svg;
}

GType
gnome_canvas_pango_get_type(void)
{
	static GType pango_type;

	if (!pango_type) {
		static const GTypeInfo object_info = {
			sizeof (GnomeCanvasPangoClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_canvas_pango_class_init,
			(GClassFinalizeFunc) NULL,
			NULL,			/* class_data */
			sizeof (GnomeCanvasPango),
			0,			/* n_preallocs */
			(GInstanceInitFunc) gnome_canvas_pango_init,
			NULL			/* value_table */
		};

		static const GInterfaceInfo print_info = {
			(GInterfaceInitFunc) gnome_canvas_pango_print_init,
			NULL, NULL
		};

		pango_type = g_type_register_static (GNOME_TYPE_CANVAS_ITEM, "GnomeCanvasPango",
							 &object_info, 0);

		g_type_add_interface_static (pango_type, G_TYPE_PRINTABLE, &print_info);
	}

	return pango_type;
}

static void
gnome_canvas_pango_class_init (GnomeCanvasPangoClass *klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
/*	GtkObjectClass *object_class = GTK_OBJECT_CLASS (klass);	*/
	GnomeCanvasItemClass *item_class = GNOME_CANVAS_ITEM_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	gobject_class->set_property = gnome_canvas_pango_set_property;
	gobject_class->get_property = gnome_canvas_pango_get_property;
	gobject_class->finalize = gnome_canvas_pango_finalize;

	g_object_class_install_property (
		gobject_class,
		PROP_LAYOUT,
		g_param_spec_object ("layout",
				     _("Layout"),
				     _("Pango layout"),
				     PANGO_TYPE_LAYOUT,
				     G_PARAM_READWRITE));
	g_object_class_install_property (
		gobject_class,
		PROP_X,
		g_param_spec_double ("x",
				     _("X"),
				     _("X position"),
				     -G_MAXDOUBLE, G_MAXDOUBLE, 0.0,
				     G_PARAM_READWRITE));
	g_object_class_install_property (
		gobject_class,
		PROP_Y,
		g_param_spec_double ("y",
				     _("Y"),
				     _("Y position"),
				     -G_MAXDOUBLE, G_MAXDOUBLE, 0.0,
				     G_PARAM_READWRITE));
	g_object_class_install_property (
		gobject_class,
		PROP_WIDTH,
		g_param_spec_double ("width",
				     _("Width"),
				     _("Width for text box"),
				     -G_MAXDOUBLE, G_MAXDOUBLE, 0.0,
				     G_PARAM_READWRITE));
	g_object_class_install_property (
		gobject_class,
		PROP_HEIGHT,
		g_param_spec_double ("height",
				     _("Height"),
				     _("Height for text box"),
				     -G_MAXDOUBLE, G_MAXDOUBLE, 0.0,
				     G_PARAM_READWRITE));
	g_object_class_install_property (
		gobject_class,
		PROP_ANCHOR,
		g_param_spec_enum ("anchor",
				   _("Anchor"),
				   _("Anchor point for text"),
				   GTK_TYPE_ANCHOR_TYPE,
				   GTK_ANCHOR_NW,
				   G_PARAM_READWRITE));
	g_object_class_install_property (
		gobject_class,
		PROP_FILL_COLOR,
		g_param_spec_string ("fill_color",
					_("Color"),
					_("Text color, as string"),
					NULL,
					(G_PARAM_READWRITE)));
	g_object_class_install_property (
		gobject_class,
		PROP_EDITING,
		g_param_spec_boolean ("editing",
				      _("Editing"),
				      _("Is this rich text item currently edited?"),
				      FALSE,
				      G_PARAM_READWRITE));

	item_class->update = gnome_canvas_pango_update;
	item_class->realize = gnome_canvas_pango_realize;
	item_class->unrealize = gnome_canvas_pango_unrealize;
	item_class->draw = gnome_canvas_pango_draw;
	item_class->point = gnome_canvas_pango_point;
	item_class->render = gnome_canvas_pango_render;
	item_class->event = gnome_canvas_pango_event;
	item_class->bounds = gnome_canvas_pango_get_bounds;
}

static void
gnome_canvas_pango_init (GnomeCanvasPango *text)
{
	text->_priv = g_new0 (GnomeCanvasPangoPrivate, 1);
	text->_priv->anchor = GTK_ANCHOR_NW;
}

static void
gnome_canvas_pango_finalize (GObject *object)
{
	GnomeCanvasPango *text = GNOME_CANVAS_PANGO (object);
	g_return_if_fail (text);
	if (text->_priv->layout)
		g_object_unref (text->_priv->layout);
	if (text->_priv->color_name != NULL)
		g_free (text->_priv->color_name);
	g_free (text->_priv);
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gnome_canvas_pango_set_property (GObject *object, guint property_id,
						const GValue *value, GParamSpec *pspec)
{
	GnomeCanvasPango *text = GNOME_CANVAS_PANGO (object);

	switch (property_id) {
	case PROP_LAYOUT:
		if (text->_priv->layout)
			g_object_unref (text->_priv->layout);
		text->_priv->layout = g_value_get_object (value);
		g_object_ref (text->_priv->layout);
		break;
	case PROP_X:
		text->_priv->x = g_value_get_double (value);
		break;
	case PROP_Y:
		text->_priv->y = g_value_get_double (value);
		break;
	case PROP_WIDTH:
		text->_priv->width = g_value_get_double (value);
		break;
	case PROP_HEIGHT:
		text->_priv->height = g_value_get_double (value);
		break;
	case PROP_ANCHOR:
		text->_priv->anchor = g_value_get_enum (value);
		break;
	case PROP_FILL_COLOR: {
		const char *color_name;
		GdkColor color;

		if (text->_priv->color_name != NULL) {
			g_free (text->_priv->color_name);
			text->_priv->color_name = NULL;
		}

		color_name = g_value_get_string (value);
		if (color_name) {
			text->_priv->color_name = g_strdup (color_name);
			gdk_color_parse (color_name, &color);
			text->_priv->rgba = ((color.red & 0xff00) << 16 |
			(color.green & 0xff00) << 8 |
			(color.blue & 0xff00) |
			0xff);
		} else
			text->_priv->rgba = 0xff;
		break;
	}
	case PROP_EDITING: {
		gboolean editing = g_value_get_boolean (value);
		if (editing == text->_priv->editing)
			break;
		text->_priv->editing = editing;
		if (editing) {
			text->_priv->cursor_visible = TRUE;
			text->_priv->blink_timeout = g_timeout_add (
				CURSOR_ON_TIME, blink_cb, text);
		} else {
			text->_priv->cursor_visible = FALSE;
			if (text->_priv->blink_timeout != 0) {
				g_source_remove (text->_priv->blink_timeout);
				text->_priv->blink_timeout = 0;
			}
			while (	g_idle_remove_by_data (object));
		}
		break;
	}
		       
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}

	gnome_canvas_item_request_update (GNOME_CANVAS_ITEM (text));
}

static void
gnome_canvas_pango_get_property (GObject *object, guint property_id,
						GValue *value, GParamSpec *pspec)
{
	GnomeCanvasPango *text = GNOME_CANVAS_PANGO (object);

	switch (property_id) {
	case PROP_LAYOUT:
		g_value_set_object (value, text->_priv->layout);
		break;
	case PROP_X:
		g_value_set_double (value, text->_priv->x);
		break;
	case PROP_Y:
		g_value_set_double (value, text->_priv->y);
		break;
	case PROP_WIDTH:
		g_value_set_double (value, text->_priv->width);
		break;
	case PROP_HEIGHT:
		g_value_set_double (value, text->_priv->height);
		break;
	case PROP_ANCHOR:
		g_value_set_enum (value, text->_priv->anchor);
		break;
	case PROP_FILL_COLOR:
		g_value_set_string (value, text->_priv->color_name);
		break;
	case PROP_EDITING:
		g_value_set_boolean (value, text->_priv->editing);
		break;

	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}

static void
adjust_for_anchors(GnomeCanvasPango *text, double *ax, double *ay)
{
	double x, y;
	double width = (text->_priv->width > 0)? text->_priv->width: text->_priv->_width;
	double height = (text->_priv->height > 0)? text->_priv->height: text->_priv->_height;

	x = text->_priv->x;
	y = text->_priv->y;

	/* Anchor text */
	/* X coordinates */
	switch (text->_priv->anchor) {
	case GTK_ANCHOR_NW:
	case GTK_ANCHOR_W:
	case GTK_ANCHOR_SW:
		break;

	case GTK_ANCHOR_N:
	case GTK_ANCHOR_CENTER:
	case GTK_ANCHOR_S:
		x -= width / 2;
		break;

	case GTK_ANCHOR_NE:
	case GTK_ANCHOR_E:
	case GTK_ANCHOR_SE:
		x -= width;
		break;
	default:
		break;
	}

	/* Y coordinates */
	switch (text->_priv->anchor) {
	case GTK_ANCHOR_NW:
	case GTK_ANCHOR_N:
	case GTK_ANCHOR_NE:
		break;

	case GTK_ANCHOR_W:
	case GTK_ANCHOR_CENTER:
	case GTK_ANCHOR_E:
		y -= height / 2;
		break;

	case GTK_ANCHOR_SW:
	case GTK_ANCHOR_S:
	case GTK_ANCHOR_SE:
		y -= height;
		break;
	default:
		break;
	}

	if (ax)
		*ax = x;
	if (ay)
		*ay = y;
} /* adjust_for_anchors */

static void
gnome_canvas_pango_update (GnomeCanvasItem *item, double *affine,
					  ArtSVP *clip_path, int flags)
{
	GnomeCanvasPango *text = GNOME_CANVAS_PANGO (item);
	double i2w[6], w2c[6], i2c[6];
	double x1, y1, x2, y2;
	double width = (text->_priv->width > 0)? text->_priv->width: text->_priv->_width;
	ArtPoint ip, cp;

	GNOME_CANVAS_ITEM_CLASS (parent_class)->update ( item, affine, clip_path, flags);

	gnome_canvas_item_i2w_affine (item, i2w);
	gnome_canvas_w2c_affine (item->canvas, w2c);
	art_affine_multiply (i2c, i2w, w2c);

	if (text->_priv->layout) {
		PangoRectangle rect;
		pango_layout_get_extents (text->_priv->layout, NULL, &rect);
		text->_priv->_width = rect.width / PANGO_SCALE;
		text->_priv->_height = rect.height / PANGO_SCALE;
	}

	adjust_for_anchors (text, &x1, &y1);
	if (width < 1.)
		width = 1.;
	x2 = x1 + width;
	y2 = y1 + ((text->_priv->height > 0)? text->_priv->height: text->_priv->_height);

	ip.x = x1;
	ip.y = y1;
	art_affine_point (&cp, &ip, i2c);
	x1 = cp.x;
	y1 = cp.y;

	ip.x = x2;
	ip.y = y2;
	art_affine_point (&cp, &ip, i2c);
	x2 = cp.x;
	y2 = cp.y;

	gnome_canvas_update_bbox (item, x1, y1, x2, y2);
}

static void
gnome_canvas_pango_realize (GnomeCanvasItem *item)
{
}

static void
gnome_canvas_pango_unrealize (GnomeCanvasItem *item)
{
	GnomeCanvasPango *text = GNOME_CANVAS_PANGO (item);

	if (text->_priv->blink_timeout != 0) {
		g_source_remove (text->_priv->blink_timeout);
		text->_priv->blink_timeout = 0;
	}
	/*remove idle calls */
	while (g_idle_remove_by_data (item));

	(* GNOME_CANVAS_ITEM_CLASS(parent_class)->unrealize)(item);
}

static double
gnome_canvas_pango_point (GnomeCanvasItem *item, 
					   double x, double y,
					   int cx, int cy, 
					   GnomeCanvasItem **actual_item)
{
	GnomeCanvasPango *text = GNOME_CANVAS_PANGO (item);
	double ax, ay;
	double x1, x2, y1, y2;
	double dx, dy;

	*actual_item = item;

	/* This is a lame cop-out. Anywhere inside of the bounding box. */

	adjust_for_anchors(text, &ax, &ay);

	x1 = ax;
	y1 = ay;
	x2 = ax + ((text->_priv->width > 0)? text->_priv->width: text->_priv->_width);
	y2 = ay + ((text->_priv->height > 0)? text->_priv->height: text->_priv->_height);

	if ((x > x1) && (y > y1) && (x < x2) && (y < y2))
		return 0.0;

	if (x < x1)
		dx = x1 - x;
	else if (x > x2)
		dx = x - x2;
	else
		dx = 0.0;

	if (y < y1)
		dy = y1 - y;
	else if (y > y2)
		dy = y - y2;
	else
		dy = 0.0;

	return sqrt(dx * dx + dy * dy);
}

static void
gnome_canvas_pango_draw (GnomeCanvasItem *item, 
					GdkDrawable *drawable,
					int x, int y, int width, int height)
{
}

static void
gnome_canvas_pango_render (GnomeCanvasItem *item,
					  GnomeCanvasBuf *buf)
{
	GnomeCanvasPango *text = GNOME_CANVAS_PANGO (item);
	double x0, y0;
	double i2w[6], w2c[6], i2c[6];
	double ax, ay;
	double x1, y1, x2, y2;
	int x, y, px, py, h, w, dw, i;
	ArtPoint ip, cp;
	guint8 r, g, b, alpha, *dst, *src;
	guchar *data;
	cairo_surface_t *surf;
	cairo_t *cr;
	cairo_matrix_t matrix;

	g_return_if_fail (text);
	g_return_if_fail (text->_priv->layout);

	adjust_for_anchors (text, &ax, &ay);

	gnome_canvas_buf_ensure_buf (buf);

	gnome_canvas_item_i2w_affine(item, i2w);
	gnome_canvas_w2c_affine(item->canvas, w2c);
	art_affine_multiply(i2c, i2w, w2c);

	matrix.xx = i2c[0];
	matrix.xy = i2c[1];
	matrix.yx = i2c[2];
	matrix.yy = i2c[3];

	ip.x = ax;
	ip.y = ay;
	art_affine_point(&cp, &ip, i2c);
	x0 = x1 = floor (cp.x + 0.5);
	y0 = y1 = floor (cp.y + 0.5);

	ip.x = ax + ((text->_priv->width > 0)? text->_priv->width: text->_priv->_width);
	ip.y = ay + ((text->_priv->height > 0)? text->_priv->height: text->_priv->_height);
	art_affine_point(&cp, &ip, i2c);
	x2 = floor (cp.x + 0.5);
	y2 = floor (cp.y + 0.5);

	w = x2 - x1 + 1;
	h = y2 - y1 + 1;
	if (x0 < buf->rect.x0) {
		w -= buf->rect.x0 - x0;
		x = 0;
		px = x0 - buf->rect.x0;
		x0 = buf->rect.x0;
	} else  {
		x = x0 - buf->rect.x0;
		px = 0;
	}
	if (x0 + w >= buf->rect.x1)
		w = buf->rect.x1 - x0;
	if (y0 < buf->rect.y0)
	{
		h -= buf->rect.y0 - y0;
		y = 0;
		py = y0 - buf->rect.y0;
		y0 = buf->rect.y0;
	} else {
		y = y0 - buf->rect.y0;
		py = 0;
	}
	if (y0 + h >= buf->rect.y1)
		h = buf->rect.y1 - y0;
	if (((int) w <= 0) || ((int) h <= 0))
		return;

	matrix.x0 = px;
	matrix.y0 = py;
	// cairo image surfaces must have width that are multiples of 4
	dw = (w / 4 + 1) * 4;
	data = g_malloc0 (dw * h);
	surf = cairo_image_surface_create_for_data (data, CAIRO_FORMAT_A8,
				dw, h, dw);
	cr = cairo_create (surf);
	cairo_set_matrix (cr, &matrix);
	pango_cairo_update_layout (cr, text->_priv->layout);
	pango_cairo_show_layout (cr, text->_priv->layout);
	pango_context_set_matrix (pango_layout_get_context (text->_priv->layout), NULL);

	if (text->_priv->cursor_visible) {
		PangoRectangle rect;
		pango_layout_get_cursor_pos (text->_priv->layout,
								text->_priv->index, &rect, NULL);
		cairo_new_path (cr);
		cairo_move_to (cr, rect.x / PANGO_SCALE, rect.y / PANGO_SCALE);
		cairo_rel_line_to (cr, 0, rect.height / PANGO_SCALE);
		cairo_stroke (cr);
	}

	r = (guint8) text->_priv->rgba >> 24;
	g = (guint8) (text->_priv->rgba >> 16) & 0xff;
	b = (guint8) (text->_priv->rgba >> 8) & 0xff;

	src = data;
	dst = buf->buf + (int) y * buf->buf_rowstride + (int) x * 3;

	while (h-- > 0) {
		for (i = w; i-- > 0 ; dst += 3, src++) {
			alpha = *src;
			dst[0] = (dst[0] * (255 - alpha) + r * alpha) / 255;
			dst[1] = (dst[1] * (255 - alpha) + g * alpha) / 255;
			dst[2] = (dst[2] * (255 - alpha) + b * alpha) / 255;
		}
		dst += buf->buf_rowstride - w * 3;
		src += dw - w;
	}

	cairo_destroy (cr);
	cairo_surface_destroy (surf);
	g_free (data);
}

static gint
gnome_canvas_pango_event (GnomeCanvasItem *item, 
					 GdkEvent *event)
{
	return FALSE;
}

static void
gnome_canvas_pango_get_bounds (GnomeCanvasItem *item, double *px1, double *py1,
	   double *px2, double *py2)
{
	GnomeCanvasPango *text = GNOME_CANVAS_PANGO (item);

	adjust_for_anchors(text, px1, py1);

	*px2 = *px1 + ((text->_priv->width > 0)? text->_priv->width: text->_priv->_width);
	*py2 = *py1 + ((text->_priv->height > 0)? text->_priv->height: text->_priv->_height);
}

PangoLayout*
gnome_canvas_pango_get_layout (GnomeCanvasPango *text)
{
	g_return_val_if_fail (GNOME_IS_CANVAS_PANGO (text), NULL);
	return text->_priv->layout;
}

void
gnome_canvas_pango_set_layout (GnomeCanvasPango *text, PangoLayout* layout)
{
	g_return_if_fail (GNOME_IS_CANVAS_PANGO (text));
	if (text->_priv->layout)
		g_object_unref (text->_priv->layout);
	text->_priv->layout = layout;
}

static void
gnome_canvas_pango_print (GPrintable *gprintable, GnomePrintContext *pc)
{
	GnomeCanvasPango *text = GNOME_CANVAS_PANGO (gprintable);
	double ax, ay;
	g_return_if_fail (text);
	adjust_for_anchors (text, &ax, &ay);
	gnome_print_gsave (pc);
	gnome_print_translate (pc, ax, ay);
#ifdef GP_HAS_PANGO
	pango_layout_print (pc, text->_priv->layout);
#else	
	gpc_print_pango_layout_print (pc, text->_priv->layout);
#endif
	gnome_print_grestore (pc);
}

extern void pango_layout_to_svg (PangoLayout* layout, xmlDocPtr doc, xmlNodePtr node, double x, double y);

static void
gnome_canvas_pango_export_svg (GPrintable *gprintable, xmlDocPtr doc, xmlNodePtr node)
{
	GnomeCanvasPango *text = GNOME_CANVAS_PANGO (gprintable);
	double ax, ay;
	g_return_if_fail (text);
	adjust_for_anchors (text, &ax, &ay);
	pango_layout_to_svg (text->_priv->layout, doc, node, ax, ay);
}
