// -*- C++ -*-

/* 
 * GChemPaint text plugin
 * texttool.cc 
 *
 * Copyright (C) 2002-2006 Jean Bréfort <jean.brefort@normalesup.org>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301
 * USA
 */

#include "gchempaint-config.h"
#include "texttool.h"
#include "lib/text.h"
#include "lib/document.h"
#include "lib/application.h"
#include "lib/settings.h"
#include "lib/theme.h"
#include <gdk/gdkkeysyms.h>
#include <unistd.h>

extern GtkTextTagTable* TextTagTable;
extern bool new_ui;

static bool on_toggled (GtkWidget *btn, gcpApplication* App)
{
	gcpTextTool* tool = (gcpTextTool*) App->GetTool ("Text");
	return (tool)? tool->OnToggled (btn): false;
}

static bool on_font (GtkWidget* w, gcpApplication* App)
{
	gcpTextTool* tool = (gcpTextTool*) App->GetTool ("Text");
	return (tool)? tool->OnFont (): false;
}

static void on_get_data (GtkClipboard *clipboard, GtkSelectionData *selection_data,  guint info, gcpTextTool* tool)
{
	tool->OnGetData (clipboard, selection_data, info);
}

gcpTextTool::gcpTextTool (gcpApplication* App, string Id): gcpTool (App, Id)
{
	m_Active = NULL;
	m_bUndo = true;
	m_CurNode = m_InitNode = NULL;
	if (!m_pApp->GetCallback ("texttools")) {
		m_pApp->SetCallback ("texttools", (GCallback) on_toggled);
		m_pApp->SetCallback ("font", (GCallback) on_font);
	}
	m_FontDesc = NULL;
}

gcpTextTool::~gcpTextTool ()
{
	map<string, PangoFontFamily*>::iterator i, iend = m_Families.end ();
	for (i = m_Families.begin (); i != iend; i++) {
		g_object_unref ((*i).second);
	}
	map<string, PangoFontFace*>::iterator j, jend = m_Faces.end ();
	for (j = m_Faces.begin (); j != jend; j++) {
		g_object_unref ((*j).second);
	}
}

bool gcpTextTool::OnClicked()
{
	if (m_Active && ((m_pObject == NULL) || (m_pObject->GetType () != TextType) ||
			(m_Active != g_object_get_data (G_OBJECT (m_pData->Items[m_pObject]), "text")))) {
		Unselect ();
	}
	if (!m_pObject)
	{
		gcpTheme *pTheme = m_pView->GetDoc ()->GetTheme ();
		gcpText *text = new gcpText(m_x0 / pTheme->GetZoomFactor (), m_y0 / pTheme->GetZoomFactor ());
		m_pView->GetDoc()->AddObject(text);
		m_pView->GetDoc()->AbortOperation();
		m_pObject = text;
	}
	if (m_pObject)
	{
		if (m_pObject->GetType() != TextType) return false;
		m_pObject->SetSelected(m_pWidget, SelStateUpdating);
		m_Active = GNOME_CANVAS_RICH_TEXT_EXT(g_object_get_data(G_OBJECT(m_pData->Items[m_pObject]), "text"));
		m_pView->SetGnomeCanvasRichTextActive(m_Active);
		g_object_set(G_OBJECT(m_Active), "editable", true, "cursor_visible", true, NULL);
		m_CurNode = ((gcpText*)m_pObject)->SaveSelected();
		m_InitNode = ((gcpText*)m_pObject)->SaveSelected();
		m_pApp->ActivateMenu ("Image", false);
	}
	return true;
}

bool gcpTextTool::OnEvent (GdkEvent* event)
{
	if (m_Active) {
		if ((event->type == GDK_KEY_PRESS) || (event->type == GDK_KEY_RELEASE)) {
			if (event->key.state & GDK_CONTROL_MASK) {
				switch (event->key.keyval) {
				case GDK_Right:
				case GDK_Left:
				case GDK_Up:
				case GDK_Down:
				case GDK_End:
				case GDK_Home:
				case GDK_Delete:
				case GDK_KP_Delete:
				case GDK_BackSpace:
					break;
				case GDK_a:
					m_pView->OnSelectAll ();
					return true;
				case GDK_z:
					m_pView->GetDoc()->OnUndo ();
					return true;
				case GDK_Z:
					m_pView->GetDoc()->OnRedo ();
					return true;
				case GDK_c:
					CopySelection (gtk_clipboard_get (GDK_SELECTION_CLIPBOARD));
					return true;
				case GDK_v:
					PasteSelection (gtk_clipboard_get (GDK_SELECTION_CLIPBOARD));
					return true;
				case GDK_x:
					CutSelection (gtk_clipboard_get (GDK_SELECTION_CLIPBOARD));
					return true;
				case GDK_i:
					if (new_ui) {
						m_Style = (m_Style == PANGO_STYLE_NORMAL)? PANGO_STYLE_ITALIC: PANGO_STYLE_NORMAL;
						SelectBestFontFace ();
					} else
						gtk_toggle_tool_button_set_active ((GtkToggleToolButton*) m_pApp->GetToolItem ("italic"),
										!gtk_toggle_tool_button_get_active ((GtkToggleToolButton*) m_pApp->GetToolItem ("italic")));
					BuildAttributeList ();
					return true;
				case GDK_u:
					if (new_ui) {
					} else
						gtk_toggle_tool_button_set_active ((GtkToggleToolButton*)m_pApp->GetToolItem ("underline"),
										!gtk_toggle_tool_button_get_active ((GtkToggleToolButton*) m_pApp->GetToolItem ("underline")));
					BuildAttributeList ();
					return true;
				case GDK_b:
					if (new_ui) {
						m_Weight = (m_Weight == PANGO_WEIGHT_NORMAL)? PANGO_WEIGHT_BOLD: PANGO_WEIGHT_NORMAL;
						SelectBestFontFace ();
					} else
						gtk_toggle_tool_button_set_active ((GtkToggleToolButton*) m_pApp->GetToolItem ("bold"),
										!gtk_toggle_tool_button_get_active ((GtkToggleToolButton*) m_pApp->GetToolItem ("bold")));
					BuildAttributeList ();
					return true;
				case GDK_plus:
				case GDK_dead_circumflex:
					if (new_ui) {
					} else
						gtk_toggle_tool_button_set_active ((GtkToggleToolButton*) m_pApp->GetToolItem ("superscript"),
										!gtk_toggle_tool_button_get_active ((GtkToggleToolButton*) m_pApp->GetToolItem ("superscript")));
					BuildAttributeList ();
					return true;
				case GDK_equal:
				case GDK_underscore:
					if (new_ui) {
					} else
						gtk_toggle_tool_button_set_active ((GtkToggleToolButton*) m_pApp->GetToolItem ("subscript"),
										!gtk_toggle_tool_button_get_active ((GtkToggleToolButton*) m_pApp->GetToolItem ("subscript")));
					BuildAttributeList ();
					return true;
				case GDK_space:
					if (new_ui) {
					} else {
						if (gtk_toggle_tool_button_get_active ((GtkToggleToolButton*) m_pApp->GetToolItem ("superscript")))
							gtk_toggle_tool_button_set_active ((GtkToggleToolButton*) m_pApp->GetToolItem ("superscript"), false);
						else if (gtk_toggle_tool_button_get_active ((GtkToggleToolButton*) m_pApp->GetToolItem ("subscript")))
							gtk_toggle_tool_button_set_active ((GtkToggleToolButton*) m_pApp->GetToolItem ("subscript"), false);
					}
					BuildAttributeList ();
					return true;
				default:
					return false;
				}
			}
			if (!g_utf8_validate(((GdkEventKey*)event)->string, -1, NULL))
			{
				gsize r, w;
				gchar* newstr = g_locale_to_utf8(((GdkEventKey*)event)->string, ((GdkEventKey*)event)->length, &r, &w, NULL);
				g_free(((GdkEventKey*)event)->string);
				((GdkEventKey*)event)->string = newstr;
				((GdkEventKey*)event)->length = w;
			}
			gnome_canvas_item_grab_focus((GnomeCanvasItem*)m_Active);
			GnomeCanvasItemClass* klass = GNOME_CANVAS_ITEM_CLASS(((GTypeInstance*)m_Active)->g_class);
			GtkTextBuffer* buf = gnome_canvas_rich_text_ext_get_buffer(m_Active);
			gtk_text_buffer_begin_user_action(buf);
			klass->event((GnomeCanvasItem*)m_Active, event);
			gtk_text_buffer_end_user_action(buf);
			return true;
		}
		else if (event->type == GDK_BUTTON_PRESS)
		{
			 switch (event->button.button)
			 {
				case 2:
				return true;
			}
		}
	}
	return false;
}

void gcpTextTool::Activate()
{
	m_pApp->ActivateToolItem ("bold", true);
	m_pApp->ActivateToolItem ("italic", true);
	m_pApp->ActivateToolItem ("underline", true);
	m_pApp->ActivateToolItem ("strikethrough", true);
	m_pApp->ActivateToolItem ("subscript", true);
	m_pApp->ActivateToolItem ("superscript", true);
	m_pApp->ActivateToolItem ("Font", true);
	gcpTheme *pTheme = m_pApp->GetActiveDocument ()->GetTheme ();
	m_Style = pTheme->GetFontStyle ();
	m_Weight = pTheme->GetFontWeight ();
	m_Stretch = pTheme->GetFontStretch ();
	m_Variant = pTheme->GetFontVariant ();
	m_Size = pTheme->GetFontSize ();
}

bool gcpTextTool::Deactivate()
{
	if (m_Active) Unselect();
	m_pApp->ActivateToolItem ("bold", false);
	m_pApp->ActivateToolItem ("italic", false);
	m_pApp->ActivateToolItem ("underline", false);
	m_pApp->ActivateToolItem ("strikethrough", false);
	m_pApp->ActivateToolItem ("subscript", false);
	m_pApp->ActivateToolItem ("superscript", false);
	m_pApp->ActivateToolItem ("Font", false);
	return true;
}

bool gcpTextTool::NotifyViewChange()
{
	return (m_Active)? Unselect(): true;
}

bool gcpTextTool::Unselect()
{
	if (!m_Active) return true;
	g_object_set(G_OBJECT(m_Active), "editable", false, "cursor_visible", false, NULL);
	m_pView->SetGnomeCanvasRichTextActive(NULL);
	Object *pObj = (Object*)g_object_get_data(G_OBJECT(m_Active), "object");
	pObj->SetSelected(m_pWidget, SelStateUnselected);
	GtkTextBuffer* buf = gnome_canvas_rich_text_ext_get_buffer(m_Active);
	m_Active = NULL;
	GtkTextIter start, end;
	gtk_text_buffer_get_start_iter(buf, &start);
	gtk_text_buffer_place_cursor(buf, &start);
	gtk_text_buffer_get_end_iter(buf, &end);
	char* text = gtk_text_buffer_get_text(buf, &start, &end, true);
	while (!m_UndoList.empty())
	{
		xmlFree(m_UndoList.front());
		m_UndoList.pop_front();
	}
	while (!m_RedoList.empty())
	{
		xmlFree(m_RedoList.front());
		m_RedoList.pop_front();
	}
	xmlBufferPtr initbuf = xmlBufferCreate();
	xmlBufferPtr endbuf = xmlBufferCreate();
	xmlNodeDump (initbuf, m_pApp->GetXmlDoc(), m_InitNode, 0, 0);
	xmlNodeDump (endbuf, m_pApp->GetXmlDoc(), m_CurNode, 0, 0);
	if (strcmp((char*)initbuf->content,(char*)endbuf->content))
	{
		char* initval = (char*) xmlNodeGetContent (m_InitNode);
		char* endval = (char*) xmlNodeGetContent (m_CurNode);
		gcpOperation *pOp = NULL;
		if ((initval && strlen(initval)))
		{
			if (endval && strlen(endval))
			{
				pOp = m_pView->GetDoc()->GetNewOperation(GCP_MODIFY_OPERATION);
				pOp->AddNode(m_InitNode, 0);
				pOp->AddNode(m_CurNode, 1);
				m_CurNode = m_InitNode = NULL;
			}
			else
			{
				pOp = m_pView->GetDoc()->GetNewOperation(GCP_DELETE_OPERATION);
				pOp->AddNode(m_InitNode);
				m_InitNode = NULL;
			}
		}
		else if (endval && strlen(endval))
		{
			pOp = m_pView->GetDoc()->GetNewOperation(GCP_ADD_OPERATION);
			pOp->AddNode(m_CurNode);
			m_CurNode = NULL;
		}
		if (initval)
			xmlFree (initval);
		if (endval)
			xmlFree (endval);
		if (pOp) m_pView->GetDoc()->PushOperation(pOp, m_bUndo);
		m_bUndo = true;
	}
	xmlBufferFree(initbuf);
	xmlBufferFree(endbuf);
	if (m_CurNode) xmlFree(m_CurNode);
	if (m_InitNode) xmlFree(m_InitNode);
	m_CurNode = m_InitNode = NULL;
	if (!*text)
	{
		Object* pMol = pObj->GetMolecule();	//if pObj is a fragment
		if (pMol) pObj = pMol;
		m_pView->GetDoc()->Remove(pObj);
		m_pView->GetDoc()->AbortOperation();
	}
	m_pApp->ActivateMenu ("Image", m_pView->GetDoc()->HasChildren ());
	return true;
}

bool gcpTextTool::OnToggled(GtkWidget *btn)
{
	if (!m_Active) return true;
	if (((gcpTextObject*)g_object_get_data(G_OBJECT(m_Active), "object"))->IsLocked()) return true;
	GtkTextBuffer* buf = gnome_canvas_rich_text_ext_get_buffer(m_Active);
	GtkTextIter start, end;
	gtk_text_buffer_get_selection_bounds(buf, &start, &end);
	const char* name = gtk_widget_get_name (btn);
	if (!strcmp(name, "subscript"))
	{
		if (gtk_toggle_tool_button_get_active(GTK_TOGGLE_TOOL_BUTTON(btn)))
			gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(m_pApp->GetToolItem("superscript")), false);
	}
	else if (!strcmp(name, "superscript"))
	{
		if (gtk_toggle_tool_button_get_active(GTK_TOGGLE_TOOL_BUTTON(btn)))
			gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(m_pApp->GetToolItem("subscript")), false);
	}
	if (gtk_toggle_tool_button_get_active(GTK_TOGGLE_TOOL_BUTTON(btn)))
		gtk_text_buffer_apply_tag_by_name(buf, name, &start, &end);
	else
		gtk_text_buffer_remove_tag_by_name(buf, name, &start, &end);

	gcpText *text = (gcpText*)g_object_get_data(G_OBJECT(m_Active), "object");
	text->OnChanged(buf);
	if (gtk_text_iter_compare(&start, &end)) PushNode(text->SaveSelected());
		
	return true;
}

bool gcpTextTool::OnFont()
{
	GtkTextBuffer* buf = NULL;
	GtkTextIter start, end;
	GtkWidget *dialog = gtk_font_selection_dialog_new("");
	GtkToggleToolButton* BoldBtn = (GtkToggleToolButton*)m_pApp->GetToolItem("bold"),
		*ItalicBtn = (GtkToggleToolButton*)m_pApp->GetToolItem("italic");
	gtk_window_set_modal(GTK_WINDOW(dialog), true);
	PangoFontDescription* pfd = pango_font_description_from_string(m_pApp->GetFontName());
	if (gtk_toggle_tool_button_get_active(BoldBtn)) pango_font_description_set_weight(pfd, PANGO_WEIGHT_BOLD);
	if (gtk_toggle_tool_button_get_active(ItalicBtn)) pango_font_description_set_style(pfd, PANGO_STYLE_ITALIC);
	char* ExactFontName = pango_font_description_to_string(pfd);
	pango_font_description_free(pfd);
	gtk_font_selection_dialog_set_font_name(GTK_FONT_SELECTION_DIALOG(dialog), ExactFontName);
	gint result = gtk_dialog_run(GTK_DIALOG(dialog));
	if (result == GTK_RESPONSE_OK)
	{
		if (m_Active)
		{
			buf = gnome_canvas_rich_text_ext_get_buffer(m_Active);
			gtk_text_buffer_get_selection_bounds(buf, &start, &end);
			gtk_text_buffer_remove_tag_by_name(buf, m_pApp->GetFontName(), &start, &end);
		}
		ExactFontName = gtk_font_selection_dialog_get_font_name(GTK_FONT_SELECTION_DIALOG(dialog));
		pfd =pango_font_description_from_string(ExactFontName);
		if (pango_font_description_get_style(pfd) == PANGO_STYLE_ITALIC)
		{
			pango_font_description_set_style(pfd, PANGO_STYLE_NORMAL);
			gtk_toggle_tool_button_set_active(ItalicBtn, true);
		}
		else gtk_toggle_tool_button_set_active(ItalicBtn, false);
		if (pango_font_description_get_weight(pfd) == PANGO_WEIGHT_BOLD)
		{
			pango_font_description_set_weight(pfd, PANGO_WEIGHT_NORMAL);
			gtk_toggle_tool_button_set_active(BoldBtn, true);
		}
		else gtk_toggle_tool_button_set_active(BoldBtn, false);
		m_pApp->SetFontName(pango_font_description_to_string(pfd));
		GtkTextTag* tag = gtk_text_tag_table_lookup(TextTagTable, m_pApp->GetFontName());
		if (!tag)
		{
			tag = gtk_text_tag_new(m_pApp->GetFontName());
			g_object_set(G_OBJECT(tag),
								"family", pango_font_description_get_family(pfd),
								"size", pango_font_description_get_size(pfd),
								NULL);
			gtk_text_tag_table_add(TextTagTable, tag);
			g_object_unref((GObject*)tag);
		}
		pango_font_description_free(pfd);
		if (m_Active) {
			gtk_text_buffer_apply_tag_by_name(buf, m_pApp->GetFontName(), &start, &end);
			gcpText *text = (gcpText*)g_object_get_data(G_OBJECT(m_Active), "object");
			text->OnChanged(buf);
			if (gtk_text_iter_compare(&start, &end)) PushNode(text->SaveSelected());
		}
	}
	gtk_widget_destroy(dialog);
	return true;
}

bool gcpTextTool::DeleteSelection()
{
	if (!m_Active) return false;
	GtkTextBuffer* buf = gnome_canvas_rich_text_ext_get_buffer(m_Active);
	GtkTextIter start, end;
	gtk_text_buffer_get_selection_bounds(buf, &start, &end);
	if (gtk_text_iter_equal(&start, &end)) return false; //Nothing to delete
	gtk_text_buffer_delete(buf, &start, &end);
	return true;
}

bool gcpTextTool::CopySelection (GtkClipboard *clipboard)
{
	if (!m_Active)
		return false;
	GtkTextBuffer* buf = gnome_canvas_rich_text_ext_get_buffer (m_Active);
	GtkTextIter start, end;
	gtk_text_buffer_get_selection_bounds (buf, &start, &end);
	if (gtk_text_iter_equal (&start, &end))
		return false; //Nothing to copy
	m_pData->Copy (clipboard); //To clean the xmlDoc
	xmlDocPtr pDoc = m_pData->GetXmlDoc (clipboard);
	if (!pDoc)
		return false;
	pDoc->children = xmlNewDocNode (pDoc, NULL, (xmlChar*) "chemistry", NULL);
	gcpText *text = (gcpText*) g_object_get_data (G_OBJECT (m_Active), "object");
	xmlNodePtr node = text->SaveSelection (pDoc);
	if (node)
		xmlAddChild (pDoc->children, node);
	else
		return false;
	gtk_clipboard_set_with_data (clipboard, targets, 2,
		(GtkClipboardGetFunc) on_get_data,
		(GtkClipboardClearFunc) on_clear_data, this);
	gtk_clipboard_request_contents (clipboard,
			gdk_atom_intern ("TARGETS", FALSE),
			(GtkClipboardReceivedFunc) on_receive_targets,
			m_pApp);
	return true;
}

bool gcpTextTool::CutSelection(GtkClipboard *clipboard)
{
	if (!CopySelection(clipboard)) return false;
	return DeleteSelection();
}

bool gcpTextTool::PasteSelection(GtkClipboard *clipboard)
{
	if (!m_Active) return false;
	guint *DataType = (clipboard == gtk_clipboard_get(GDK_SELECTION_CLIPBOARD))? &ClipboardDataType: &ClipboardDataType1;
	GdkAtom targets_atom  = gdk_atom_intern (targets[*DataType].target, FALSE);
	gtk_clipboard_request_contents(clipboard, targets_atom,  (GtkClipboardReceivedFunc)on_receive, m_pView);
	return true;
}

bool gcpTextTool::OnReceive(GtkClipboard *clipboard, GtkSelectionData *data, int type)
{
	if (!m_Active) return false;
	guint *DataType = (clipboard == gtk_clipboard_get(GDK_SELECTION_CLIPBOARD))? &ClipboardDataType: &ClipboardDataType1;
	g_return_val_if_fail((data->target == gdk_atom_intern (targets[*DataType].target, FALSE)), FALSE);
	gcpText *text = (gcpText*)g_object_get_data(G_OBJECT(m_Active), "object");
	GtkTextBuffer* buf = gnome_canvas_rich_text_ext_get_buffer(m_Active);
	GtkTextIter start, end, iter;
	gtk_text_buffer_get_selection_bounds(buf, &start, &end);
	int offset = gtk_text_iter_get_offset(&start);
	switch (*DataType)
	{
		case 0:
		{
			xmlDocPtr xml = xmlParseMemory((const char*)data->data, data->length);
			xmlNodePtr node = xml->children;
			if ((strcmp((char*)node->name, "chemistry")) || (node->children->next))
			{
				xmlFreeDoc(xml);
				return false;
			}
			node = node->children;
			if (!strcmp((char*)node->name, "text"))
			{
				gtk_text_buffer_delete(buf, &start, &end);
				gtk_text_buffer_get_iter_at_offset(buf, &start, offset);
				text->LoadSelection(node, &start);
			}
			else if (!strcmp((char*)node->name, "fragment"))
			{
				gtk_text_buffer_delete(buf, &start, &end);
				gtk_text_buffer_get_iter_at_offset(buf, &iter, offset);
				gcpFragment* fragment = new gcpFragment();
				m_pView->GetDoc()->AddChild(fragment);
				fragment->Load(node);
				GtkTextBuffer* buf0 = (GtkTextBuffer*)fragment->GetTextBuffer();
				gtk_text_buffer_get_bounds(buf0, &start, &end);
				gtk_text_buffer_insert_range(buf, &iter, &start, &end);
				delete fragment;
				text->OnChanged(buf);
			}
			else
			{
				xmlFreeDoc(xml);
				return false;
			}
			xmlFreeDoc(xml);
			return true;
		}
		case 1:
			{
				gtk_text_buffer_delete(buf, &start, &end);
				gtk_text_buffer_get_iter_at_offset(buf, &start, offset);
				gtk_text_buffer_insert(buf, &start, (const char*)data->data, data->length);
			}
			break;
		case 2:
			{
				gtk_text_buffer_get_selection_bounds(buf, &start, &end);
				gtk_text_buffer_delete(buf, &start, &end);
				gtk_text_buffer_get_iter_at_offset(buf, &start, offset);
				if (!g_utf8_validate( (const char*)data->data, data->length, NULL))
				{
					gsize r, w;
					gchar* newstr = g_locale_to_utf8((const char*)data->data, data->length, &r, &w, NULL);
					gtk_text_buffer_insert(buf, &start, newstr, w);
					g_free(newstr);
				}
				else
					gtk_text_buffer_insert(buf, &start, (const char*)data->data, data->length);
			}
			break;
	}
	text->OnChanged(buf);
	return true;
}

bool gcpTextTool::OnUndo()
{
	if (m_UndoList.empty())
	{
		if (m_pView->GetDoc()->CanUndo())
		{
			if (!m_RedoList.empty())
			{
				if (m_CurNode) xmlFree(m_CurNode);
				m_CurNode = m_RedoList.back();
				m_RedoList.pop_back();
			}
			m_bUndo = false;
			Unselect();
		}
		return false;
	}
	xmlNodePtr node = m_UndoList.front();
	gcpTextObject *text = (gcpTextObject*)g_object_get_data(G_OBJECT(m_Active), "object");
	text->LoadSelected(node);
	m_UndoList.pop_front();
	if (m_UndoList.empty() && !m_pView->GetDoc()->CanUndo())
		m_pApp->ActivateMenu("Undo", false);
	m_RedoList.push_front(m_CurNode);
	m_pApp->ActivateMenu("Redo", true);
	GtkTextBuffer* buf = gnome_canvas_rich_text_ext_get_buffer(m_Active);
	GtkTextIter cursor;
	char* tmp = (char*) xmlGetProp (m_CurNode, (xmlChar*) "cursor");
	int offset = (int) strtoul (tmp, NULL, 10);
	xmlFree (tmp);
	gtk_text_buffer_get_iter_at_offset(buf, &cursor, offset);
	gtk_text_buffer_place_cursor(buf, &cursor);
	m_CurNode = node;
	return true;
}

bool gcpTextTool::OnRedo()
{
	if (m_RedoList.empty())
		return false;
	xmlNodePtr node = m_RedoList.front();
	gcpTextObject *text = (gcpTextObject*)g_object_get_data(G_OBJECT(m_Active), "object");
	text->LoadSelected(node);
	m_RedoList.pop_front();
	if (m_RedoList.empty())
		m_pApp->ActivateMenu("Redo", false);
	m_UndoList.push_front(m_CurNode);
	m_pApp->ActivateMenu("Undo", true);
	GtkTextBuffer* buf = gnome_canvas_rich_text_ext_get_buffer(m_Active);
	GtkTextIter cursor;
	char* tmp = (char*) xmlGetProp (m_CurNode, (xmlChar*) "cursor");
	int offset = (int) strtoul (tmp, NULL, 10);
	xmlFree (tmp);
	gtk_text_buffer_get_iter_at_offset(buf, &cursor, offset);
	gtk_text_buffer_place_cursor(buf, &cursor);
	m_CurNode = node;
	return true;
}

void gcpTextTool::PushNode(xmlNodePtr node)
{
	while (!m_RedoList.empty())
	{
		xmlFree(m_RedoList.front());
		m_RedoList.pop_front();
		m_pApp->ActivateMenu("Redo", false);
	}
	m_UndoList.push_front(m_CurNode);
	m_CurNode = node;
	m_pApp->ActivateMenu("Undo", true);
}

void gcpTextTool::OnGetData(GtkClipboard *clipboard, GtkSelectionData *selection_data,  guint info)
{
	xmlDocPtr pDoc = gcpWidgetData::GetXmlDoc(clipboard);
	guint *DataType = (clipboard == gtk_clipboard_get(GDK_SELECTION_CLIPBOARD))? &ClipboardDataType: &ClipboardDataType1;
	if (ClipboardData)
	{
		xmlFree(ClipboardData);
	} 
	*DataType = info;
	gint size;
	if (info)
	{
		gcpText *text = new gcpText();
		text->Load(pDoc->children->children);
		GtkTextBuffer *buf = (GtkTextBuffer*)text->GetTextBuffer();
		GtkTextIter start, end;
		gtk_text_buffer_get_bounds(buf, &start, &end);
		ClipboardData = xmlStrdup((xmlChar*)gtk_text_buffer_get_text(buf, &start, &end, false));
		delete text;
		size = strlen((char*) ClipboardData);
		gtk_selection_data_set_text(selection_data, (const gchar*) ClipboardData, size);
	}
	else
	{
		xmlDocDumpFormatMemory (pDoc, &ClipboardData, &size, info);
		gtk_selection_data_set(selection_data, gdk_atom_intern (GCHEMPAINT_ATOM_NAME, FALSE), 8,  (const guchar*) ClipboardData, size);
	}
	cleared = false;
	if (clipboard == gtk_clipboard_get (GDK_SELECTION_CLIPBOARD)) m_pApp->ActivateMenu("Paste", true);
}

void gcpTextTool::BuildAttributeList ()
{
	m_Dirty = false;
}

static void on_select_family (GtkTreeSelection *selection, gcpTextTool *tool)
{
	tool->OnSelectFamily (selection);
}

static void on_select_face (GtkTreeSelection *selection, gcpTextTool *tool)
{
	tool->OnSelectFace (selection);
}

static void on_select_size (GtkTreeSelection *selection, gcpTextTool *tool)
{
	int size;
	GtkTreeModel *model;
	GtkTreeIter iter;
	gtk_tree_selection_get_selected (selection, &model, &iter);
	gtk_tree_model_get (model, &iter, 0, &size, -1);
	tool->OnSelectSize (size * PANGO_SCALE);
}

static void on_size_activate (GtkEntry *entry, gcpTextTool *tool)
{
}

static void on_size_focus_out (GtkEntry *entry, GdkEventFocus *event, gcpTextTool *tool)
{
}

/* These are what we use as the standard font sizes, for the size list.
 */
static const guint16 font_sizes[] = {
  8, 9, 10, 11, 12, 13, 14, 16, 18, 20, 22, 24, 26, 28,
  32, 36, 40, 48, 56, 64, 72
};

GtkWidget *gcpTextTool::GetPropertyPage ()
{
	GladeXML *xml = glade_xml_new (DATADIR"/gchempaint-unstable/ui/fontsel.glade", "fontsel", GETTEXT_PACKAGE);
	GtkWidget *w = glade_xml_get_widget (xml, "underline");
	gtk_combo_box_set_active (GTK_COMBO_BOX (w), 0); // FIXME: set the current underline style
	PangoFontFamily **families;
	int i, nb;
	gcpTheme *pTheme = m_pApp->GetActiveDocument ()->GetTheme ();
	// Initialize faces list
	m_FaceList = gtk_list_store_new (1, G_TYPE_STRING);
	gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (m_FaceList), 0, GTK_SORT_ASCENDING);
	m_FacesTree = (GtkTreeView *) glade_xml_get_widget (xml, "style");
	gtk_tree_view_set_model (m_FacesTree, GTK_TREE_MODEL (m_FaceList));
	GtkCellRenderer *renderer = gtk_cell_renderer_text_new ();
	GtkTreeViewColumn *column = gtk_tree_view_column_new_with_attributes (NULL, renderer, "text", 0, NULL);
	gtk_tree_view_append_column (m_FacesTree, column);
	GtkTreeSelection *selection = gtk_tree_view_get_selection (m_FacesTree);
	m_FaceSel = selection;
	m_FaceSignal = g_signal_connect (m_FaceSel, "changed", G_CALLBACK (on_select_face), this);
	// Initialize sizes list
	m_SizeList = gtk_list_store_new (1, G_TYPE_INT);
	GtkTreeView *tree = (GtkTreeView *) glade_xml_get_widget (xml, "size-list");
	gtk_tree_view_set_model (tree, GTK_TREE_MODEL (m_SizeList));
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (NULL, renderer, "text", 0, NULL);
	gtk_tree_view_append_column (tree, column);
	GtkTreeIter iter, selected;
	for (i = 0; i < (int) G_N_ELEMENTS (font_sizes); i++) {
		gtk_list_store_append (m_SizeList, &iter);
		gtk_list_store_set (m_SizeList, &iter,
				  0, font_sizes[i],
				  -1);
	}
	selection = gtk_tree_view_get_selection (tree);
	m_SizeSel = selection;
	m_SizeSignal = g_signal_connect (m_SizeSel, "changed", G_CALLBACK (on_select_size), this);
	// Size entry
	m_SizeEntry = (GtkEntry*) glade_xml_get_widget (xml, "size-entry");
	g_signal_connect (m_SizeEntry, "activate", G_CALLBACK (on_size_activate), this);
	g_signal_connect_after (m_SizeEntry, "focus_out_event", G_CALLBACK (on_size_focus_out), this);
	PangoContext *pc = gtk_widget_get_pango_context (w);
	pango_context_list_families (pc, &families, &nb);
	m_FamilyList = gtk_list_store_new (1, G_TYPE_STRING);
	gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (m_FamilyList), 0, GTK_SORT_ASCENDING);
	tree = (GtkTreeView *) glade_xml_get_widget (xml, "family");
	gtk_tree_view_set_model (tree, GTK_TREE_MODEL (m_FamilyList));
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (NULL, renderer, "text", 0, NULL);
	gtk_tree_view_append_column (tree, column);
	GtkTreePath *path = NULL;
	string name;
	bool default_found = false;
	for (i = 0; i < nb; i++) {
		PangoFontFace **faces;
		int *sizes, n;
		pango_font_family_list_faces (families[i], &faces, &n);
		if (n <= 0)
			continue;
		pango_font_face_list_sizes (faces[0], &sizes, &n);
		if (n > 0) // Do not use bitmap fonts
			continue;
		name = pango_font_family_get_name (families[i]);
		m_Families[name] = (PangoFontFamily*) g_object_ref (families[i]);
		gtk_list_store_append (m_FamilyList, &iter);
		gtk_list_store_set (m_FamilyList, &iter,
				  0, name.c_str (),
				  -1);
		if (name == pTheme->GetTextFontFamily ()) {
			selected = iter;
			default_found = true;
		}
	}
	if (!default_found) {
		if	(!gtk_tree_model_get_iter_first (GTK_TREE_MODEL (m_FamilyList), &iter))
			return NULL;
		selected = iter;
	}
	path = gtk_tree_model_get_path (GTK_TREE_MODEL (m_FamilyList), &selected);
	selection = gtk_tree_view_get_selection (tree);
	gtk_tree_selection_set_mode (selection, GTK_SELECTION_BROWSE);
	g_signal_connect (G_OBJECT (selection), "changed", G_CALLBACK (on_select_family), this);
	// Select default font
	if (path) {
		gtk_tree_selection_select_path (selection, path);
		gtk_tree_view_scroll_to_cell (tree, path, column, FALSE, 0., 0.);
		gtk_tree_path_free (path);
	}

	return glade_xml_get_widget (xml, "fontsel");
}

void gcpTextTool::OnSelectFamily (GtkTreeSelection *selection)
{
	GtkTreeModel *model;
	GtkTreeIter iter, selected;
	char const *name;
	if (!gtk_tree_selection_get_selected (selection, &model, &iter))
		return;
	gtk_tree_model_get (model, &iter, 0, &name, -1);
	PangoFontFamily *family = m_Families[name];
	PangoFontFace **faces;
	int i, besti, nb;
	g_signal_handler_block (m_FaceSel, m_FaceSignal);
	pango_font_family_list_faces (family, &faces, &nb);
	gtk_list_store_clear (m_FaceList);
	map<string, PangoFontFace*>::iterator j, jend = m_Faces.end ();
	for (j = m_Faces.begin (); j != jend; j++) {
		g_object_unref ((*j).second);
	}
	m_Faces.clear ();
	PangoFontDescription *desc;
	int distance, best;
	PangoStyle Style;
	PangoWeight Weight;
	PangoVariant Variant;
	PangoStretch Stretch;

	best = 32000; // This should be enough
	for (i = 0; i < nb; i++) {
		name = pango_font_face_get_face_name (faces[i]);
		desc = pango_font_face_describe (faces[i]);
		m_Faces[name] = (PangoFontFace*) g_object_ref (faces[i]);
		gtk_list_store_append (m_FaceList, &iter);
		gtk_list_store_set (m_FaceList, &iter,
				  0, name,
				  -1);
		// Try to select the best available face
		Style = pango_font_description_get_style (desc);
		Weight = pango_font_description_get_weight (desc);
		Variant = pango_font_description_get_variant (desc);
		Stretch = pango_font_description_get_stretch (desc);
		distance = abs (Weight - m_Weight)
						+ abs ((Style? Style + 2: 0) - (m_Style? m_Style + 2: 0)) * 1000
						+ abs (Variant - m_Variant) * 10 + abs (Stretch - m_Stretch);
		if (distance < best) {
			best = distance;
			selected = iter;
			besti = i;
		}
		// TODO: write this code
		pango_font_description_free (desc);
	}
	g_signal_handler_unblock (m_FaceSel, m_FaceSignal);
	GtkTreePath *path = gtk_tree_model_get_path (GTK_TREE_MODEL (m_FaceList), &selected);
	if (path) {
		gtk_tree_selection_select_path (GTK_TREE_SELECTION (m_FaceSel), path);
		gtk_tree_path_free (path);
	} else {
		//TODO: choose a face when default is not available
	}
}

void gcpTextTool::OnSelectFace (GtkTreeSelection *selection)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	char const *name;
	if (!gtk_tree_selection_get_selected (selection, &model, &iter))
		return;
	gtk_tree_model_get (model, &iter, 0, &name, -1);
	PangoFontFace *face = m_Faces[name];
	PangoFontDescription *desc = pango_font_face_describe (face);
	m_Style = pango_font_description_get_style (desc);
	m_Weight = pango_font_description_get_weight (desc);
	m_Variant = pango_font_description_get_variant (desc);
	m_Stretch = pango_font_description_get_stretch (desc);
	pango_font_description_free (desc);
	BuildAttributeList ();
}

void gcpTextTool::OnSelectSize (int size)
{
	BuildAttributeList ();
}

void gcpTextTool::SelectBestFontFace ()
{
	PangoFontDescription *desc;
	int distance, best;
	PangoStyle Style;
	PangoWeight Weight;
	PangoVariant Variant;
	PangoStretch Stretch;
	map <string, PangoFontFace*>::iterator i, iend = m_Faces.end ();
	char const *name = NULL, *buf;
	GtkTreeIter iter;

	best = 32000; // This should be enough
	for (i = m_Faces.begin (); i != iend; i++) {
		desc = pango_font_face_describe ((*i).second);
		// Try to select the best available face
		Style = pango_font_description_get_style (desc);
		Weight = pango_font_description_get_weight (desc);
		Variant = pango_font_description_get_variant (desc);
		Stretch = pango_font_description_get_stretch (desc);
		distance = abs (Weight - m_Weight)
						+ abs ((Style? Style + 2: 0) - (m_Style? m_Style + 2: 0)) * 1000
						+ abs (Variant - m_Variant) * 10 + abs (Stretch - m_Stretch);
		if (distance < best) {
			best = distance;
			name = (*i).first.c_str ();
		}
		pango_font_description_free (desc);
	}
	// select the found face
	gtk_tree_model_get_iter_first (GTK_TREE_MODEL (m_FaceList), &iter);
	do {
		gtk_tree_model_get (GTK_TREE_MODEL (m_FaceList), &iter, 0, &buf, -1);
		if (!strcmp (name, buf)) {
			m_Dirty = true;
			GtkTreePath *path = gtk_tree_model_get_path (GTK_TREE_MODEL (m_FaceList), &iter);
			gtk_tree_view_set_cursor (m_FacesTree, path, NULL, FALSE);
			gtk_tree_path_free (path);
			if (m_Dirty)
				OnSelectFace ((GtkTreeSelection*) m_FaceSel);
			break;
		}
	} while (gtk_tree_model_iter_next (GTK_TREE_MODEL (m_FaceList), &iter));
}
