/* 
 * GChemPaint GOffice component
 * gchempaint.cc
 *
 * Copyright (C) 2005
 *
 * Developed by Jean Bréfort <jean.brefort@normalesup.org>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gchempaint-config.h"
#include "gchempaint.h"
#include "gofficeapp.h"
#include <lib/document.h>

#include <gsf/gsf-impl-utils.h>
#include <goffice/app/module-plugin-defs.h>
#include <libgnomecanvas/gnome-canvas.h>
#include <math.h>
#include <openbabel/mol.h>

gcpGOfficeApplication *app;
static double defzoom;

typedef struct _GOGChemPaintComponent
{
	GOComponent parent;

	gcpDocument *document;
};

extern "C"
{

extern GOPluginModuleDepend const go_plugin_depends [] = {
    { "goffice", GOFFICE_API_VERSION }
};
extern GOPluginModuleHeader const go_plugin_header =
	{ GOFFICE_MODULE_PLUGIN_MAGIC_NUMBER, G_N_ELEMENTS (go_plugin_depends) };

static gboolean
go_gchempaint_component_get_data (GOComponent *component, gpointer *data, int *length,
									void (**clearfunc) (gpointer))
{
	GOGChemPaintComponent *gogcp = GO_GCHEMPAINT_COMPONENT (component);
	xmlDocPtr xml = NULL;
	char *old_num_locale, *old_time_locale;
	bool result;
	
	old_num_locale = g_strdup (setlocale (LC_NUMERIC, NULL));
	setlocale(LC_NUMERIC, "C");
	old_time_locale = g_strdup (setlocale (LC_TIME, NULL));
	setlocale (LC_TIME, "C");

	try {
		xml = gogcp->document->BuildXMLTree ();
		xmlChar *mem;
		int size;
		xmlDocDumpMemory (xml, &mem, &size);
		xmlFreeDoc (xml);
		*data = mem;
		*length = size;
		*clearfunc = xmlFree;
		result = true;
	}
	catch (int) {
		if (xml)
			xmlFreeDoc (xml);
		xml = NULL;
//		Error (SAVE);
		result = false;
	}
	setlocale (LC_NUMERIC, old_num_locale);
	g_free (old_num_locale);
	setlocale (LC_TIME, old_time_locale);
	g_free (old_time_locale);

	return result;
}

static void
go_gchempaint_component_set_data (GOComponent *component, char const *data, int length)
{
	GOGChemPaintComponent *gogcp = GO_GCHEMPAINT_COMPONENT (component);
	gogcp->document = app->ImportDocument (component->mime_type, data, length);
	GtkWidget *w = gogcp->document->GetWidget ();
	gnome_canvas_update_now (GNOME_CANVAS (w));
	ArtDRect rect;
	gcpWidgetData *pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	pData->GetObjectBounds (gogcp->document, &rect);
	double y = gogcp->document->GetYAlign ();
	y += gogcp->document->GetView ()->GetBaseLineOffset ();
	y *= pData->ZoomFactor;
	if (rect.x0 || rect.y0) {
		gogcp->document->Move (- rect.x0 / pData->ZoomFactor, - rect.y0 / pData->ZoomFactor);
		go_component_emit_changed (component);
	}
	gogcp->document->GetView ()->Update (gogcp->document);
	if (y < rect.y0)
		y = rect.y1;
	component->ascent = (y - rect.y0) * defzoom;
	component->descent = (rect.y1 - y) * defzoom;
	component->width = (rect.x1 - rect.x0) * defzoom;
}

static void
go_gchempaint_component_draw (GOComponent *component, double zoom)
{
	GOGChemPaintComponent *gogcp = GO_GCHEMPAINT_COMPONENT (component);
	GtkWidget *w = gogcp->document->GetWidget ();
	gnome_canvas_set_pixels_per_unit (GNOME_CANVAS (w), defzoom * zoom);
	gnome_canvas_update_now (GNOME_CANVAS (w));
	gcpWidgetData *pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	gdk_pixbuf_fill (component->pixbuf, 0xffffffff);
	GnomeCanvasBuf buf;
	buf.buf = gdk_pixbuf_get_pixels (component->pixbuf);
	buf.rect.x0 = 0;
	buf.rect.x1 = gdk_pixbuf_get_width (component->pixbuf);
	buf.rect.y0 = 0;
	buf.rect.y1 = gdk_pixbuf_get_height (component->pixbuf);
	buf.buf_rowstride = gdk_pixbuf_get_rowstride (component->pixbuf);
	buf.bg_color = 0xffffff;
	buf.is_buf = 1;
	(* GNOME_CANVAS_ITEM_GET_CLASS (pData->Group)->render) (GNOME_CANVAS_ITEM (pData->Group), &buf);
}

static void
go_gchempaint_component_print (GOComponent *component, GnomePrintContext *gpc,
												double width, double height)
{
	GOGChemPaintComponent *gogcp = GO_GCHEMPAINT_COMPONENT (component);
	GtkWidget *w = gogcp->document->GetWidget ();
	gnome_canvas_update_now (GNOME_CANVAS (w));
	gogcp->document->Print(gpc, width, height);
}

static gboolean
go_gchempaint_component_edit (GOComponent *component)
{
	return false;
}

static void
go_gchempaint_component_init (GOComponent *component)
{
	component->resizable = false;
	component->editable = true;
}

static void
go_gchempaint_component_class_init (GOComponentClass *klass)
{
	klass->get_data = go_gchempaint_component_get_data;
	klass->set_data = go_gchempaint_component_set_data;
	klass->draw = go_gchempaint_component_draw;
	klass->print = go_gchempaint_component_print;
	klass->edit = go_gchempaint_component_edit;
}

GSF_DYNAMIC_CLASS (GOGChemPaintComponent, go_gchempaint_component,
	go_gchempaint_component_class_init, go_gchempaint_component_init,
	GO_COMPONENT_TYPE)

/*************************************************************************************/

G_MODULE_EXPORT void
go_plugin_init (GOPlugin *plugin, GOCmdContext *cc)
{
	GTypeModule *module = go_plugin_get_type_module (plugin);
	go_gchempaint_component_register_type (module);
	OBMol mol;
	app = new gcpGOfficeApplication ();
	defzoom = GOCXres / 96.;
}

G_MODULE_EXPORT void
go_plugin_shutdown (GOPlugin *plugin, GOCmdContext *cc)
{
	delete app;
}

}	// extern "C"
