// -*- C++ -*-

/* 
 * GChemPaint library
 * widgetdata.cc
 *
 * Copyright (C) 2002-2004 Jean Bréfort <jean.brefort@normalesup.org>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301
 * USA
 */

#include "gchempaint-config.h"
#include "widgetdata.h"
#include "view.h"
#include "settings.h"
#include "document.h"
#include "application.h"
#include "operation.h"

static xmlDocPtr pXmlDoc = NULL, pXmlDoc1 = NULL;
xmlChar* ClipboardData = NULL;
guint ClipboardDataType, ClipboardDataType1;
bool cleared = true;

void on_receive_targets (GtkClipboard *clipboard, GtkSelectionData *selection_data, gcpApplication *App)
{
	GtkClipboard* sel_clipboard = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);
	guint *DataType = (clipboard == sel_clipboard)? &ClipboardDataType: &ClipboardDataType1;
	if (selection_data->target == gdk_atom_intern ("TARGETS", FALSE)) {
		static char const *formats [] =
		{
			GCHEMPAINT_ATOM_NAME,
			"UTF8_STRING",
			"STRING",
			NULL
		};

		GdkAtom const *targets = (GdkAtom *) selection_data->data;
		unsigned const atom_count = (selection_data->length / sizeof (GdkAtom));
		unsigned i, j;
		/* Nothing on clipboard? */
		if (selection_data->length < 0) {
			if (clipboard == sel_clipboard)
				App->ActivateMenu("Paste", false);
			return;
		}

		gchar* name;
		*DataType = 3;
		for ( j = 0; j < atom_count ; j++) {
			name = gdk_atom_name (targets [j]);
			for (i = 0; i < *DataType; i++)
				if (!strcmp (name, formats[i])) {
					*DataType = i;
					break;
				}
			g_free (name);
		}
	}
	if (clipboard == sel_clipboard && App != NULL)
		App->ActivateMenu ("Paste", ClipboardDataType < 3);
}

static void on_get_data(GtkClipboard *clipboard, GtkSelectionData *selection_data,  guint info, gcpApplication *App)
{
	xmlDocPtr pDoc = (clipboard == gtk_clipboard_get(GDK_SELECTION_CLIPBOARD))? pXmlDoc: pXmlDoc1;
	guint *DataType = (clipboard == gtk_clipboard_get(GDK_SELECTION_CLIPBOARD))? &ClipboardDataType: &ClipboardDataType1;
	g_return_if_fail(pDoc);
	if (ClipboardData)
	{
		xmlFree(ClipboardData);
	} 
	*DataType = info;
	gint size;
	if (info)
	{
		xmlKeepBlanksDefault(0);
		xmlDocDumpFormatMemory (pDoc, &ClipboardData, &size, info);
		gtk_selection_data_set_text(selection_data, (const gchar*) ClipboardData, size);
		xmlKeepBlanksDefault(1);
	}
	else
	{
		xmlDocDumpFormatMemory (pDoc, &ClipboardData, &size, info);
		gtk_selection_data_set(selection_data, gdk_atom_intern (GCHEMPAINT_ATOM_NAME, FALSE), 8,  (const guchar*) ClipboardData, size);
	}
	cleared = false;
	if (clipboard == gtk_clipboard_get(GDK_SELECTION_CLIPBOARD)) App->ActivateMenu("Paste", true);
}

void on_clear_data(GtkClipboard *clipboard, gcpApplication *App)
{
	if (ClipboardData)
	{
		xmlFree(ClipboardData);
		ClipboardData = NULL;
	}
	cleared =true;
	gtk_clipboard_request_contents(clipboard, gdk_atom_intern ("TARGETS", FALSE),  (GtkClipboardReceivedFunc)on_receive_targets, App);
}

bool gcpWidgetData::IsSelected(Object *obj)
{
	std::list<Object*>::iterator i;
	Object* pGroup = obj->GetGroup ();
	for (i = SelectedObjects.begin(); i != SelectedObjects.end(); i++)
		if ((*i == obj) || (*i == pGroup)) return true;
	return false;
}

void gcpWidgetData::Unselect(Object* obj)
{
	SelectedObjects.remove(obj);
	obj->SetSelected(Canvas, SelStateUnselected);
	View->Update(obj);
}

void gcpWidgetData::UnselectAll()
{
	Object* obj;
	while (!SelectedObjects.empty())
	{
		obj = SelectedObjects.front();
		SelectedObjects.pop_front();
		obj->SetSelected(Canvas, SelStateUnselected);
		View->Update(obj);//FIXME: is it really useful
	}
}

void gcpWidgetData::SetSelected(Object* obj)
{
	if (!IsSelected (obj)) {
		SelectedObjects.push_front(obj);
		obj->SetSelected(Canvas, SelStateSelected);
	}
}

void gcpWidgetData::MoveSelectedItems(double dx, double dy)
{
	std::list<Object*>::iterator i;
	for (i = SelectedObjects.begin(); i != SelectedObjects.end(); i++) MoveItems(*i, dx, dy);
}

void gcpWidgetData::MoveItems(Object* obj, double dx, double dy)
{
	Object* pObject;
	GnomeCanvasGroup* group;
	if ((group = Items[obj])) gnome_canvas_item_move((GnomeCanvasItem*)group, dx, dy);
	else
		Items.erase(obj);
	std::map<std::string, Object*>::iterator i;
	pObject = obj->GetFirstChild(i);
	while (pObject)
	{
		MoveItems(pObject, dx, dy);
		pObject = obj->GetNextChild(i);
	}
}

void gcpWidgetData::MoveSelection(double dx, double dy)
{
	std::list<Object*>::iterator i;
	gcpDocument* pDoc = View->GetDoc();
	gcpOperation* pOp = pDoc-> GetNewOperation(GCP_MODIFY_OPERATION);
	for (i = SelectedObjects.begin(); i != SelectedObjects.end(); i++)
	{
		pOp->AddObject(*i,0);
		(*i)->Move(dx / ZoomFactor, dy / ZoomFactor);
		View->Update(*i);
		pOp->AddObject(*i,1);
	}
	pDoc->FinishOperation();
}

void gcpWidgetData::RotateSelection(double dx, double dy, double angle)
{
	std::list<Object*>::iterator i;
	Matrix2D m(angle);
	for (i = SelectedObjects.begin(); i != SelectedObjects.end(); i++) {
		(*i)->Transform2D (m, dx / ZoomFactor, dy / ZoomFactor);
		View->Update(*i);
	}
}

void gcpWidgetData::Copy(GtkClipboard* clipboard)
{
	/*First build the XML tree from current selection*/
	xmlDocPtr *pDoc = (clipboard == gtk_clipboard_get(GDK_SELECTION_CLIPBOARD))? &pXmlDoc: &pXmlDoc1;
	if (*pDoc) xmlFreeDoc(*pDoc);
	*pDoc = xmlNewDoc((xmlChar*)"1.0");
	if (!*pDoc) return;
	if (SelectedObjects.empty()) return;
	(*pDoc)->children = xmlNewDocNode(*pDoc, NULL, (xmlChar*)"chemistry", NULL);
//FIXME: implement exception handling
	std::list<Object*>::iterator i;
	xmlNodePtr child;
	for (i = SelectedObjects.begin(); i != SelectedObjects.end(); i++)
		if ((child = (*i)->Save(pXmlDoc))) xmlAddChild((*pDoc)->children, child);
	gcpApplication* App = View->GetDoc()->GetApplication();
	gtk_clipboard_set_with_data(clipboard, targets, 2, (GtkClipboardGetFunc)on_get_data, (GtkClipboardClearFunc)on_clear_data, App);
	gtk_clipboard_request_contents(clipboard, gdk_atom_intern ("TARGETS", FALSE),  (GtkClipboardReceivedFunc)on_receive_targets, App);
}

void gcpWidgetData::GetObjectBounds(Object* obj, ArtDRect &rect)
{
	Object* pObject;
	GnomeCanvasGroup* group;
	double x1, y1, x2,  y2;
	if ((group = Items[obj]))
	{
		gnome_canvas_item_get_bounds(GNOME_CANVAS_ITEM(group), &x1, &y1, &x2, &y2);
		if (rect.x0 < 0)
		{
			rect.x0 = x1;
			rect.y0 = y1;
			rect.x1 = x2;
			rect.y1 = y2;
		}
		else
		{
			if (rect.x0 > x1) rect.x0 = x1;
			if (rect.y0 > y1) rect.y0 = y1;
			if (rect.x1 < x2) rect.x1 = x2;
			if (rect.y1 < y2) rect.y1 = y2;
		}
	}
	else
		Items.erase(obj);
	std::map<std::string, Object*>::iterator i;
	pObject = obj->GetFirstChild(i);
	while (pObject)
	{
		GetObjectBounds(pObject, rect);
		pObject = obj->GetNextChild(i);
	}
}

void gcpWidgetData::GetSelectionBounds(ArtDRect &rect)
{
	std::list<Object*>::iterator i, j;
	rect.x0 = -1.;
	for (i = SelectedObjects.begin(); i != SelectedObjects.end(); i++)
		GetObjectBounds(*i, rect);
}

void gcpWidgetData::GetObjectBounds(Object* obj, ArtDRect *rect)
{
	rect->x0 = -1.;
	GetObjectBounds(obj, *rect);
}

void gcpWidgetData::SelectAll()
{
	std::map<Object*, GnomeCanvasGroup*>::iterator i;
	Object *pObject;
	for (i = Items.begin(); i != Items.end(); i++)
	{
		pObject = (*i).first->GetGroup();
		if (pObject) {
			if (!IsSelected(pObject))
				SetSelected(pObject);
		}
		else if (!IsSelected((*i).first))
			SetSelected((*i).first);
	}
}

xmlDocPtr gcpWidgetData::GetXmlDoc(GtkClipboard* clipboard)
{
	return (clipboard == gtk_clipboard_get(GDK_SELECTION_CLIPBOARD))? pXmlDoc: pXmlDoc1;
}

void gcpWidgetData::ShowSelection(bool state)
{
	std::list<Object*>::iterator i, j;
	for (i = SelectedObjects.begin(); i != SelectedObjects.end(); i++)
		(*i)->SetSelected(Canvas, (state)? SelStateSelected: SelStateUnselected);
}
