// -*- C++ -*-

/* 
 * GChemPaint arrows plugin
 * arrowtool.cc 
 *
 * Copyright (C) 2001-2005
 *
 * Developed by Jean Bréfort <jean.brefort@normalesup.org>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301
 * USA
 */

#include "gchempaint-config.h"
#include "arrowtool.h"
#include "retrosynthesisarrow.h"
#include "lib/settings.h"
#include "libgcpcanvas/gcp-canvas-line.h"
#include "libgcpcanvas/gcp-canvas-group.h"
#include "libgcpcanvas/gcp-canvas-bpath.h"
#include "lib/document.h"
#include "lib/application.h"
#include "lib/mesomery-arrow.h"
#include "lib/reaction-arrow.h"
#include "lib/settings.h"
#include <math.h>
#include <gconf/gconf-client.h>

static char* ToolNames[] = {
	"SimpleArrow",
	"ReversibleArrow",
	"ReversibleArrow",
	"DoubleHeadedArrow",
	"RetrosynthesisArrow"
};


class gcpArrowToolDlg: public gcpDialog
{
public:
	gcpArrowToolDlg (gcpApplication* App, bool FullHeads);
	virtual ~gcpArrowToolDlg ();

	virtual bool Apply ();
	void OnDefault ();
	
private:
	GtkToggleButton *full_btn;
};

static void on_default (gcpArrowToolDlg* dlg)
{
	dlg->OnDefault ();
}

gcpArrowToolDlg::gcpArrowToolDlg (gcpApplication* App, bool FullHeads): gcpDialog(App, DATADIR"/gchempaint/ui/arrowtool.glade", "arrowtool")
{
	GtkTable *table = GTK_TABLE (glade_xml_get_widget (xml, "heads_table"));
	GnomeCanvas *canvas = GNOME_CANVAS (gnome_canvas_new_aa ());
	double width = (DefaultArrowLength + 2 * DefaultArrowPadding) * DefaultZoomFactor,
		height = (DefaultArrowDist + DefaultBondWidth + 2 * (DefaultArrowHeadB + DefaultPadding)) * DefaultZoomFactor;
	gtk_widget_set_size_request (GTK_WIDGET (canvas), (int) width, (int) height);
	GnomeCanvasGroup *group = gnome_canvas_root (canvas);
	GnomeCanvasPoints *points = gnome_canvas_points_new (2);
	gnome_canvas_set_scroll_region(canvas, 0, 0, DefaultArrowLength, DefaultArrowDist + DefaultBondWidth + 2 * DefaultArrowHeadB);
	points->coords[0] = (width - DefaultArrowLength * DefaultZoomFactor) / 2.;
	points->coords[1] = points->coords[3] = (height - DefaultArrowDist * DefaultZoomFactor) / 2.;
	points->coords[2] = (width + DefaultArrowLength * DefaultZoomFactor) / 2.;
	gnome_canvas_item_new (
						group,
						gnome_canvas_line_ext_get_type (),
						"points", points,
						"fill_color", "black",
						"width_units", DefaultBondWidth * DefaultZoomFactor,
						"last_arrowhead", true,
						"arrow_shape_a",DefaultArrowHeadA * DefaultZoomFactor,
						"arrow_shape_b",DefaultArrowHeadB * DefaultZoomFactor,
						"arrow_shape_c",DefaultArrowHeadC * DefaultZoomFactor,
						"last_arrowhead_style", (unsigned char)ARROW_HEAD_LEFT,
						NULL);
	points->coords[0] = (width + DefaultArrowLength * DefaultZoomFactor) / 2.;
	points->coords[1] = points->coords[3] = (height + DefaultArrowDist * DefaultZoomFactor) / 2.;
	points->coords[2] = (width - DefaultArrowLength * DefaultZoomFactor) / 2.;
	gnome_canvas_item_new (
						group,
						gnome_canvas_line_ext_get_type (),
						"points", points,
						"fill_color", "black",
						"width_units",DefaultBondWidth * DefaultZoomFactor,
						"last_arrowhead", true,
						"arrow_shape_a",DefaultArrowHeadA * DefaultZoomFactor,
						"arrow_shape_b",DefaultArrowHeadB * DefaultZoomFactor,
						"arrow_shape_c",DefaultArrowHeadC * DefaultZoomFactor,
						"last_arrowhead_style", (unsigned char)ARROW_HEAD_LEFT,
						NULL);
	gtk_table_attach (table, GTK_WIDGET (canvas), 1, 2, 0, 1, GTK_FILL, GTK_FILL, 10, 0);
	canvas = GNOME_CANVAS (gnome_canvas_new_aa ());
	gtk_widget_set_size_request (GTK_WIDGET (canvas), (int) width, (int) height);
	group = gnome_canvas_root (canvas);
	gnome_canvas_set_scroll_region(canvas, 0, 0, DefaultArrowLength, DefaultArrowDist + DefaultBondWidth + 2 * DefaultArrowHeadB);
	points->coords[0] = (width - DefaultArrowLength * DefaultZoomFactor) / 2.;
	points->coords[1] = points->coords[3] = (height - DefaultArrowDist * DefaultZoomFactor) / 2.;
	points->coords[2] = (width + DefaultArrowLength * DefaultZoomFactor) / 2.;
	gnome_canvas_item_new (
						group,
						gnome_canvas_line_ext_get_type (),
						"points", points,
						"fill_color", "black",
						"width_units", DefaultBondWidth * DefaultZoomFactor,
						"last_arrowhead", true,
						"arrow_shape_a",DefaultArrowHeadA * DefaultZoomFactor,
						"arrow_shape_b",DefaultArrowHeadB * DefaultZoomFactor,
						"arrow_shape_c",DefaultArrowHeadC * DefaultZoomFactor,
						"last_arrowhead_style", (unsigned char)ARROW_HEAD_BOTH,
						NULL);
	points->coords[0] = (width + DefaultArrowLength * DefaultZoomFactor) / 2.;
	points->coords[1] = points->coords[3] = (height + DefaultArrowDist * DefaultZoomFactor) / 2.;
	points->coords[2] = (width - DefaultArrowLength * DefaultZoomFactor) / 2.;
	gnome_canvas_item_new (
						group,
						gnome_canvas_line_ext_get_type (),
						"points", points,
						"fill_color", "black",
						"width_units",DefaultBondWidth * DefaultZoomFactor,
						"last_arrowhead", true,
						"arrow_shape_a",DefaultArrowHeadA * DefaultZoomFactor,
						"arrow_shape_b",DefaultArrowHeadB * DefaultZoomFactor,
						"arrow_shape_c",DefaultArrowHeadC * DefaultZoomFactor,
						"last_arrowhead_style", (unsigned char)ARROW_HEAD_BOTH,
						NULL);
	gtk_table_attach (table, GTK_WIDGET (canvas), 1, 2, 1, 2, GTK_FILL, GTK_FILL, 10, 0);
	gnome_canvas_points_free (points);
	full_btn = GTK_TOGGLE_BUTTON (glade_xml_get_widget (xml, "full"));
	if (FullHeads) gtk_toggle_button_set_active (full_btn, true);
	GtkWidget* w = glade_xml_get_widget (xml, "default");
	g_signal_connect_swapped (w, "clicked", G_CALLBACK (on_default), this);
	App->SetDialog ("ArrowHeads", this);
	gtk_widget_show_all (GTK_WIDGET (dialog));
}

gcpArrowToolDlg::~gcpArrowToolDlg ()
{
	m_App->SetDialog ("ArrowHeads", NULL);
}

bool gcpArrowToolDlg::Apply ()
{
	gcpArrowTool *tool = (gcpArrowTool*) m_App->GetTool ("ReversibleArrow");
	if (tool)
		tool->SetArrowType (gtk_toggle_button_get_active (full_btn)? gcpFullReversibleArrow: gcpReversibleArrow);
	return true;
}

void gcpArrowToolDlg::OnDefault ()
{
	GConfClient *conf_client = gconf_client_get_default ();
	gconf_client_set_bool (conf_client, "/apps/gchempaint/plugins/arrows/full-arrows-heads", gtk_toggle_button_get_active (full_btn), NULL);
	g_object_unref (conf_client);
}

gcpArrowTool::gcpArrowTool(gcpApplication* App, unsigned ArrowType): gcpTool(App, ToolNames[ArrowType])
{
	points = gnome_canvas_points_new(2);
	m_ArrowType = ArrowType;
}

gcpArrowTool::~gcpArrowTool()
{
	gnome_canvas_points_free(points);
}	

bool gcpArrowTool::OnClicked()
{
	if (m_pObject) return false;
	m_y1 = m_y0;
	m_x1 = m_x0 + m_pData->ArrowLength;
	switch(m_ArrowType)
	{
		case gcpSimpleArrow:
			points->coords[0] = m_x0;
			points->coords[1] = m_y0;
			points->coords[2] = m_x1;
			points->coords[3] = m_y0;
			m_pItem = gnome_canvas_item_new(
										m_pGroup,
										gnome_canvas_line_ext_get_type(),
										"points", points,
										"fill_color", AddColor,
										"width_units", m_pData->BondWidth,
										"last_arrowhead", true,
										"arrow_shape_a", m_pData->ArrowHeadA,
										"arrow_shape_b", m_pData->ArrowHeadB,
										"arrow_shape_c", m_pData->ArrowHeadC,
										"last_arrowhead_style", (unsigned char)ARROW_HEAD_BOTH,
										NULL);
			break;
		case gcpReversibleArrow:
			points->coords[0] = m_x0;
			points->coords[1] = points->coords[3] = m_y0 - m_pData->ArrowDist / 2;
			points->coords[2] = m_x1;
			m_pItem = gnome_canvas_item_new(m_pGroup, gnome_canvas_group_ext_get_type(), NULL);
			gnome_canvas_item_new(
								GNOME_CANVAS_GROUP(m_pItem),
								gnome_canvas_line_ext_get_type(),
								"points", points,
								"fill_color", AddColor,
								"width_units", m_pData->BondWidth,
								"last_arrowhead", true,
								"arrow_shape_a", m_pData->ArrowHeadA,
								"arrow_shape_b", m_pData->ArrowHeadB,
								"arrow_shape_c", m_pData->ArrowHeadC,
								"last_arrowhead_style", (unsigned char)ARROW_HEAD_LEFT,
								NULL);
			points->coords[2] = m_x0;
			points->coords[1] = points->coords[3] = m_y0 + m_pData->ArrowDist / 2;
			points->coords[0] = m_x0 + m_pData->ArrowLength;
			gnome_canvas_item_new(
								GNOME_CANVAS_GROUP(m_pItem),
								gnome_canvas_line_ext_get_type(),
								"points", points,
								"fill_color", AddColor,
								"width_units", m_pData->BondWidth,
								"last_arrowhead", true,
								"arrow_shape_a", m_pData->ArrowHeadA,
								"arrow_shape_b", m_pData->ArrowHeadB,
								"arrow_shape_c", m_pData->ArrowHeadC,
								"last_arrowhead_style", (unsigned char)ARROW_HEAD_LEFT,
								NULL);
			break;
		case gcpFullReversibleArrow:
			points->coords[0] = m_x0;
			points->coords[1] = points->coords[3] = m_y0 - m_pData->ArrowDist / 2;
			points->coords[2] = m_x1;
			m_pItem = gnome_canvas_item_new(m_pGroup, gnome_canvas_group_ext_get_type(), NULL);
			gnome_canvas_item_new(
								GNOME_CANVAS_GROUP(m_pItem),
								gnome_canvas_line_ext_get_type(),
								"points", points,
								"fill_color", AddColor,
								"width_units", m_pData->BondWidth,
								"last_arrowhead", true,
								"arrow_shape_a", m_pData->ArrowHeadA,
								"arrow_shape_b", m_pData->ArrowHeadB,
								"arrow_shape_c", m_pData->ArrowHeadC,
								"last_arrowhead_style", (unsigned char)ARROW_HEAD_BOTH,
								NULL);
			points->coords[2] = m_x0;
			points->coords[1] = points->coords[3] = m_y0 + m_pData->ArrowDist / 2;
			points->coords[0] = m_x0 + m_pData->ArrowLength;
			gnome_canvas_item_new(
								GNOME_CANVAS_GROUP(m_pItem),
								gnome_canvas_line_ext_get_type(),
								"points", points,
								"fill_color", AddColor,
								"width_units", m_pData->BondWidth,
								"last_arrowhead", true,
								"arrow_shape_a", m_pData->ArrowHeadA,
								"arrow_shape_b", m_pData->ArrowHeadB,
								"arrow_shape_c", m_pData->ArrowHeadC,
								"last_arrowhead_style", (unsigned char)ARROW_HEAD_BOTH,
								NULL);
			break;
		case gcpDoubleHeadedArrow:
			points->coords[0] = m_x0;
			points->coords[1] = m_y0;
			points->coords[2] = m_x1;
			points->coords[3] = m_y0;
			m_pItem = gnome_canvas_item_new(
										m_pGroup,
										gnome_canvas_line_ext_get_type(),
										"points", points,
										"fill_color", AddColor,
										"width_units", m_pData->BondWidth,
										"first_arrowhead", true,
										"last_arrowhead", true,
										"arrow_shape_a", m_pData->ArrowHeadA,
										"arrow_shape_b", m_pData->ArrowHeadB,
										"arrow_shape_c", m_pData->ArrowHeadC,
										"first_arrowhead_style", (unsigned char)ARROW_HEAD_BOTH,
										"last_arrowhead_style", (unsigned char)ARROW_HEAD_BOTH,
										NULL);
			break;
		case gcpDoubleQueuedArrow: {
			GnomeCanvasPathDef* path = gnome_canvas_path_def_new ();
			gnome_canvas_path_def_moveto (path, m_x0, m_y0 - m_pData->ArrowDist / 2.);
			gnome_canvas_path_def_lineto (path, m_x1 - m_pData->ArrowDist / 2., m_y0 - m_pData->ArrowDist / 2.);
			gnome_canvas_path_def_moveto (path, m_x0, m_y0 + m_pData->ArrowDist / 2.);
			gnome_canvas_path_def_lineto (path, m_x1 - m_pData->ArrowDist / 2., m_y0 + m_pData->ArrowDist / 2.);
			gnome_canvas_path_def_moveto (path, m_x1 - m_pData->ArrowDist / 2. - m_pData->ArrowHeadC, m_y0 - m_pData->ArrowDist / 2. - m_pData->ArrowHeadC);
			gnome_canvas_path_def_lineto (path, m_x1, m_y0);
			gnome_canvas_path_def_lineto (path, m_x1 - m_pData->ArrowDist / 2. - m_pData->ArrowHeadC, m_y0 + m_pData->ArrowDist / 2. + m_pData->ArrowHeadC);
			m_pItem = gnome_canvas_item_new(
									m_pGroup,
									gnome_canvas_bpath_ext_get_type(),
									"bpath", path,
									"outline_color", AddColor,
									"width_units", m_pData->BondWidth,
									"cap-style", GDK_CAP_BUTT,
									"join-style", GDK_JOIN_MITER,
									NULL);
			break;
		}
	}
	return true;
}

void gcpArrowTool::OnDrag()
{
	double x1, y1, x2, y2;
	if (m_pItem)
	{
		gnome_canvas_item_get_bounds(GNOME_CANVAS_ITEM(m_pItem), &x1, &y1, &x2, &y2);
		gtk_object_destroy(GTK_OBJECT(GNOME_CANVAS_ITEM(m_pItem)));
		gnome_canvas_request_redraw(GNOME_CANVAS(m_pWidget), (int)x1, (int)y1, (int)x2, (int)y2);
		m_pItem = NULL;
	}
	double dAngle;
	m_x -= m_x0;
	m_y -= m_y0;
	if (m_x == 0)
	{
		if (m_y == 0) return;
		dAngle = (m_y < 0) ? 90 : 270;
	}
	else
	{
		dAngle = atan(-m_y/m_x) * 180 / M_PI;
		if (!(m_nState & GDK_CONTROL_MASK)) dAngle = rint(dAngle / 5) * 5;
		if (m_x < 0) dAngle += 180;
	}
	m_dAngle = dAngle * M_PI / 180;
	double d = (m_nState & GDK_SHIFT_MASK)? sqrt(square(m_x) + square(m_y)): m_pData->ArrowLength;
	char tmp[32];
	if (dAngle < 0) dAngle += 360;
	snprintf(tmp, sizeof(tmp) - 1, _("Orientation: %g"), dAngle);
	m_pApp->SetStatusText(tmp);
	m_x1 = m_x0 + d * cos(m_dAngle);
	m_y1 = m_y0 - d * sin(m_dAngle);
	switch(m_ArrowType)
	{
		case gcpSimpleArrow:
			points->coords[2] = m_x1;
			points->coords[3] = m_y1;
			m_pItem = gnome_canvas_item_new(
										m_pGroup,
										gnome_canvas_line_ext_get_type(),
										"points", points,
										"fill_color", AddColor,
										"width_units", m_pData->BondWidth,
										"last_arrowhead", true,
										"arrow_shape_a", m_pData->ArrowHeadA,
										"arrow_shape_b", m_pData->ArrowHeadB,
										"arrow_shape_c", m_pData->ArrowHeadC,
										"last_arrowhead_style", (unsigned char)ARROW_HEAD_BOTH,
										NULL);
			break;
		case gcpReversibleArrow:
			points->coords[0] = m_x0 - m_pData->ArrowDist / 2 * sin(m_dAngle);
			points->coords[1] = m_y0 - m_pData->ArrowDist / 2 * cos(m_dAngle);
			points->coords[2] = m_x1 - m_pData->ArrowDist / 2 * sin(m_dAngle);
			points->coords[3] = m_y1 - m_pData->ArrowDist / 2 * cos(m_dAngle);
			m_pItem = gnome_canvas_item_new(m_pGroup, gnome_canvas_group_ext_get_type(), NULL);
			gnome_canvas_item_new(
								GNOME_CANVAS_GROUP(m_pItem),
								gnome_canvas_line_ext_get_type(),
								"points", points,
								"fill_color", AddColor,
								"width_units", m_pData->BondWidth,
								"last_arrowhead", true,
								"arrow_shape_a", m_pData->ArrowHeadA,
								"arrow_shape_b", m_pData->ArrowHeadB,
								"arrow_shape_c", m_pData->ArrowHeadC,
								"last_arrowhead_style", (unsigned char)ARROW_HEAD_LEFT,
								NULL);
			points->coords[2] = m_x0 + m_pData->ArrowDist / 2 * sin(m_dAngle);
			points->coords[3] = m_y0 + m_pData->ArrowDist / 2 * cos(m_dAngle);
			points->coords[0] = m_x1 + m_pData->ArrowDist / 2 * sin(m_dAngle);
			points->coords[1] = m_y1 + m_pData->ArrowDist / 2 * cos(m_dAngle);
			gnome_canvas_item_new(
								GNOME_CANVAS_GROUP(m_pItem),
								gnome_canvas_line_ext_get_type(),
								"points", points,
								"fill_color", AddColor,
								"width_units", m_pData->BondWidth,
								"last_arrowhead", true,
								"arrow_shape_a", m_pData->ArrowHeadA,
								"arrow_shape_b", m_pData->ArrowHeadB,
								"arrow_shape_c", m_pData->ArrowHeadC,
								"last_arrowhead_style", (unsigned char)ARROW_HEAD_LEFT,
								NULL);
			break;
		case gcpFullReversibleArrow:
			points->coords[0] = m_x0 - m_pData->ArrowDist / 2 * sin(m_dAngle);
			points->coords[1] = m_y0 - m_pData->ArrowDist / 2 * cos(m_dAngle);
			points->coords[2] = m_x1 - m_pData->ArrowDist / 2 * sin(m_dAngle);
			points->coords[3] = m_y1 - m_pData->ArrowDist / 2 * cos(m_dAngle);
			m_pItem = gnome_canvas_item_new(m_pGroup, gnome_canvas_group_ext_get_type(), NULL);
			gnome_canvas_item_new(
								GNOME_CANVAS_GROUP(m_pItem),
								gnome_canvas_line_ext_get_type(),
								"points", points,
								"fill_color", AddColor,
								"width_units", m_pData->BondWidth,
								"last_arrowhead", true,
								"arrow_shape_a", m_pData->ArrowHeadA,
								"arrow_shape_b", m_pData->ArrowHeadB,
								"arrow_shape_c", m_pData->ArrowHeadC,
								"last_arrowhead_style", (unsigned char)ARROW_HEAD_BOTH,
								NULL);
			points->coords[2] = m_x0 + m_pData->ArrowDist / 2 * sin(m_dAngle);
			points->coords[3] = m_y0 + m_pData->ArrowDist / 2 * cos(m_dAngle);
			points->coords[0] = m_x1 + m_pData->ArrowDist / 2 * sin(m_dAngle);
			points->coords[1] = m_y1 + m_pData->ArrowDist / 2 * cos(m_dAngle);
			gnome_canvas_item_new(
								GNOME_CANVAS_GROUP(m_pItem),
								gnome_canvas_line_ext_get_type(),
								"points", points,
								"fill_color", AddColor,
								"width_units", m_pData->BondWidth,
								"last_arrowhead", true,
								"arrow_shape_a", m_pData->ArrowHeadA,
								"arrow_shape_b", m_pData->ArrowHeadB,
								"arrow_shape_c", m_pData->ArrowHeadC,
								"last_arrowhead_style", (unsigned char)ARROW_HEAD_BOTH,
								NULL);
			break;
		case gcpDoubleHeadedArrow:
			points->coords[2] = m_x1;
			points->coords[3] = m_y1;
			m_pItem = gnome_canvas_item_new(
										m_pGroup,
										gnome_canvas_line_ext_get_type(),
										"points", points,
										"fill_color", AddColor,
										"width_units", m_pData->BondWidth,
										"first_arrowhead", true,
										"last_arrowhead", true,
										"arrow_shape_a", m_pData->ArrowHeadA,
										"arrow_shape_b", m_pData->ArrowHeadB,
										"arrow_shape_c", m_pData->ArrowHeadC,
										"first_arrowhead_style", (unsigned char)ARROW_HEAD_BOTH,
										"last_arrowhead_style", (unsigned char)ARROW_HEAD_BOTH,
										NULL);
			break;
		case gcpDoubleQueuedArrow: {
			double x1, y1;
			x1 = m_pData->ArrowDist / 2 * sin (m_dAngle);
			y1 = m_pData->ArrowDist / 2 * cos (m_dAngle);
			GnomeCanvasPathDef* path = gnome_canvas_path_def_new ();
			gnome_canvas_path_def_moveto (path, m_x0 - x1, m_y0 - y1);
			gnome_canvas_path_def_lineto (path, m_x1 - x1 - y1, m_y1 - y1 + x1);
			gnome_canvas_path_def_moveto (path, m_x0 + x1, m_y0 + y1);
			gnome_canvas_path_def_lineto (path, m_x1 + x1 - y1, m_y1 + y1 + x1);
			x1 += m_pData->ArrowHeadC * sin (m_dAngle);
			y1 += m_pData->ArrowHeadC * cos (m_dAngle);
			gnome_canvas_path_def_moveto (path, m_x1 - x1 - y1, m_y1 - y1 + x1);
			gnome_canvas_path_def_lineto (path, m_x1, m_y1);
			gnome_canvas_path_def_lineto (path, m_x1 + x1 - y1, m_y1 + y1 + x1);
			m_pItem = gnome_canvas_item_new(
									m_pGroup,
									gnome_canvas_bpath_ext_get_type(),
									"bpath", path,
									"outline_color", AddColor,
									"width_units", m_pData->BondWidth,
									"cap-style", GDK_CAP_BUTT,
									"join-style", GDK_JOIN_MITER,
									NULL);
			break;
		}
	}
}

void gcpArrowTool::OnRelease()
{
	double x1, y1, x2, y2;
	if (m_pItem)
	{
		gnome_canvas_item_get_bounds(GNOME_CANVAS_ITEM(m_pItem), &x1, &y1, &x2, &y2);
		gtk_object_destroy(GTK_OBJECT(GNOME_CANVAS_ITEM(m_pItem)));
		gnome_canvas_request_redraw(GNOME_CANVAS(m_pWidget), (int)x1, (int)y1, (int)x2, (int)y2);
		m_pItem = NULL;
	}
	else return;
	m_pApp->ClearStatus();
	gcpDocument* pDoc = m_pView->GetDoc();
	gcpArrow* a;
	switch(m_ArrowType)
	{
	case gcpDoubleHeadedArrow:
		{
			a = new gcpMesomeryArrow(NULL);
			a->SetCoords(m_x0 / m_dZoomFactor, m_y0 / m_dZoomFactor, m_x1 / m_dZoomFactor, m_y1 / m_dZoomFactor);
			pDoc->AddObject(a);
		}
		break;
	case gcpDoubleQueuedArrow: {
			a = new gcpRetrosynthesisArrow (NULL);
			a->SetCoords(m_x0 / m_dZoomFactor, m_y0 / m_dZoomFactor, m_x1 / m_dZoomFactor, m_y1 / m_dZoomFactor);
			pDoc->AddObject(a);
		}
		break;
	default:
		{
			a = new gcpReactionArrow(NULL, m_ArrowType);
			a->SetCoords(m_x0 / m_dZoomFactor, m_y0 / m_dZoomFactor, m_x1 / m_dZoomFactor, m_y1 / m_dZoomFactor);
			pDoc->AddObject(a);
		}
	}
	pDoc->FinishOperation();
}

void gcpArrowTool::SetOptions ()
{
	if (m_ArrowType != gcpReversibleArrow && m_ArrowType != gcpFullReversibleArrow)
		return;
	gcpArrowToolDlg* dlg = (gcpArrowToolDlg*) m_pApp->GetDialog("ArrowHeads");
	if (!dlg)
	{
		dlg = new gcpArrowToolDlg (m_pApp, m_ArrowType == gcpFullReversibleArrow);
	}
	else gdk_window_raise (GTK_WIDGET(dlg->GetWindow())->window); 
}
