// -*- C++ -*-

/* 
 * GChemPaint templates plugin
 * templatetool.cc 
 *
 * Copyright (C) 2004-2005
 *
 * Developed by Jean Bréfort <jean.brefort@ac-dijon.fr>
 *
 * Developed by Jean Bréfort <jean.brefort@normalesup.org>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301
 * USA
 */

#include "templatetool.h"
#include "templatetree.h"
#include "lib/document.h"
#include <gtk/gtk.h>

xmlDocPtr xml = xmlNewDoc ((const xmlChar*) "1.0");

class gcpNewTemplateToolDlg: public gcpDialog
{
public:
	gcpNewTemplateToolDlg(gcpApplication* App);
	virtual ~gcpNewTemplateToolDlg();

	virtual bool Apply();
	void SetTemplate (xmlNodePtr node);
	
private:
	gcpTemplate *temp;
	gcpDocument *pDoc;
	gcpWidgetData* pData;
	xmlNodePtr m_node;
	GtkEntry *category_entry;
};

gcpTemplateTool::gcpTemplateTool(gcpApplication* App): gcpTool(App, "Templates")
{
	m_Template = NULL;
}

gcpTemplateTool::~gcpTemplateTool()
{
}

bool gcpTemplateTool::OnClicked()
{
	gcpNewTemplateToolDlg *dlg = (gcpNewTemplateToolDlg*) m_pApp->GetDialog ("New Template");
	if (dlg) {
		m_pObject = m_pObject->GetMolecule ();
		if (m_pObject) {
			xmlNodePtr node = m_pObject->Save (xml);
			if (node) {
				dlg->SetTemplate (node);
				gdk_window_raise (GTK_WIDGET(dlg->GetWindow())->window);
			}
		}
		return false;
	}
	if (!m_Template) return false;
	gcpDocument* pDoc = m_pView->GetDoc ();
	pDoc->AddData (m_Template->node);
	m_pObject = m_pData->SelectedObjects.front();
	ArtDRect rect;
	double dx, dy;
//	while(gtk_events_pending()) gtk_main_iteration();
	gnome_canvas_update_now (GNOME_CANVAS (m_pWidget));
	pDoc->AbortOperation();
	m_pData->GetSelectionBounds(rect);
	dx = m_x0 - (rect.x0 + rect.x1) / 2.;
	dy = m_y0 - (rect.y0 + rect.y1) / 2.;
	m_x0 -= dx;
	m_y0 -= dy;
	m_pData->MoveSelectedItems (dx, dy);
	return true;
}

void gcpTemplateTool::OnDrag()
{
	double dx = m_x - m_x1, dy = m_y - m_y1;
	m_x1 = m_x;
	m_y1 = m_y;
	m_pData->MoveSelectedItems (dx, dy);
}

void gcpTemplateTool::OnRelease()
{
	gcpDocument* pDoc = m_pView->GetDoc ();
	double dx = m_x - m_x0, dy = m_y - m_y0;
	m_pData->MoveSelectedItems(-dx, -dy);
	m_pData->MoveSelection(dx, dy);
	pDoc->PopOperation ();
	m_pData->UnselectAll ();
	gcpOperation* pOp = pDoc->GetNewOperation (GCP_ADD_OPERATION);
	pOp->AddObject (m_pObject);
	pDoc->FinishOperation ();
}

class gcpTemplateToolDlg: public gcpDialog
{
public:
	gcpTemplateToolDlg(gcpApplication* App, gcpTemplate* t);
	virtual ~gcpTemplateToolDlg();

	virtual bool Apply();
	void OnChanged(GtkTreeSelection *Selection);
	void OnAddTemplate ();
	void OnDeleteTemplate ();
	void SetTemplate (gcpTemplate* t);
	
private:
	GtkTreeModel* model;
	gcpTemplateTree* tree;
	gcpTemplate *temp;
	gcpDocument *pDoc;
	gcpWidgetData* pData;
};

void gcpTemplateTool::SetOptions()
{
	gcpTemplateToolDlg* dlg = (gcpTemplateToolDlg*) m_pApp->GetDialog("Templates");
	if (!dlg)
		dlg = new gcpTemplateToolDlg (m_pApp, m_Template);
	else gdk_window_raise (GTK_WIDGET(dlg->GetWindow())->window); 
}

void gcpTemplateTool::SetTemplate (gcpTemplate* temp)
{
	if (temp == m_Template)
		return;
	m_Template = temp;
	gcpTemplateToolDlg* dlg = (gcpTemplateToolDlg*) m_pApp->GetDialog("Templates");
	if (dlg)
		dlg->SetTemplate (temp);
}

enum {
	NAME_COLUMN,
	NUM_COLUMNS
};

static void on_changed(GtkTreeSelection *Selection, gcpTemplateToolDlg* dlg)
{
	dlg->OnChanged(Selection);
}

static void on_add_template (GtkWidget *w, gcpTemplateToolDlg *dlg)
{
	dlg->OnAddTemplate ();
}

static void on_delete_template (GtkWidget *w, gcpTemplateToolDlg *dlg)
{
	dlg->OnDeleteTemplate ();
}

gcpTemplateToolDlg::gcpTemplateToolDlg(gcpApplication* App, gcpTemplate *t): gcpDialog(App, DATADIR"/gchempaint/ui/templates.glade", "templates")
{
	tree = (gcpTemplateTree*) App->GetTool ("TemplateTree");
	if (!tree) {delete this; return;}
	GtkTreeView* view = (GtkTreeView*)glade_xml_get_widget(xml, "template_tree_view");
	gtk_tree_view_set_model (view, model = tree->GetModel());
	GtkTreeSelection* Selection = gtk_tree_view_get_selection(view);
	GtkCellRenderer* renderer = (GtkCellRenderer*)gtk_cell_renderer_text_new ();
	g_object_set (renderer, "xalign", 0.0, NULL);
	gtk_tree_view_insert_column_with_attributes (view,
								-1, "Template name",
								renderer, "text",
								NAME_COLUMN,
								NULL);
	g_signal_connect (G_OBJECT(Selection), "changed", G_CALLBACK (on_changed), this);
	temp = t;
	pDoc = new gcpDocument(m_App, true);
	pDoc->SetEditable (false);
	GtkWidget* w;
	GtkScrolledWindow* scroll = (GtkScrolledWindow*)glade_xml_get_widget(xml, "scrolledcanvas");
	gtk_scrolled_window_add_with_viewport(scroll, w = pDoc->GetWidget());
	pData = (gcpWidgetData*) g_object_get_data (G_OBJECT (w), "data");
	if (t) {
		const char* path_string = tree->GetPath(t);
		if (path_string && *path_string) {
			GtkTreePath* path = gtk_tree_path_new_from_string (path_string);
			gtk_tree_view_expand_to_path (view, path);
			gtk_tree_selection_select_path (Selection, path);
			gtk_tree_path_free (path);
			pDoc->AddData (t->node);
			pData->UnselectAll ();
			g_signal_emit_by_name (G_OBJECT (w), "update_bounds");
			gtk_widget_set_sensitive (glade_xml_get_widget (xml, "delete"), t->writeable);
		}
		else
			gtk_widget_set_sensitive (glade_xml_get_widget (xml, "delete"), false);
	} else
		gtk_widget_set_sensitive (glade_xml_get_widget (xml, "delete"), false);
	App->SetDialog("Templates", this);
	gtk_widget_show_all(GTK_WIDGET(dialog));
	w = glade_xml_get_widget(xml, "add");
	g_signal_connect (G_OBJECT (w), "clicked", G_CALLBACK (on_add_template), this);
	w = glade_xml_get_widget(xml, "delete");
	g_signal_connect (G_OBJECT (w), "clicked", G_CALLBACK (on_delete_template), this);
}

gcpTemplateToolDlg::~gcpTemplateToolDlg()
{
	m_App->SetDialog("Templates", NULL);
}

bool gcpTemplateToolDlg::Apply()
{
	gcpTemplateTool* tool = (gcpTemplateTool*) m_App->GetTool ("Templates");
	tool->SetTemplate (temp);
	tree->SetTemplate (temp);
	return true;
}

void gcpTemplateToolDlg::OnChanged(GtkTreeSelection *Selection)
{
	GtkTreeIter iter;
	GtkTreePath *path;
	char* path_string;
	if (gtk_tree_selection_get_selected(Selection, &model, &iter)) {
		path = gtk_tree_model_get_path(model, &iter);
		path_string = gtk_tree_path_to_string (path);
		temp = tree->GetTemplate (path_string);
		g_free (path_string);
		gtk_tree_path_free (path);
		map<string, Object*>::iterator it;
		Object* obj;
		while (obj = pDoc->GetFirstChild(it)) {
			pDoc->Remove (obj);
			pDoc->PopOperation ();
		}
		if (temp) {
			pDoc->AddData (temp->node);
			pData->UnselectAll ();
			pDoc->GetView () ->OnSize (0, 0);
			g_signal_emit_by_name (G_OBJECT (pDoc->GetWidget ()), "update_bounds");
			gtk_widget_set_sensitive (glade_xml_get_widget (xml, "delete"), temp->writeable);
		}
		else
			gtk_widget_set_sensitive (glade_xml_get_widget (xml, "delete"), false);
	}
}

void gcpTemplateToolDlg::OnAddTemplate ()
{
	gcpNewTemplateToolDlg* dlg = (gcpNewTemplateToolDlg*) m_App->GetDialog("New Template");
	if (!dlg)
	{
		dlg = new gcpNewTemplateToolDlg (m_App);
	}
	else gdk_window_raise (GTK_WIDGET(dlg->GetWindow())->window); 
}

void gcpTemplateToolDlg::OnDeleteTemplate ()
{
	gcpTemplateTool* tool = (gcpTemplateTool*) m_App->GetTool ("Templates");
	if (temp == tool->GetTemplate()) {
		tool->SetTemplate (NULL);
		tree->SetTemplate (NULL);
	}
	string key = temp->category + "/" + temp->name;
	if (Templates[key] != temp) {
		int i = 0;
		char* str = g_strdup_printf ("%d", i);
		while (Templates[key + str] != temp) {
			g_free (str);
			str = g_strdup_printf ("%d", ++i);
		}
		key += str;
		g_free (str);
	}
	tree->DeleteTemplate (key);
	map<string, Object*>::iterator it;
	Object* obj = pDoc->GetFirstChild(it);
	if (obj) pDoc->Remove (obj);
	pDoc->PopOperation ();
	temp = NULL;
}

void gcpTemplateToolDlg::SetTemplate (gcpTemplate* t)
{
	tree = (gcpTemplateTree*) m_App->GetTool ("TemplateTree");
	if (tree && t) {
	GtkTreeView* view = (GtkTreeView*)glade_xml_get_widget(xml, "template_tree_view");
	GtkTreeSelection* Selection = gtk_tree_view_get_selection(view);
		const char* path_string = tree->GetPath(t);
		if (path_string && *path_string) {
			GtkTreePath* path = gtk_tree_path_new_from_string (path_string);
			gtk_tree_view_expand_to_path (view, path);
			gtk_tree_selection_select_path (Selection, path);
			gtk_tree_path_free (path);
		}
	}
}

gcpNewTemplateToolDlg::gcpNewTemplateToolDlg(gcpApplication* App): gcpDialog(App, DATADIR"/gchempaint/ui/new-template.glade", "new_template")
{
	pDoc = new gcpDocument(m_App, true);
	pDoc->SetEditable (false);
	GtkWidget* w;
	GtkScrolledWindow* scroll = (GtkScrolledWindow*)glade_xml_get_widget(xml, "scrolledcanvas");
	gtk_scrolled_window_add_with_viewport(scroll, w = pDoc->GetWidget());
	pData = (gcpWidgetData*) g_object_get_data (G_OBJECT (w), "data");
	m_App->SetDialog("New Template", this);
	m_node = NULL;
	/* build the categories list */
	GtkListStore *model = gtk_list_store_new (NUM_COLUMNS, G_TYPE_STRING);
	GtkTreeIter iter;
	set<string>::iterator it = categories.begin(), end = categories.end();
	for (; it != end; it++) {
		gtk_list_store_append (model, &iter);
		gtk_list_store_set (model, &iter,
			  NAME_COLUMN, (*it).c_str(),
			  -1);
	}
	w = gtk_combo_box_entry_new_with_model (GTK_TREE_MODEL (model), NAME_COLUMN);
	g_object_unref (model);
	gtk_table_attach_defaults (GTK_TABLE (glade_xml_get_widget(xml, "table1")), w, 1, 2, 1, 2);
	gtk_widget_show (w);
	category_entry = GTK_ENTRY (gtk_bin_get_child (GTK_BIN (w)));
}

gcpNewTemplateToolDlg::~gcpNewTemplateToolDlg()
{
	m_App->SetDialog("New Template", NULL);
	if (m_node){
		xmlUnlinkNode (m_node);
		xmlFreeNode (m_node);
	}
}

bool gcpNewTemplateToolDlg::Apply()
{
	const char* name = gtk_entry_get_text ((GtkEntry*) glade_xml_get_widget (xml, "name"));
	const char* category = gtk_entry_get_text (category_entry);
	if (!m_node || (*name == 0) || (*category == 0)) {
		char* msg;
		if (!m_node)
			msg = _("Please provide an object.");
		else if (*name == 0)
			msg = _("Please give a name.");
		else
			msg = _("Please choose a category.");
		GtkWidget* message = gtk_message_dialog_new(NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, 
															msg);
		g_signal_connect_swapped (G_OBJECT (message), "response", G_CALLBACK (gtk_widget_destroy), G_OBJECT (message));
		gtk_window_set_icon_name (GTK_WINDOW (message), "gchempaint");
		gtk_widget_show(message);
		return false;
	}
	gcpTemplate *temp = new gcpTemplate ();
	temp->node = m_node;
	temp->writeable = true;
	temp->name = name;
	temp->category = category;
	if (!user_templates) {
		user_templates = xmlNewDoc ((xmlChar*) "1.0");
		user_templates->children =  xmlNewDocNode(user_templates, NULL, (xmlChar*) "templates", NULL);
		char* filename = g_strconcat (getenv ("HOME"), "/.gchempaint/templates/templates.xml", NULL);
		user_templates->URL = xmlStrdup ((xmlChar*) filename);
		g_free (filename);
	}
	xmlNodePtr node = xmlNewDocNode (user_templates, NULL, (xmlChar*) "template", NULL);
	xmlNodePtr child = xmlNewDocNode (user_templates, NULL, (xmlChar*)"category", (xmlChar*) category);
	xmlAddChild (node, child);
	child = xmlNewDocNode (user_templates, NULL, (xmlChar*)"name", (xmlChar*) name);
	xmlAddChild (node, child);
	xmlUnlinkNode (m_node);
	xmlAddChild (node, m_node);
	set<string>::iterator it = categories.find (category);
	if (it == categories.end()) categories.insert (category);
	string key = temp->name;
	if (TempbyName[key]) {
		int i = 0;
		char* str = g_strdup_printf ("%d", i);
		while (TempbyName[key + str]) {
			g_free (str);
			str = g_strdup_printf ("%d", ++i);
		}
		key += str;
		g_free (str);
	}
	TempbyName[key] = temp;
	key = string((char*) category) + "/" + (char*) name;
	if (Templates[key]) {
		int i = 0;
		char* str = g_strdup_printf ("%d", i);
		while (Templates[key + str]) {
			g_free (str);
			str = g_strdup_printf ("%d", ++i);
		}
		key += str;
		g_free (str);
	}
	Templates[key] = temp;
	m_node = NULL;
	xmlAddChild (user_templates->children, node);
	xmlIndentTreeOutput = true;
	xmlKeepBlanksDefault (0);
	xmlSaveFormatFile((char*) user_templates->URL, user_templates, true);
	gcpTemplateTree *tree = (gcpTemplateTree*) m_App->GetTool ("TemplateTree");
	if (tree)
		tree->AddTemplate (key);
	gcpTemplateToolDlg *dlg = (gcpTemplateToolDlg*) m_App->GetDialog ("Templates");
	if (dlg) {
		dlg->SetTemplate (temp);
	}
	return true;
}

void gcpNewTemplateToolDlg::SetTemplate (xmlNodePtr node)
{
	map<string, Object*>::iterator it;
	Object* obj = pDoc->GetFirstChild(it);
	if (obj) pDoc->Remove (obj);
	pDoc->PopOperation ();
	if (m_node) {
		xmlUnlinkNode (m_node);
		xmlFreeNode (m_node);
	}
	pDoc->AddData (node);
	ArtDRect rect;
	while (gtk_events_pending ()) gtk_main_iteration ();
	pDoc->AbortOperation ();
	pData->GetSelectionBounds (rect);
	pData->MoveSelection (- rect.x0, - rect.y0);
	pDoc->PopOperation ();
	pData->UnselectAll ();
	xmlUnlinkNode (node);
	xmlFreeNode (node);
	obj = pDoc->GetFirstChild(it);
	m_node = obj->Save (::xml);
}
