// -*- C++ -*-

/* 
 * GChemPaint text plugin
 * fragmenttool.cc 
 *
 * Copyright (C) 2003-2004
 *
 * Developed by Jean Bréfort <jean.brefort@normalesup.org>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gchempaint-config.h"
#include "fragmenttool.h"
#include "lib/fragment.h"
#include "lib/document.h"
#include "lib/application.h"
#include <gdk/gdkkeysyms.h>

extern GtkTextTagTable *TextTagTable;
extern xmlDocPtr pXmlDoc;

using namespace gcu;

gcpFragmentTool::gcpFragmentTool(gcpApplication *App): gcpTextTool(App, "Fragment")
{
}

gcpFragmentTool::~gcpFragmentTool()
{
	if (ClipboardData)
		xmlFree (ClipboardData);
}

static void on_get_data(GtkClipboard *clipboard, GtkSelectionData *selection_data,  guint info, gcpFragmentTool* tool)
{
	tool->OnGetData(clipboard, selection_data, info);
}

bool gcpFragmentTool::OnClicked()
{
	void *func;
	if (m_Active && ((m_pObject == NULL) || (m_pObject->GetType() != FragmentType) || (m_Active != m_pData->Items[m_pObject]->item_list->data)))
	{
		if (!Unselect()) return false;
	}
	if (!m_pObject)
	{
		gcpDocument* pDoc = m_pView->GetDoc();
		gcpFragment *fragment = new gcpFragment(m_x0 / m_pData->ZoomFactor, m_y0 / m_pData->ZoomFactor);
		pDoc->AddFragment(fragment);
		pDoc->AbortOperation();
		pDoc->EmptyTranslationTable();
		m_pObject = fragment;
	}
	if (m_pObject)
	{
		switch(m_pObject->GetType())
		{
			case AtomType:
			{
				gcpDocument* pDoc = m_pView->GetDoc();
				gcpAtom* pAtom = (gcpAtom*)m_pObject;
				if (pAtom->GetTotalBondsNumber() > 1) return false;
				double x, y;
				pAtom->GetCoords(&x, &y);
				gcpMolecule *pMol = (gcpMolecule*)pAtom->GetMolecule();
				map<Atom*, Bond*>::iterator i;
				gcpBond *pBond = (gcpBond*)pAtom->GetFirstBond(i);
				gcpFragment *pFragment = new gcpFragment(x, y);
				gcpAtom* pFragAtom = (gcpAtom*)pFragment->m_Atom;
				map<string, Object*>::iterator ie;
				Object* electron = pAtom->GetFirstChild (ie);
				while (electron){
					m_pView->Remove (electron);
					delete electron;
					electron = pAtom->GetNextChild (ie);
				}
				pMol->Remove(pAtom);
				pAtom->SetParent(NULL);
				pMol->AddFragment(pFragment);
				pDoc->AddFragment(pFragment);
				pDoc->AbortOperation();
				gcpOperation* pOp = pDoc->GetNewOperation(GCP_MODIFY_OPERATION);
				pOp->AddObject(pAtom, 0);
				if (pBond) pOp->AddObject(pBond, 0);
				m_pView->Remove(pAtom);
				pFragAtom->SetZ(pAtom->GetZ());
				pFragAtom->SetId((gchar*)pAtom->GetId());
				int n = pAtom->GetAttachedHydrogens();
				if (n)
				{
					char* buf;
					if (n > 1) buf = g_strdup_printf("H%d", n);
					else buf = g_strdup("H");
					GtkTextIter iter;
					if (pAtom->GetBestSide())
						gtk_text_buffer_get_end_iter(pFragment->m_buf, &iter);
					else
						gtk_text_buffer_get_start_iter(pFragment->m_buf, &iter);
					gtk_text_buffer_insert(pFragment->m_buf, &iter, buf, -1);
					pFragment->AnalContent();
					gtk_text_buffer_set_modified(pFragment->m_buf, false);
					g_free(buf);
				}
				delete pAtom;
				if (pBond)
				{
					pBond->ReplaceAtom(pAtom, pFragAtom);
					pFragAtom->AddBond(pBond);
					pOp->AddObject(pBond, 1);
				}
				pOp->AddObject(pFragment, 1);
				pDoc->FinishOperation();
				pDoc->EmptyTranslationTable();
				m_pObject = pFragment;
				break;
			}
			case BondType:
				return false;
			case FragmentType:
				break;
			default:
				return false;
		}
		m_pObject->SetSelected(m_pWidget, SelStateUpdating);
		m_Active = GNOME_CANVAS_RICH_TEXT_EXT(g_object_get_data(G_OBJECT(m_pData->Items[m_pObject]), "fragment"));
		m_pView->SetGnomeCanvasRichTextActive(m_Active);
		g_object_set(G_OBJECT(m_Active), "editable", true, "cursor_visible", true, NULL);
		m_CurNode = ((gcpFragment*)m_pObject)->SaveSelected();
		m_InitNode = ((gcpFragment*)m_pObject)->SaveSelected();
		m_pApp->ActivateMenu ("Image", false);
	}
	return true;
}

bool gcpFragmentTool::Deactivate()
{
	if (m_Active && !Unselect()) return false;
	return true;
}

void gcpFragmentTool::Activate()
{
}

bool gcpFragmentTool::OnEvent(GdkEvent* event)
{
	if (m_Active)
	{
		if ((event->type == GDK_KEY_PRESS) || (event->type == GDK_KEY_RELEASE))
		{
			if (event->key.state & GDK_CONTROL_MASK)
			{
				switch(event->key.keyval)
				{
					case GDK_Right:
					case GDK_Left:
					case GDK_Up:
					case GDK_Down:
					case GDK_End:
					case GDK_Home:
					case GDK_Delete:
					case GDK_KP_Delete:
					case GDK_BackSpace:
					break;
					case GDK_z:
						m_pView->GetDoc()->OnUndo();
						return true;
					case GDK_Z:
						m_pView->GetDoc()->OnRedo();
						return true;
					case GDK_c:
						CopySelection(gtk_clipboard_get(GDK_SELECTION_CLIPBOARD));
						return true;
					case GDK_v:
						PasteSelection(gtk_clipboard_get(GDK_SELECTION_CLIPBOARD));
						return true;
					case GDK_x:
						CutSelection(gtk_clipboard_get(GDK_SELECTION_CLIPBOARD));
						return true;
					default:
						return true;
				}
			}
			if (event->key.keyval == GDK_KP_Enter || event->key.keyval == GDK_Return ||
								event->key.keyval == GDK_space) return true;
			if (!g_utf8_validate(((GdkEventKey*)event)->string, -1, NULL))
			{
				gsize r, w;
				gchar* newstr = g_locale_to_utf8(((GdkEventKey*)event)->string, ((GdkEventKey*)event)->length, &r, &w, NULL);
				g_free(((GdkEventKey*)event)->string);
				((GdkEventKey*)event)->string = newstr;
				((GdkEventKey*)event)->length = w;
			}
			gnome_canvas_item_grab_focus((GnomeCanvasItem*)m_Active);
			GnomeCanvasItemClass* klass = GNOME_CANVAS_ITEM_CLASS(((GTypeInstance*)m_Active)->g_class);
			GtkTextBuffer* buf = gnome_canvas_rich_text_ext_get_buffer(m_Active);
			gtk_text_buffer_begin_user_action(buf);
			klass->event((GnomeCanvasItem*)m_Active, event);
			gtk_text_buffer_end_user_action(buf);
			return true;
		}
	}
	return false;
}

bool gcpFragmentTool::CopySelection(GtkClipboard *clipboard)
{
	if (!m_Active) return false;
	GtkTextBuffer* buf = gnome_canvas_rich_text_ext_get_buffer(m_Active);
	GtkTextIter start, end;
	gtk_text_buffer_get_selection_bounds(buf, &start, &end);
	if (gtk_text_iter_equal(&start, &end)) return false; //Nothing to copy
	m_pData->Copy(clipboard); //To clean the xmlDoc
	xmlDocPtr pDoc = m_pData->GetXmlDoc(clipboard);
	if (!pDoc) return false;
	pDoc->children = xmlNewDocNode(pDoc, NULL, (xmlChar*)"chemistry", NULL);
	gcpFragment *fragment = (gcpFragment*)g_object_get_data(G_OBJECT(m_Active), "object");
	xmlNodePtr node = fragment->SaveSelection(pDoc);
	if (node) xmlAddChild(pDoc->children, node);
	else return false;
	gtk_clipboard_set_with_data(clipboard, targets, 2,
				(GtkClipboardGetFunc)on_get_data,
				(GtkClipboardClearFunc)on_clear_data, NULL);
	gtk_clipboard_request_contents(clipboard, gdk_atom_intern ("TARGETS", FALSE),  (GtkClipboardReceivedFunc)on_receive_targets, NULL);
	return true;
}

bool gcpFragmentTool::CutSelection(GtkClipboard *clipboard)
{
	if (!CopySelection(clipboard)) return false;
	return DeleteSelection();
}

/**
* This method does nothing and always return false because pasting something inside
* a fragment is somewhat unsafe and will not be implemented in a foreseable future.
*/
bool gcpFragmentTool::OnReceive(GtkClipboard *clipboard, GtkSelectionData *data, int type)
{
	return false;
}

bool gcpFragmentTool::Unselect()
{
	if (!m_Active) return true;
	gcpFragment *fragment = (gcpFragment*)g_object_get_data(G_OBJECT(m_Active), "object");
	if (fragment->Validate())
		return gcpTextTool::Unselect();
	return false;
}

void gcpFragmentTool::OnGetData (GtkClipboard *clipboard, GtkSelectionData *selection_data,  guint info)
{
	xmlDocPtr pDoc = gcpWidgetData::GetXmlDoc (clipboard);
	if (ClipboardData) {
		xmlFree (ClipboardData);
		ClipboardData = NULL;
	} 
	ClipboardDataType = info;
	gint size;
	if (info)
	{
		ClipboardData = xmlNodeGetContent (pDoc->children->children);
		size = strlen((char*) ClipboardData);
		gtk_selection_data_set_text (selection_data, (const gchar*) ClipboardData, size);
	}
	else
	{
		xmlDocDumpFormatMemory (pDoc, &ClipboardData, &size, info);
		gtk_selection_data_set(selection_data, gdk_atom_intern (GCHEMPAINT_ATOM_NAME, FALSE), 8,  (const guchar*) ClipboardData, size);
	}
	cleared = false;
	if (clipboard == gtk_clipboard_get(GDK_SELECTION_CLIPBOARD)) m_pApp->ActivateMenu("Paste", true);
}
