// -*- C++ -*-

/* 
 * GChemPaint library
 * mesomery.cc 
 *
 * Copyright (C) 2002-2004
 *
 * Developed by Jean Bréfort <jean.brefort@normalesup.org>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gchempaint-config.h"
#include "document.h"
#include "widgetdata.h"
#include "view.h"
//#include "settings.h"
//#include "libgcpcanvas/gcp-canvas-line.h"
//#include "libgcpcanvas/gcp-canvas-group.h"
#include "mesomery.h"
#include "mesomery-arrow.h"
#include "mesomer.h"
#include <glib/gi18n.h>
#include <math.h>
#include <set>

gcpMesomery::gcpMesomery (): Object (MesomeryType)
{
	SetId ("msy1");
}

gcpMesomery::~gcpMesomery ()
{
	if (IsLocked ())
		return;
	map<string, Object*>::iterator i;
	Object *pObj;
	gcpMesomeryArrow *arrow;
	gcpDocument *pDoc = reinterpret_cast<gcpDocument *> (GetDocument ());
	gcpOperation *pOp = pDoc->GetCurrentOperation ();
	while (pObj = GetFirstChild (i)) {
		if (pObj->GetType () == MesomeryArrowType) {
			arrow = reinterpret_cast<gcpMesomeryArrow*> (pObj);
			arrow->SetStartMesomer (NULL);
			arrow->SetEndMesomer (NULL);
			arrow->SetParent (GetParent ());
			if (pOp)
				pOp->AddObject (arrow, 1);
			
		} else
			delete pObj;
	}
}

xmlNodePtr gcpMesomery::Save (xmlDocPtr xml)
{
#warning "Do we really need this one?");
	return Object::Save (xml);
}

bool gcpMesomery::Load (xmlNodePtr node)
{
	xmlChar* tmp;
	xmlNodePtr child;
	Object* pObject;
	list<xmlNodePtr> arrows;

	Lock ();
	tmp = xmlGetProp (node, (xmlChar*) "id");
	if (tmp) {
		SetId ((char*) tmp);
		xmlFree (tmp);
	}
	child = node->children;
	while (child) {
		if (!strcmp ((const char*) child->name, "mesomery-arrow"))
			arrows.push_front (child);
		else {
			pObject = CreateObject ((const char*) child->name, this);
			if (pObject) {
				if (!pObject->Load (child))
					delete pObject;
			} else {
				Lock (false);
				return false;
			}
		}
		child = child->next;
	}
	while (!arrows.empty ()) {
		child = arrows.back ();
		pObject = CreateObject ("mesomery-arrow", this);
		if (pObject) {
			if (!pObject->Load (child))
				delete pObject;
		} else {
			Lock (false);
			return false;
		}
		arrows.pop_back ();
	}
	Lock (false);
	return true;
}

void gcpMesomery::Add (GtkWidget* w)
{
	map<string, Object*>::iterator i;
	Object* p = GetFirstChild (i);
	while (p) {
		p->Add (w);
		p = GetNextChild (i);
	}
}

typedef struct
{
	double x, y;
	ArtDRect r;
	gcpMesomer *mes;
} ObjectData;

bool gcpMesomery::Build (list<Object*>& Children) throw (invalid_argument)
{
	gcpDocument *pDoc = reinterpret_cast<gcpDocument *> (GetDocument ());
	gcpView *pView = pDoc->GetView ();
	gcpWidgetData  *pData= reinterpret_cast<gcpWidgetData *> (g_object_get_data (G_OBJECT (pDoc->GetWidget ()), "data"));
	list<Object *>::iterator i, iend = Children.end ();
	map<Object *, ObjectData> Objects;
	list<Object *> Arrows;
	double minright, minleft, x, y, x0, y0, x1, y1, l, d, ps;
	Object *Left, *Right;
	ObjectData od;
	gcpMesomeryArrow *arrow;
	unsigned narrows = 0, nmol = 0;
	for (i = Children.begin (); i != iend; i++) {
		pData->GetObjectBounds (*i, &od.r);
		od.x = (od.r.x0 + od.r.x1) / 2.;
		od.y = (*i)->GetYAlign () * pData->ZoomFactor;
		switch ((*i)->GetType ()) {
		case MoleculeType:
			od.mes = new gcpMesomer (this, reinterpret_cast<gcpMolecule *>(*i));
			nmol++;
			break;
		case MesomeryArrowType:
			narrows++;
			Arrows.push_back (*i);
			AddChild (*i);
			break;
		default:
			throw  invalid_argument (_("Something wrong happened, please file a bug report."));
		}
		Objects[*i] = od;
	}
	// now, for each arrow, search closiest object on both sides and verify it's a molecule
	list<Object *>::iterator j, jend = Arrows.end ();
	for (j = Arrows.begin (); j != jend; j++) {
		arrow = reinterpret_cast<gcpMesomeryArrow *>(*j);
		arrow->GetCoords (&x0, &y0, &x1, &y1);
		//x0 and y0 should be the center of the arrow, not the beginning, so we must transform them
		x0 = (x0 + x1) / 2;
		y0 = (y0 + y1) / 2;
		// x1, y1 will now be the coordinates of a normalized vector:
		x1 -= x0;
		y1 -= y0;
		x0 *= pData->ZoomFactor;
		y0 *= pData->ZoomFactor;
		l = sqrt (x1 * x1 + y1 * y1);
		x1 /= l;
		y1 /= l;
		l *= pData->ZoomFactor; // half length of the arrow on the screen
		// No molecule should be nearer than that
		minright = minleft = DBL_MAX;
		Left = Right = NULL;
		for (i = Children.begin (); i != iend; i++) {
			if (*i == *j)
				continue;
			od = Objects[*i];
			x = od.x - x0;
			y = od.y - y0;
			d = sqrt (x * x + y * y);
			ps = (x * x1 + y * y1) / d;
			if (ps >= -.71 && ps <= .71)
				continue;
			if (d < l) {
				if ((*i)->GetType () == MesomeryArrowType)
					continue;
				else {
					Left = *i;
					Right = *j;
					pData->UnselectAll ();
					pData->SetSelected (Left);
					pData->SetSelected (Right);
				}
			}
			if (ps < 0) {
				if (d < minleft) {
					Left = od.mes;
					minleft = d;
				}
			} else {
				if (d < minright) {
					Right = od.mes;
					minright = d;
				}
			}
		}
		if (!Left || !Right) { // Do not accept arrows with only one mesomer (?)
			Left = *j;
			pData->UnselectAll ();
			pData->SetSelected (Left);
			throw invalid_argument (_("Isolated arrows are not allowed!"));
		}
		reinterpret_cast<gcpMesomeryArrow *> (*j)->SetStartMesomer (reinterpret_cast<gcpMesomer *> (Left));
		reinterpret_cast<gcpMesomeryArrow *> (*j)->SetEndMesomer (reinterpret_cast<gcpMesomer *> (Right));
		reinterpret_cast<gcpMesomer *> (Left)->AddArrow (reinterpret_cast<gcpMesomeryArrow *> (*j), reinterpret_cast<gcpMesomer *> (Right));
		reinterpret_cast<gcpMesomer *> (Right)->AddArrow (reinterpret_cast<gcpMesomeryArrow *> (*j), reinterpret_cast<gcpMesomer *> (Left));
	}
	// now, check if each mesomer has at least one arrow, may be we should add missing arrows?
	for (i = Children.begin (); i != iend; i++) {
		if ((*i)->GetType () == MesomeryArrowType)
			continue;
		od = Objects[*i];
		if (!od.mes->Validate ()) {
			Left = *i;
			pData->UnselectAll ();
			pData->SetSelected (Left);
			throw invalid_argument (_("Isolated molecule!\n Please add missing arrows."));
		}
	}
	// Check if all mesomers are related (only connectivity is checked for now)
	if (!Validate (false))
		throw invalid_argument (_("Please add missing arrows."));
	// Align the children
	Align ();
	throw  invalid_argument (_("Sorry, not yet implemented!"));
	return false;
}

static void BuildConnectivity ( set<Object *> &Objects, gcpMesomer* Mesomer)
{
	map<gcpMesomer *, gcpMesomeryArrow *> *Arrows = Mesomer->GetArrows ();
	map<gcpMesomer *, gcpMesomeryArrow *>::iterator i, end = Arrows->end ();
	for (i = Arrows->begin (); i != end; i++) {
		Objects.insert ((*i).second);
		if (Objects.find ((*i).first) == Objects.end ()) {
			Objects.insert ((*i).first);
			BuildConnectivity (Objects, (*i).first);
		}
	}
}

bool gcpMesomery::Validate (bool split)
{
	map<string, Object*>::iterator i;
	Object *pObj = GetFirstChild (i);
	while (pObj && pObj->GetType () != MesomerType)
		pObj = GetNextChild (i);
	if (pObj == NULL)
		return false;
	set<Object *> Objects;
	Objects.insert (pObj);
	BuildConnectivity (Objects, reinterpret_cast<gcpMesomer *> (pObj));
	if (Objects.size () < GetChildrenNumber ()) {
		if (!split)
			return false;
#warning "TODO: add some code there"
	}
	return true;
}

typedef struct
{
	double x, y, dx, dy;
	ArtDRect r;
	bool Locked;
} MesomerData;

void gcpMesomery::Align ()
{
	map<string, Object*>::iterator i;
	Object *pObj = GetFirstChild (i);
	while (pObj && pObj->GetType () != MesomerType)
		pObj = GetNextChild (i);
	if (pObj == NULL)
		throw  invalid_argument (_("Something wrong happened, please file a bug report."));
//TODO: real alignment
	set<Object *> Objects;
	Objects.insert (pObj);
}
