// -*- C++ -*-

/* 
 * GChemPaint library
 * reaction.cc 
 *
 * Copyright (C) 2002-2004
 *
 * Developed by Jean Bréfort <jean.brefort@normalesup.org>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gchempaint-config.h"
#include "reaction.h"
#include "widgetdata.h"
#include "view.h"
#include "document.h"
#include "settings.h"
#include "libgcpcanvas/gcp-canvas-line.h"
#include "libgcpcanvas/gcp-canvas-group.h"
#include "libgcpcanvas/gcp-canvas-rect-ellipse.h"
#include "libgcpcanvas/gcp-canvas-text.h"
#include <gcu/xml-utils.h>
#include <math.h>

using namespace std;

TypeId ReactionStepType = NoType;

gcpReactionStep::gcpReactionStep(): Object (ReactionStepType)
{
	SetId ("rs1");
}

gcpReactionStep::gcpReactionStep (gcpReaction *reaction, map<double, Object*>& Children, map<Object*, ArtDRect> Objects): Object (ReactionStepType)
{
	SetId ("rs1");
	reaction->AddChild (this);
	GetDocument ()->EmptyTranslationTable();
	gcpDocument *pDoc = (gcpDocument*) GetDocument ();
	gcpWidgetData  *pData= (gcpWidgetData*) g_object_get_data (G_OBJECT (pDoc->GetWidget ()), "data");
	map<double, Object*>::iterator im, endm;
	double x, y, x0, y0, x1, y1;
	ArtDRect *rect;
	Object *cur;
	im = Children.begin ();
	new gcpReactant (this, (*im).second);
	endm = Children.end ();
	rect = &Objects[(*im).second];
	x = rect->x1;
	y = (*im).second->GetYAlign ();
	for (im++; im != endm; im++) {
		x += pData->SignPadding;
		//Add a sign
		gcpReactionOperator* pOp = new gcpReactionOperator(NULL);
		AddChild (pOp);
		pOp->SetCoords(x / pData->ZoomFactor, y);
		pDoc->AddObject(pOp);
		gnome_canvas_update_now (GNOME_CANVAS (pData->Canvas));
		gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (pData->Items[pOp]), &x0, &y0, &x1, &y1);
		pOp->Move ((x - x0) / pData->ZoomFactor, 0);
		x += pData->SignPadding + x1 - x0;
		cur = (*im).second;
		new gcpReactant (this, cur);
		rect = &Objects[cur];
		y0 = cur->GetYAlign ();
		cur->Move ((x - rect->x0) / pData->ZoomFactor, y - y0);
		x+= rect->x1 - rect->x0;
	}
	Update (pData->Canvas);
	gnome_canvas_update_now (GNOME_CANVAS (pData->Canvas));
}

gcpReactionStep::~gcpReactionStep()
{
}
	
xmlNodePtr gcpReactionStep::Save(xmlDocPtr xml)
{
	xmlNodePtr node;
	node = xmlNewDocNode (xml, NULL, (xmlChar*) "reaction-step", NULL);
	if (!node) return NULL;
	SaveId (node);
	map<string, Object*>::iterator i;
	Object *obj = GetFirstChild (i);
	xmlNodePtr child;
	while (obj)
	{
		if ((*i).second->GetType () != ReactionOperatorType) {
			if (child = (*i).second->Save (xml)) xmlAddChild (node, child);
			else return NULL;
		}
		obj = GetNextChild (i);
	}
	return node;
}
	
bool gcpReactionStep::Load(xmlNodePtr node)
{
	if (!Object::Load (node))
		return false;
	map<Object*, ArtDRect> Objects;
	map<double, Object*> Children;
	ArtDRect rect;
	map<string, Object*>::iterator i;
	Object *pObj = GetFirstChild (i);
	gcpDocument *pDoc = (gcpDocument*) GetDocument ();
	gcpWidgetData  *pData= (gcpWidgetData*) g_object_get_data (G_OBJECT (pDoc->GetWidget ()), "data");
	map<double, Object*>::iterator im, endm;
	double x, y, x0, y0, x1, y1;
	gnome_canvas_update_now (GNOME_CANVAS (pData->Canvas));
	while (pObj) {
		pData->GetObjectBounds (pObj, &rect);
		x = (rect.x0 + rect.x1) / 2;
		while (Children[x] != NULL)
			x += 1e-5;
		Children[x] = pObj;
		Objects[pObj] = rect;
		pObj = GetNextChild (i);
	}
	im = Children.begin ();
	endm = Children.end ();
	rect = Objects[(*im).second];
	x = rect.x1;
	y = (*im).second->GetYAlign ();
	for (im++; im != endm; im++) {
		x += pData->SignPadding;
		//Add a sign
		gcpReactionOperator* pOp = new gcpReactionOperator(NULL);
		AddChild (pOp);
		pOp->SetCoords(x / pData->ZoomFactor, y);
		pDoc->AddObject(pOp);
		gnome_canvas_update_now (GNOME_CANVAS (pData->Canvas));
		gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (pData->Items[pOp]), &x0, &y0, &x1, &y1);
		pOp->Move ((x - x0) / pData->ZoomFactor, 0);
		x += pData->SignPadding + x1 - x0;
		pObj = (*im).second;
		rect = Objects[pObj];
		x+= rect.x1 - rect.x0;
	}
	Update (pData->Canvas);
	return true;
}

double gcpReactionStep::GetYAlign ()
{
	map<string, Object*>::iterator i;
	Object *child = GetFirstChild (i);
	return ((*i).second)? (*i).second->GetYAlign (): 0.;
}

bool gcpReactionStep::OnSignal (SignalId Signal, Object *Child)
{
	if (Signal == OnChangedSignal) {
		map<Object*, ArtDRect> Objects;
		map<double, Object*> Children;
		list<Object*> Operators;
		ArtDRect rect;
		map<string, Object*>::iterator i;
		Object *pObj = GetFirstChild (i);
		gcpDocument *pDoc = (gcpDocument*) GetDocument ();
		gcpView *pView = pDoc->GetView ();
		gcpWidgetData  *pData= (gcpWidgetData*) g_object_get_data (G_OBJECT (pDoc->GetWidget ()), "data");
		map<double, Object*>::iterator im, endm;
		double x, y, x0, y0, x1, y1;
		gnome_canvas_update_now (GNOME_CANVAS (pData->Canvas));
		while (pObj) {
			if (pObj->GetType () == ReactionOperatorType)
				Operators.push_front (pObj);
			else {
				pData->GetObjectBounds (pObj, &rect);
				x = (rect.x0 + rect.x1) / 2;
				while (Children[x] != NULL)
					x += 1e-5;
				Children[x] = pObj;
				Objects[pObj] = rect;
			}
			pObj = GetNextChild (i);
		}
		while (!Operators.empty ()) {
			pObj = Operators.front ();
			pView->Remove (pObj);
			delete pObj;
			Operators.pop_front ();
		}
		im = Children.begin ();
		endm = Children.end ();
		rect = Objects[(*im).second];
		x = rect.x1;
		y = (*im).second->GetYAlign ();
		for (im++; im != endm; im++) {
			x += pData->SignPadding;
			//Add a sign
			gcpReactionOperator* pOp = new gcpReactionOperator(NULL);
			AddChild (pOp);
			pOp->SetCoords(x / pData->ZoomFactor, y);
			pDoc->AddObject(pOp);
			gnome_canvas_update_now (GNOME_CANVAS (pData->Canvas));
			gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (pData->Items[pOp]), &x0, &y0, &x1, &y1);
			pOp->Move ((x - x0) / pData->ZoomFactor, 0);
			x += pData->SignPadding + x1 - x0;
			pObj = (*im).second;
			rect = Objects[pObj];
			x+= rect.x1 - rect.x0;
		}
		Update (pData->Canvas);
		return true;
	} else
		return true;
}

gcpReactant::gcpReactant (): Object (ReactantType)
{
	SetId ("r1");
	m_Stoech = 1;
}

gcpReactant::gcpReactant (gcpReactionStep* step, Object *object)	throw (invalid_argument): Object (ReactantType)
{
	SetId ("r1");
	step->AddChild (this);
	GetDocument ()->EmptyTranslationTable();
	static const set<TypeId>& allowed_types = Object::GetRules ("reactant", RuleMayContain);
	if (allowed_types.find (object->GetType ()) == allowed_types.end ())
		throw invalid_argument ("invalid reactant");
	AddChild (object);
	m_Stoech = 1;
}

gcpReactant::~gcpReactant ()
{
}

static void do_add_stoechiometry (gcpReactant *reactant)
{
}

bool gcpReactant::BuildContextualMenu (GtkUIManager *UIManager)
{
	bool result = false;
/*	if (m_Stoech == 1) {
		GtkActionGroup *group = gtk_action_group_new ("reactant");
		GtkAction *action = gtk_action_new ("stoechiometry", _("Add a stoechiometry coefficient"), NULL, NULL);
		gtk_action_group_add_action (group, action);
		gtk_ui_manager_insert_action_group (UIManager, group, 0);
		char buf[] = "<ui><popup><menuitem action='stoechiometry'/></popup></ui>";
		gtk_ui_manager_add_ui_from_string (UIManager, buf, -1, NULL);
		GtkWidget *w = gtk_ui_manager_get_widget (UIManager, "/popup/stoechiometry");
		g_signal_connect_swapped (w, "activate", G_CALLBACK (do_add_stoechiometry), this);
		result = true;
	}*/
	return result | GetParent ()->BuildContextualMenu (UIManager);
}

xmlNodePtr gcpReactant::Save (xmlDocPtr xml)
{
	xmlNodePtr node = Object::Save (xml);
	if (m_Stoech != 1) {
	}
	return node;
}

bool gcpReactant::Load (xmlNodePtr node)
{
	return Object::Load (node);
}

double gcpReactant::GetYAlign ()
{
	map<string, Object*>::iterator i;
	Object *child = GetFirstChild (i);
	return ((*i).second)? (*i).second->GetYAlign (): 0.;
}

gcpReactionArrow::gcpReactionArrow(gcpReaction* react, unsigned Type): gcpArrow(ReactionArrowType)
{
	SetId ("ra1");
	m_Type = Type;
	m_Start = m_End = NULL;
	if (react) react->AddChild(this);
}


gcpReactionArrow::~gcpReactionArrow()
{
}

xmlNodePtr gcpReactionArrow::Save(xmlDocPtr xml)
{
	xmlNodePtr parent, node;
	node = xmlNewDocNode(xml, NULL, (xmlChar*)"reaction-arrow", NULL);
	if (!node) return NULL;
	if (!gcpArrow::Save(xml, node)) {xmlFreeNode(node); return NULL;}
	xmlNewProp(node, (xmlChar*)"type", (xmlChar*)((m_Type == gcpSimpleArrow)? "single": "double"));
	if (m_Start)
		xmlNewProp (node, (xmlChar*)"start",  (xmlChar*) m_Start->GetId ());
	if (m_End)
		xmlNewProp (node, (xmlChar*)"end",  (xmlChar*) m_End->GetId ());
	gcpReaction* r = (gcpReaction*)GetReaction();
	if (!r)
	{
		//save the arrow as an object (this is NOT safe)
		parent = xmlNewDocNode(xml, NULL, (xmlChar*)"object", NULL);
		if (node && parent) xmlAddChild(parent, node);
			else {xmlFreeNode(node); return NULL;}
	}
	else parent = node;
	return parent;
}

bool gcpReactionArrow::Load (xmlNodePtr node)
{
	char* tmp;
	if (gcpArrow::Load (node))
	{
		tmp = (char*) xmlGetProp (node, (xmlChar*) "type");
		if (tmp) {
			if (!strcmp (tmp, "double"))
				m_Type = gcpReversibleArrow;
			xmlFree (tmp);
		}
		return true;
	}
	return false;
}

void gcpReactionArrow::Add(GtkWidget* w)
{
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	GnomeCanvasPoints *points = gnome_canvas_points_new(2);
	GnomeCanvasGroup* group = GNOME_CANVAS_GROUP(gnome_canvas_item_new(pData->Group, gnome_canvas_group_ext_get_type(), NULL));
	GnomeCanvasItem* item;
	switch(m_Type)
	{
		case gcpSimpleArrow:
			points->coords[0] = m_x * pData->ZoomFactor;
			points->coords[1] = m_y * pData->ZoomFactor;
			points->coords[2] = (m_x + m_width) * pData->ZoomFactor;
			points->coords[3] = (m_y + m_height) * pData->ZoomFactor;
			item = gnome_canvas_item_new(
										group,
										gnome_canvas_line_ext_get_type(),
										"points", points,
										"fill_color", (pData->IsSelected(this))? SelectColor: Color,
										"width_units", pData->BondWidth,
										"last_arrowhead", true,
										"arrow_shape_a", pData->ArrowHeadA,
										"arrow_shape_b", pData->ArrowHeadB,
										"arrow_shape_c", pData->ArrowHeadC,
										"last_arrowhead_style", (unsigned char)ARROW_HEAD_BOTH,
										NULL);
			g_object_set_data(G_OBJECT(item), "object", this);
			g_object_set_data(G_OBJECT(group), "arrow", item);
			g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), w);
			break;
		case gcpReversibleArrow:
			double dAngle = atan(- m_height / m_width);
			if (m_width < 0) dAngle += M_PI;
			points->coords[0] = m_x * pData->ZoomFactor - pData->ArrowDist  / 2 * sin(dAngle);
			points->coords[1] = m_y * pData->ZoomFactor - pData->ArrowDist  / 2 * cos(dAngle);
			points->coords[2] = (m_x + m_width) * pData->ZoomFactor - pData->ArrowDist  / 2 * sin(dAngle);
			points->coords[3] = (m_y + m_height) * pData->ZoomFactor - pData->ArrowDist  / 2 * cos(dAngle);
			item = gnome_canvas_item_new(
								group,
								gnome_canvas_line_ext_get_type(),
								"points", points,
								"fill_color", (pData->IsSelected(this))? SelectColor: Color,
								"width_units", pData->BondWidth,
								"last_arrowhead", true,
								"arrow_shape_a", pData->ArrowHeadA,
								"arrow_shape_b", pData->ArrowHeadB,
								"arrow_shape_c", pData->ArrowHeadC,
								"last_arrowhead_style", (unsigned char)ARROW_HEAD_LEFT,
								NULL);
			g_object_set_data(G_OBJECT(item), "object", this);
			g_object_set_data(G_OBJECT(group), "direct", item);
			g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), w);
			points->coords[2] = m_x * pData->ZoomFactor + pData->ArrowDist / 2 * sin(dAngle);
			points->coords[3] = m_y * pData->ZoomFactor + pData->ArrowDist  / 2 * cos(dAngle);
			points->coords[0] = (m_x + m_width) * pData->ZoomFactor + pData->ArrowDist  / 2 * sin(dAngle);
			points->coords[1] = (m_y + m_height) * pData->ZoomFactor + pData->ArrowDist  / 2 * cos(dAngle);
			item = gnome_canvas_item_new(
								group,
								gnome_canvas_line_ext_get_type(),
								"points", points,
								"fill_color", (pData->IsSelected(this))? SelectColor: Color,
								"width_units", pData->BondWidth,
								"last_arrowhead", true,
								"arrow_shape_a", pData->ArrowHeadA,
								"arrow_shape_b", pData->ArrowHeadB,
								"arrow_shape_c", pData->ArrowHeadC,
								"last_arrowhead_style", (unsigned char)ARROW_HEAD_LEFT,
								NULL);
			g_object_set_data(G_OBJECT(item), "object", this);
			g_object_set_data(G_OBJECT(group), "reverse", item);
			g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), w);
			break;
	}
	pData->Items[this] = group;
	gnome_canvas_points_free(points);
}

void gcpReactionArrow::Update(GtkWidget* w)
{
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	GnomeCanvasGroup* group = pData->Items[this];
	GnomeCanvasPoints *points = gnome_canvas_points_new(2);
	switch(m_Type)
	{
		case gcpSimpleArrow:
			points->coords[0] = m_x * pData->ZoomFactor;
			points->coords[1] = m_y * pData->ZoomFactor;
			points->coords[2] = (m_x + m_width) * pData->ZoomFactor;
			points->coords[3] = (m_y + m_height) * pData->ZoomFactor;
			g_object_set(G_OBJECT(g_object_get_data(G_OBJECT(group), "arrow")),
								"points", points,
								NULL);
			break;
		case gcpReversibleArrow:
			double dAngle = atan(- m_height / m_width);
			if (m_width < 0) dAngle += M_PI;
			points->coords[0] = m_x * pData->ZoomFactor - pData->ArrowDist / 2 * sin(dAngle);
			points->coords[1] = m_y * pData->ZoomFactor - pData->ArrowDist / 2 * cos(dAngle);
			points->coords[2] = (m_x + m_width) * pData->ZoomFactor - pData->ArrowDist / 2 * sin(dAngle);
			points->coords[3] = (m_y + m_height) * pData->ZoomFactor - pData->ArrowDist / 2 * cos(dAngle);
			g_object_set(G_OBJECT(g_object_get_data(G_OBJECT(group), "direct")),
								"points", points,
								NULL);
			points->coords[2] = m_x * pData->ZoomFactor + pData->ArrowDist / 2 * sin(dAngle);
			points->coords[3] = m_y * pData->ZoomFactor + pData->ArrowDist / 2 * cos(dAngle);
			points->coords[0] = (m_x + m_width) * pData->ZoomFactor + pData->ArrowDist / 2 * sin(dAngle);
			points->coords[1] = (m_y + m_height) * pData->ZoomFactor + pData->ArrowDist / 2 * cos(dAngle);
			g_object_set(G_OBJECT(g_object_get_data(G_OBJECT(group), "reverse")),
								"points", points,
								NULL);
			break;
	}
	gnome_canvas_points_free(points);
}

gcpReactionOperator::gcpReactionOperator(gcpReaction* react, unsigned Step): Object(ReactionOperatorType)
{
	m_Step = Step;
	if (react) react->AddChild(this);
}

gcpReactionOperator::~gcpReactionOperator()
{
}

xmlNodePtr gcpReactionOperator::Save(xmlDocPtr xml)
{
/*	xmlNodePtr parent, node;
	node = xmlNewDocNode(xml, NULL, (xmlChar*)"reaction-operator", NULL);
	if (!node) return NULL;
	SaveId(node);
	if (!WritePosition(xml, node, NULL, m_x, m_y)) {xmlFreeNode(node); return NULL;}
	gcpReaction* r = (gcpReaction*)GetReaction();
	if (!r)
	{
		//save the arrow as an object
		parent = xmlNewDocNode(xml, NULL, (xmlChar*)"object", NULL);
		if (node && parent) xmlAddChild(parent, node);
			else {xmlFreeNode(node); return NULL;}
	}
	else parent = node;
	return parent;*/
	return NULL;
}

bool gcpReactionOperator::Load(xmlNodePtr node)
{
/*	char* tmp = (char*)xmlGetProp(node, (xmlChar*)"id");
	if (tmp) {SetId(tmp); xmlFree(tmp);}
	if (!ReadPosition(node, NULL, &m_x, &m_y)) return false;*/
	return false;
}

void gcpReactionOperator::Add(GtkWidget* w)
{
	if (!w) return;
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	double x, y;
	GetCoords(&x, &y);
	x *= pData->ZoomFactor;
	y *= pData->ZoomFactor;
	double dFontHeight = pData->View->GetFontHeight();
	char * szFontName = pData->View->GetFontName();
	GnomeCanvasItem* item;
	GnomeCanvasGroup* group;
	gint width, height;
	PangoContext* pc = pData->View->GetPangoContext();
	group = GNOME_CANVAS_GROUP(gnome_canvas_item_new(pData->Group, gnome_canvas_group_ext_get_type(), NULL));
	pData->Items[this] = group;
	g_signal_connect(G_OBJECT(group), "event", G_CALLBACK(on_event), w);
	g_object_set_data(G_OBJECT(group), "object", this);
	const gchar* symbol = "+";
	PangoLayout *pl1, *pl = pango_layout_new(pc);
	pango_layout_set_text(pl, symbol, strlen(symbol));
	PangoRectangle rect;
	pango_layout_get_extents(pl, &rect, NULL);
	width = rect.width / PANGO_SCALE;
	height =  rect.height / PANGO_SCALE;
	item = gnome_canvas_item_new(
						group,
						gnome_canvas_rect_ext_get_type(),
						"x1", x - (double)width / 2 - pData->Padding,
						"y1", y - dFontHeight / 2 - pData->Padding,
						"x2", x + (double)width / 2 + pData->Padding,
						"y2", y + dFontHeight / 2 + pData->Padding,
						"fill_color", "white",
						NULL);
	g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), w);
	g_object_set_data(G_OBJECT(group), "background",item);
	g_object_set_data(G_OBJECT(item), "object", this);
	item = gnome_canvas_item_new(
						group,
						gnome_canvas_text_ext_get_type(),
						"text", "+",
						"x", rint(x),
						"y", rint(y),
						"font", szFontName,
						"anchor", GTK_ANCHOR_CENTER,
						"fill_color", (pData->IsSelected(this))? SelectColor: Color,
						NULL);
	g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), w);
	g_object_set_data(G_OBJECT(group), "text",item);
	g_object_set_data(G_OBJECT(item), "object", this);
}

void gcpReactionOperator::Update(GtkWidget* w)
{
	if (!w) return;
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	double x, y;
	GetCoords(&x, &y);
	x *= pData->ZoomFactor;
	y *= pData->ZoomFactor;
	double dFontHeight = pData->View->GetFontHeight();
	char * szFontName = pData->View->GetFontName();
	GnomeCanvasItem* item;
	GnomeCanvasGroup* group = pData->Items[this];
	gint width, height;
	PangoContext* pc = pData->View->GetPangoContext();
	const gchar* symbol = "+";
	PangoLayout *pl1, *pl = pango_layout_new(pc);
	pango_layout_set_text(pl, symbol, strlen(symbol));
	PangoRectangle rect;
	pango_layout_get_extents(pl, &rect, NULL);
	width = rect.width / PANGO_SCALE;
	height =  rect.height / PANGO_SCALE;
	item = (GnomeCanvasItem*) g_object_get_data(G_OBJECT(group), "background");
	g_object_set(G_OBJECT(item),
						"x1", x - (double)width / 2 - pData->Padding,
						"y1", y - dFontHeight / 2 - pData->Padding,
						"x2", x + (double)width / 2 + pData->Padding,
						"y2", y + dFontHeight / 2 + pData->Padding,
						NULL);
	item = (GnomeCanvasItem*) g_object_get_data(G_OBJECT(group), "text");
	g_object_set(G_OBJECT(item),
						"x", rint(x),
						"y", rint(y),
						NULL);
}

void gcpReactionOperator::SetSelected(GtkWidget* w, int state)
{
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	GnomeCanvasGroup* group = pData->Items[this];
	gchar* color;
	switch (state)
	{	
		case SelStateUnselected: color = Color; break;
		case SelStateSelected: color = SelectColor; break;
		case SelStateUpdating: color = AddColor; break;
		case SelStateErasing: color = DeleteColor; break;
	}
	g_object_set(G_OBJECT(g_object_get_data(G_OBJECT(group), "text")), "fill_color", color, NULL);
}

void gcpReactionOperator::Move(double x, double y, double z)
{
	m_x += x;
	m_y += y;
}

void gcpReactionOperator::SetCoords(double x, double y)
{
	m_x = x;
	m_y = y;
}

bool gcpReactionOperator::GetCoords(double* x, double* y)
{
	*x = m_x;
	*y = m_y;
}
	
double gcpReactionOperator::GetYAlign ()
{
	return m_y;
}

gcpReaction::gcpReaction(): Object(ReactionType)
{
	SetId ("rxn1");
}

gcpReaction::~gcpReaction()
{
}

void gcpReaction::Add(GtkWidget* w)
{
	map<string, Object*>::iterator i;
	Object* p = GetFirstChild(i);
	while (p)
	{
		p->Add(w);
		p = GetNextChild(i);
	}
}

bool gcpReaction::Build (list<Object*>& Children) throw (invalid_argument)
{
	gcpDocument *pDoc = (gcpDocument*) GetDocument ();
	gcpWidgetData  *pData= (gcpWidgetData*)g_object_get_data(G_OBJECT(pDoc->GetWidget ()), "data");
	map<Object*, ArtDRect> Objects;
	list<Object*>::iterator i, end = Children.end ();
	list<gcpReactionArrow*> Arrows;
	list<Object*> Others;
	map<double, Object*> Left, Right;
	double x0, y0, x1, y1, x, y, xpos, ypos, l;
	bool horiz;
	ArtDRect *rect, srect;
	Object *cur;
	for (i = Children.begin (); i != end; i++) {
		// It might be better to use the objects coordinates there
		pData->GetObjectBounds (*i, &Objects[*i]);
		// Search arrows
		if ((*i)->GetType() == ReactionArrowType)
			Arrows.push_front ((gcpReactionArrow*) (*i));
		else if ((*i)->GetType() == MoleculeType)
			Others.push_front ((gcpMolecule*) (*i));
		else return false;
	}
	if (Arrows.size () == 1) {
	// FIXME: only simple reactions schemes with one arrow are supported in this version
		gcpReactionArrow *arrow = Arrows.front ();
		AddChild (arrow);
		arrow->GetCoords (&x0, &y0, &x1, &y1);
		//x0 and y0 should be the center of the arrow, not the beginning, so we must trnaform them
		x0 = (x0 + x1) / 2;
		y0 = (y0 + y1) / 2;
		// x1, y1 will now be the coordinates of a normalized vector:
		x1 -= x0;
		y1 -= y0;
		x0 *= pData->ZoomFactor;
		y0 *= pData->ZoomFactor;
		l = sqrt (x1 * x1 + y1 * y1);
		x1 /= l;
		y1 /= l;
		// Now, group objects depending of their position relative to the arrow
		// FIXME: objects above or below an arrow are not supported.
		end = Others.end ();
		for (i = Others.begin (); i != end; i++) {
			rect = &Objects[*i];
			xpos = x = (rect->x0 + rect->x1) / 2;
			y = (rect->y0 + rect->y1) / 2;
			// search the direction from the center of the arrow to the center of the object
			x -= x0;
			y -= y0;
			l = sqrt (x * x + y * y);
			x /= l;
			y /= l;
			// scalar product:
			l = x * x1 + y * y1;
			if (l > 0.71) {
				while (Right[xpos] != NULL)
					xpos += 1e-5;
				Right[xpos] = *i;
			}
			else if (l < -0.71) {
				while (Left[xpos] != NULL)
					xpos += 1e-5;
				Left[xpos] = *i;
			}
			else //Object too far from the arrow direction
				throw  invalid_argument(_("Error could not build a reaction\nfrom the selected objects."));
		}
		// We have one or two sets of objects. We must transform them in reaction steps
		gcpReactionStep *step;
		l= 0.;
		if (Left.size ()) {
			step = new gcpReactionStep (this, Left, Objects);
			// Link fisrt step to the arrow
			arrow->SetStartStep (step);
			// Move the arrow to its new position
			pData->GetObjectBounds (step, &srect);
			x0 = (srect.x0 + srect.x1) / 2;
			y0 = step->GetYAlign () * pData->ZoomFactor;
			x = srect.x1 - x0;
			y = srect.y1 - y0;
			if ((fabs (x1) > 1e-5) && (fabs (y1) > 1e-5))
				horiz = (fabs (x1) > fabs (y1));
			else if (fabs (x1) > 1e-5)
				horiz = true;
			else
				horiz = false;
			if (horiz) {
				l = x + pData->ArrowPadding;
				if (x1 < 0)
					l = -l;
				x0 += l;
				y0 += l * y1 / x1;
			} else {
				l = y + pData->ArrowPadding;
				if (y1 < 0)
					l = -l;
				x0 += l * x1 / y1;
				y0 += l;
			}
			arrow->GetCoords (&srect.x0, &srect.y0, &srect.x1, &srect.y1);
			arrow->Move (x0 / pData->ZoomFactor - srect.x0, y0 / pData->ZoomFactor - srect.y0);
		}
		arrow->GetCoords (&srect.x0, &srect.y0, &srect.x1, &srect.y1);
		xpos = srect.x1;
		ypos = srect.y1;
		// Create second step
		if (Right.size ()) {
			step = new gcpReactionStep (this, Right, Objects);
			arrow->SetEndStep (step);
			pData->GetObjectBounds (step, &srect);
			x0 = (srect.x0 + srect.x1) / 2;
			y0 = step->GetYAlign () * pData->ZoomFactor;
			if (l == 0.) {
				if ((fabs (x1) > 1e-5) && (fabs (y1) > 1e-5))
					horiz = (fabs (x1) > fabs (y1));
				else if (fabs (x1) > 1e-5)
					horiz = true;
				else
					horiz = false;
			}
			if (horiz) {
				x = srect.x1 - x0;
				l = x + pData->ArrowPadding;
				if (x1 < 0)
					l = -l;
				x0 -= l;
				y0 -= l * y1 / x1;
			} else {
				y = srect.y1 - y0;
				l = y + pData->ArrowPadding;
				if (x1 < 0)
					l = -l;
				x0 -= l * x1 / y1;
				y0 -= l;
			}
			step->Move (xpos - x0 / pData->ZoomFactor, ypos - y0 / pData->ZoomFactor);
		}
	} else
		throw  invalid_argument(_("Error could not build a reaction\nfrom the selected objects."));
	return true;
}

void gcpReaction::Transform2D(Matrix2D& m, double x, double y)
{
}
