// -*- C++ -*-

/* 
 * GChemPaint
 * standaloneapp.cc
 *
 * Copyright (C) 2004
 *
 * Developed by Jean Bréfort <jean.brefort@ac-dijon.fr>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gchempaint-config.h"
#include "standaloneapp.h"
#include "lib/about.h"
#include "lib/filechooser.h"
#include "lib/mendeleiev.h"
#include <libgnomeprint/gnome-print.h>
#include <libgnomeprint/gnome-print-job.h>
#include <libgnomeprintui/gnome-print-dialog.h>
#include <libgnomeprintui/gnome-print-job-preview.h>

static void on_quit(GtkWidget* widget, gcpStandaloneApp* App)
{
	gtk_main_quit();
}

static void on_select(GtkWidget* widget, GtkWidget* w)
{
	gcpDocument* pDoc = (gcpDocument*) g_object_get_data(G_OBJECT(w), "doc");
	gcpStandaloneApp* App = (gcpStandaloneApp*) pDoc->GetApplication();
	App->SetActive(pDoc, w);
}

static bool on_page_select(GtkWidget* widget, void* data)
{
	on_select(NULL, widget);
	return false;
}

static void on_file_new(GtkWidget* widget, gcpStandaloneApp* App)
{
	App->OnFileNew();
}

static void on_file_open(GtkWidget* widget, gcpStandaloneApp* App)
{
	App->OnFileOpen();
}

void on_file_save_as(GtkWidget* widget, gcpStandaloneApp* App)
{
	App->OnSaveAs();
}

static void on_file_save(GtkWidget* widget, gcpStandaloneApp* App)
{
	App->OnSave();
}

static void on_file_close(GtkWidget* widget, gcpStandaloneApp* App)
{
	App->OnFileClose ();
}

static void on_properties(GtkWidget* widget, gcpStandaloneApp* App)
{
	App->OnProperties();
}

void on_print(GtkWidget* widget, gcpStandaloneApp* App)
{
	App->OnPrint();
}

void on_show_elements(GtkWidget* widget, gcpStandaloneApp* App)
{
	gcpDialog *dlg = App->GetDialog ("Mendeleiev");
	if (dlg)
		dlg->Destroy ();
	else
		App->OnShowElements ();
}

static void on_cut_selection(GtkWidget* widget, gcpStandaloneApp* App)
{
	App->OnCutSelection();
}

static void on_copy_selection(GtkWidget* widget, gcpStandaloneApp* App)
{
	App->OnCopySelection();
}

static void on_undo(GtkWidget* widget, gcpStandaloneApp* App)
{
	App->OnUndo();
}

static void on_redo(GtkWidget* widget, gcpStandaloneApp* App)
{
	App->OnRedo();
}

static void on_select_all(GtkWidget* widget, gcpStandaloneApp* App)
{
	App->OnSelectAll();
}

static void on_paste_selection(GtkWidget* widget, gcpStandaloneApp* App)
{
	App->OnPasteSelection();
}

static void on_delete_selection(GtkWidget* widget, gcpStandaloneApp* App)
{
	App->OnDeleteSelection();
}

static bool on_key_release(GtkWidget* widget, GdkEventKey* ev, gcpStandaloneApp* App)
{
	return App->OnKeyReleased(widget, ev);
}

static bool on_key_press(GtkWidget* widget, GdkEventKey* ev, gcpStandaloneApp* App)
{
	return App->OnKeyPressed(widget, ev);
}

static void on_prefs(GtkWidget* widget, gcpStandaloneApp* App)
{
/*	GtkFontSelectionDialog* dlg = (GtkFontSelectionDialog*)gtk_font_selection_dialog_new("");
	gtk_dialog_run(GTK_DIALOG(dlg));
	gtk_widget_destroy(GTK_WIDGET(dlg));*/
}

static void on_close_all(GtkWidget* widget, gcpStandaloneApp* App)
{
	App->OnCloseAll ();
}

static void on_save_all(GtkWidget* widget, gcpStandaloneApp* App)
{
	App->OnSaveAll ();
}

static bool on_focus(GtkWidget *widget, GdkEventFocus *event, gcpStandaloneApp* App)
{
	gtk_clipboard_request_contents(gtk_clipboard_get(GDK_SELECTION_CLIPBOARD), gdk_atom_intern ("TARGETS", FALSE),  (GtkClipboardReceivedFunc)on_receive_targets, App);
	return false;
}

static bool on_tool_clicked(GtkWidget *widget, gcpStandaloneApp* App)
{
	App->OnToolClicked (widget);
	return false;
}

static void on_help (GtkWidget *widget, gcpStandaloneApp* App)
{
	App->OnHelp ();
}

static void on_web (GtkWidget *widget, gcpStandaloneApp* App)
{
	App->OnWeb ();
}

static void on_mail (GtkWidget *widget, gcpStandaloneApp* App)
{
	App->OnMail ();
}

static void on_bug (GtkWidget *widget, gcpStandaloneApp* App)
{
	App->OnBug ();
}

static void on_show_menu_tip (GtkWidget *proxy, gcpStandaloneApp* App)
{
	GtkAction *action = (GtkAction*) g_object_get_data (G_OBJECT (proxy), "action");
	char *tip;
	g_object_get (action, "tooltip", &tip, NULL);
	if (tip != NULL){
		App->SetStatusText (tip);
		g_free (tip);
	}
}

static void on_clear_menu_tip (gcpStandaloneApp* App)
{
		App->ClearStatus ();
}

static void on_connect_proxy (GtkUIManager *ui, GtkAction *action, GtkWidget *proxy, gcpStandaloneApp* App)
{
	/* connect whether there is a tip or not it may change later */
	if (GTK_IS_MENU_ITEM (proxy)) {
		g_object_set_data (G_OBJECT (proxy), "action", action);
		g_object_connect (proxy,
			"signal::select",  G_CALLBACK (on_show_menu_tip), App,
			"swapped_signal::deselect", G_CALLBACK (on_clear_menu_tip), App,
			NULL);
	}
}

static void on_disconnect_proxy (GtkUIManager *ui, GtkAction *action, GtkWidget *proxy, gcpStandaloneApp* App)
{
	if (GTK_IS_MENU_ITEM (proxy)) {
		g_object_set_data (G_OBJECT (proxy), "action", NULL);
		g_object_disconnect (proxy,
			"any_signal::select",  G_CALLBACK (on_show_menu_tip), App,
			"any_signal::deselect", G_CALLBACK (on_clear_menu_tip), App,
			NULL);
	}
}


/*********
 * Menus *
 *********/
static GtkActionEntry entries[] = {
  { "FileMenu", NULL, N_("_File") },
	  { "New", GTK_STOCK_NEW, N_("_New File"), NULL,
		  N_("Create a new file"), G_CALLBACK (on_file_new) },
	  { "Open", GTK_STOCK_OPEN, N_("_Open"), "<control>O",
		  N_("Open a file"), G_CALLBACK (on_file_open) },
	  { "Save", GTK_STOCK_SAVE, N_("_Save"), "<control>S",
		  N_("Save the current file"), G_CALLBACK (on_file_save) },
	  { "SaveAs", GTK_STOCK_SAVE, N_("Save _As"), "<shift><control>S",
		  N_("Save the current file with a different name "), G_CALLBACK (on_file_save_as) },
	  { "Print", GTK_STOCK_PRINT, N_("_Print"), "<control>P",
		  N_("Print the current file"), G_CALLBACK (on_print) },
	  { "Properties", GTK_STOCK_PROPERTIES, N_("Prope_rties"), NULL,
		  N_("Modify the file's properties"), G_CALLBACK (on_properties) },
	  { "Close", GTK_STOCK_CLOSE, N_("_Close"), "<control>W",
		  N_("Close the current file"), G_CALLBACK (on_file_close) },
	  { "Quit", GTK_STOCK_QUIT, N_("_Quit"), "<control>Q",
		  N_("Quit GChemPaint"), G_CALLBACK (on_quit) },
  { "EditMenu", NULL, N_("_Edit") },
	  { "Undo", GTK_STOCK_UNDO, N_("_Undo"), "<control>Z",
		  N_("Undo the lase action"), G_CALLBACK (on_undo) },
	  { "Redo", GTK_STOCK_REDO, N_("_Redo"), "<shift><control>Z",
		  N_("Redo the undone action"), G_CALLBACK (on_redo) },
	  { "Cut", GTK_STOCK_CUT, N_("Cu_t"), "<control>X",
		  N_("Cut the selection"), G_CALLBACK (on_cut_selection) },
	  { "Copy", GTK_STOCK_COPY, N_("_Copy"), "<control>C",
		  N_("Copy the selection"), G_CALLBACK (on_copy_selection) },
	  { "Paste", GTK_STOCK_PASTE, N_("_Paste"), "<control>V",
		  N_("Paste the clipborad"), G_CALLBACK (on_paste_selection) },
	  { "Delete", GTK_STOCK_CLEAR, N_("C_lear"), NULL,
		  N_("Clear the selection"), G_CALLBACK (on_delete_selection) },
	  { "SelectAll", NULL, N_("Select _All"), "<control>A",
		  N_("Select everything"), G_CALLBACK (on_select_all) },
  { "ViewMenu", NULL, N_("_View") },
	  { "ToolbarMenu", NULL, N_("_Toolbars") },
  { "WindowsMenu", NULL, N_("_Windows") },
	  { "SaveAll", GTK_STOCK_SAVE, N_("_Save All"), "<shift><control>L",
		  N_("Save all open files"), G_CALLBACK (on_save_all) },
	  { "CloseAll", GTK_STOCK_CLOSE, N_("_Close All"), "<control>W",
		  N_("Close all open files"), G_CALLBACK (on_close_all) },
  { "HelpMenu", NULL, N_("_Help") },
	  { "Help", GTK_STOCK_HELP, N_("_Contents"), "F1",
		  N_("View help for GChemPaint"), G_CALLBACK (on_help) },
	  { "Web", NULL, N_("GChemPaint on the _web"), NULL,
		  N_("Browse GChemPaint's web site"), G_CALLBACK (on_web) },
	  { "Mail", NULL, N_("_Ask a question"), NULL,
		  N_("Ask a question about GChemPaint"), G_CALLBACK (on_mail) },
	  { "Bug", NULL, N_("Report _Bugs"), NULL,
		  N_("Submit a bug report for GChemPaint"), G_CALLBACK (on_bug) },
	  { "About", NULL, N_("_About"), NULL,
		  N_("About GChemPaint"), G_CALLBACK (on_about) }
};

/* Toggle items */
static GtkToggleActionEntry toggle_entries[] = {
	  { "Mendeleiev", NULL, N_("_Periodic table"), NULL,
		  N_("Show the periodic table of the elements"), G_CALLBACK (on_show_elements), false }
};

static const char *ui_description =
"<ui>"
"  <menubar name='MainMenu'>"
"    <menu action='FileMenu'>"
"      <menuitem action='New'/>"
"      <menuitem action='Open'/>"
"      <menuitem action='Save'/>"
"      <menuitem action='SaveAs'/>"
"      <separator name='file-sep1'/>"
"      <menuitem action='Print'/>"
"      <separator name='file-sep2'/>"
"      <menuitem action='Properties'/>"
"      <separator name='file-sep3'/>"
"      <menuitem action='Close'/>"
"      <menuitem action='Quit'/>"
"    </menu>"
"    <menu action='EditMenu'>"
"      <menuitem action='Undo'/>"
"      <menuitem action='Redo'/>"
"      <separator name='edit-sep1'/>"
"      <menuitem action='Cut'/>"
"      <menuitem action='Copy'/>"
"      <menuitem action='Paste'/>"
"      <menuitem action='Delete'/>"
"      <separator name='edit-sep2'/>"
"      <menuitem action='SelectAll'/>"
"    </menu>"
"    <menu action='ViewMenu'>"
"      <menu action='ToolbarMenu'>"
"      </menu>"
"      <menuitem action='Mendeleiev'/>"
"    </menu>"
"    <menu action='WindowsMenu'>"
"      <menuitem action='SaveAll'/>"
"      <menuitem action='CloseAll'/>"
"      <separator name='windows-sep1'/>"
"      <placeholder name='windows'/>"
"    </menu>"
"    <menu action='HelpMenu'>"
"      <menuitem action='Help'/>"
"      <placeholder name='web'/>"
"      <placeholder name='mail'/>"
"      <placeholder name='bug'/>"
"      <menuitem action='About'/>"
"    </menu>"
"  </menubar>"
"  <toolbar name='MainToolbar'>"
"    <toolitem action='New'/>"
"    <toolitem action='Open'/>"
"    <toolitem action='Save'/>"
"    <toolitem action='Print'/>"
"    <toolitem action='Close'/>"
"    <toolitem action='Quit'/>"
"  </toolbar>"
"</ui>";

static const char *ui_mail_description =
"<ui>"
"  <menubar name='MainMenu'>"
"    <menu action='HelpMenu'>"
"      <placeholder name='mail'>"
"        <menuitem action='Mail'/>"
"      </placeholder>"
"    </menu>"
"  </menubar>"
"</ui>";

static const char *ui_web_description =
"<ui>"
"  <menubar name='MainMenu'>"
"    <menu action='HelpMenu'>"
"      <placeholder name='web'>"
"        <menuitem action='Web'/>"
"      </placeholder>"
"      <placeholder name='bug'>"
"        <menuitem action='Bug'/>"
"      </placeholder>"
"    </menu>"
"  </menubar>"
"</ui>";

#warning "the following lines should be removed for stable releases"
#undef PACKAGE
#define PACKAGE "gchempaint-unstable" 

gcpStandaloneApp::gcpStandaloneApp(): gcpApplication()
{
	GtkWidget *vbox;
	GtkWidget *bar;
	GtkWidget *item;
	GtkActionGroup *action_group;
	GtkAccelGroup *accel_group;
	GError *error;

	m_NumWindow = 1;
	m_NumDoc = 0;
	m_MessageId = 0;
	m_Window = GTK_WINDOW (gtk_window_new (GTK_WINDOW_TOPLEVEL));
	gtk_window_set_title (m_Window, _("GChemPaint"));
	gtk_window_set_icon_from_file (m_Window, PIXMAPSDIR"/gchempaint.png", NULL);
	g_signal_connect(G_OBJECT(m_Window), "destroy", G_CALLBACK(on_quit), this);
	g_signal_connect(G_OBJECT(m_Window), "focus_in_event", G_CALLBACK(on_focus), this);
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (m_Window), vbox);
	action_group = gtk_action_group_new ("MenuActions");
	gtk_action_group_add_actions (action_group, entries, G_N_ELEMENTS (entries), this);
	gtk_action_group_add_toggle_actions (action_group, toggle_entries, G_N_ELEMENTS (toggle_entries), this);

	m_UIManager = gtk_ui_manager_new ();
	g_object_connect (m_UIManager,
		"signal::connect_proxy",    G_CALLBACK (on_connect_proxy), this,
		"signal::disconnect_proxy", G_CALLBACK (on_disconnect_proxy), this,
		NULL);
	gtk_ui_manager_insert_action_group (m_UIManager, action_group, 0);
	if (MailAgent[0])

	accel_group = gtk_ui_manager_get_accel_group (m_UIManager);
	gtk_window_add_accel_group (GTK_WINDOW (m_Window), accel_group);
	
	if (MailAgent[0])
	error = NULL;
	if (!gtk_ui_manager_add_ui_from_string (m_UIManager, ui_description, -1, &error))
	  {
		g_message ("building menus failed: %s", error->message);
		g_error_free (error);
		exit (EXIT_FAILURE);
	  }
	if (WebBrowser[0])
		gtk_ui_manager_add_ui_from_string (m_UIManager, ui_web_description, -1, &error);
	if (MailAgent[0])
		gtk_ui_manager_add_ui_from_string (m_UIManager, ui_mail_description, -1, &error);

	bar = gtk_ui_manager_get_widget (m_UIManager, "/MainMenu");
	gtk_box_pack_start (GTK_BOX (vbox), bar, false, false, 0);
	m_Dock = bonobo_dock_new ();
	gtk_box_pack_start (GTK_BOX (vbox), GTK_WIDGET (m_Dock), true, true, 0);
	bar = gtk_ui_manager_get_widget (m_UIManager, "/MainToolbar");
	gtk_toolbar_set_style(GTK_TOOLBAR(bar), GTK_TOOLBAR_ICONS);
	gtk_toolbar_set_show_arrow(GTK_TOOLBAR(bar), false);
	gtk_toolbar_set_tooltips(GTK_TOOLBAR(bar), true);
	item = bonobo_dock_item_new (_("Main"), BONOBO_DOCK_ITEM_BEH_NORMAL);
	gtk_container_add (GTK_CONTAINER (item), bar);
	bonobo_dock_add_item (BONOBO_DOCK (m_Dock),BONOBO_DOCK_ITEM (item), BONOBO_DOCK_TOP, 0, 0, 0, true);

	m_Book = GTK_NOTEBOOK(gtk_notebook_new());
	gtk_notebook_set_tab_pos(m_Book, GTK_POS_TOP);
	bonobo_dock_set_client_area (BONOBO_DOCK (m_Dock), GTK_WIDGET (m_Book));

	m_Bar = gtk_statusbar_new ();
	m_statusId = gtk_statusbar_get_context_id (GTK_STATUSBAR (m_Bar), "status");
	gtk_statusbar_push (GTK_STATUSBAR (m_Bar), m_statusId, _("Ready"));
	gtk_box_pack_start (GTK_BOX (vbox), m_Bar, false, false, 0);
	gtk_widget_show_all (GTK_WIDGET (m_Window));
	SetMenu("Undo", gtk_ui_manager_get_widget (m_UIManager, "/MainMenu/EditMenu/Undo"));
	SetMenu("Redo", gtk_ui_manager_get_widget (m_UIManager, "/MainMenu/EditMenu/Redo"));
	SetMenu("Copy", gtk_ui_manager_get_widget (m_UIManager, "/MainMenu/EditMenu/Copy"));
	SetMenu("Cut", gtk_ui_manager_get_widget (m_UIManager, "/MainMenu/EditMenu/Cut"));
	SetMenu("Paste", gtk_ui_manager_get_widget (m_UIManager, "/MainMenu/EditMenu/Paste"));
	SetMenu("Erase", gtk_ui_manager_get_widget (m_UIManager, "/MainMenu/EditMenu/Delete"));
	ActivateMenu("Undo", false);
	ActivateMenu("Redo", false);
	ActivateMenu("Copy", false);
	ActivateMenu("Cut", false);
	ActivateMenu("Paste", false);
	ActivateMenu("Erase", false);

	Callbacks["tools"] = (GCallback) on_tool_clicked;

	if (m_pNode && !strcmp(bonobo_ui_node_get_name(m_pNode), "Root")) {
		BonoboUINode* Child = bonobo_ui_node_children (m_pNode);
		AppendUI (Child);
	}
		
	g_signal_connect(GTK_OBJECT(m_Window), "key_press_event", (GCallback)on_key_press, this);
	g_signal_connect(GTK_OBJECT(m_Window), "key_release_event", (GCallback)on_key_release, this);

	Callbacks["tools"] = (GCallback) on_tool_clicked;

	SetMenu("Mendeleiev", gtk_ui_manager_get_widget (m_UIManager, "/MainMenu/ViewMenu/Mendeleiev"));
	gcpMendeleievDlg::SetElement (this, m_CurZ);
}

gcpStandaloneApp::~gcpStandaloneApp()
{
	while (!m_Docs.empty())
		delete *m_Docs.begin();
}

void gcpStandaloneApp::OnToolClicked (GtkWidget* widget)
{
	const char* name = gtk_widget_get_name(widget);
	bool active = gtk_toggle_tool_button_get_active(GTK_TOGGLE_TOOL_BUTTON(widget));
	if (m_pActiveTool)
	{
		if (active && !strcmp(name, m_pActiveTool->GetName().c_str()))
		{
			m_pActiveTool->SetOptions ();
			return;
		}
		m_pActiveTool->Activate(false);
	}
	m_pActiveTool = Tools[name];
	if (m_pActiveTool)
		m_pActiveTool->Activate(true);
}

void gcpStandaloneApp::OnDoubleClick(GtkWidget* widget)
{
	const char* name = gtk_widget_get_name(widget);
	gcpTool* tool = Tools[name];
	if (tool) tool->SetOptions();
}

void gcpStandaloneApp::OnRightClick(GtkWidget* widget)
{
	const char* name = gtk_widget_get_name(widget);
	gcpTool* tool = Tools[name];
	if (tool) tool->SetOptions();
}

void gcpStandaloneApp::OnFileNew()
{
	gchar tmp[32];
	if (m_pActiveDoc && !m_pActiveDoc->GetView()->PrepareUnselect()) return;
	g_snprintf(tmp, sizeof(tmp), _("Untitled%d"), m_NumWindow++);
	GtkWidget* w = CreateView();
	if (w)
	{
		GtkWidget *label = gtk_label_new(tmp);
		g_snprintf(tmp, sizeof(tmp), "win%d", m_NumDoc++);
		GtkAction *action = gtk_action_new (tmp, gtk_label_get_text (GTK_LABEL (label)), NULL, NULL);
		GtkActionGroup *group = gtk_action_group_new (tmp);
		GError* error;
		gtk_action_group_add_action (group, action);
		gtk_ui_manager_insert_action_group (m_UIManager, group, 0);
		gchar* buf = g_strconcat ("<ui><menubar name='MainMenu'><menu action='WindowsMenu'><menuitem action='",
		tmp, "'/></menu></menubar></ui>", NULL);
		error = NULL;
		int id = gtk_ui_manager_add_ui_from_string (m_UIManager, buf, -1, NULL);
		g_free (buf);
		buf = g_strconcat ("/MainMenu/WindowsMenu/", tmp, NULL);
		GtkWidget* item = gtk_ui_manager_get_widget (m_UIManager, buf);
		g_free (buf);
		g_signal_connect(G_OBJECT(item), "activate", (GtkSignalFunc)on_select, w); 
		g_signal_connect(G_OBJECT(w), "expose_event", (GtkSignalFunc)on_page_select, NULL); 
		g_object_set_data(G_OBJECT(w), "menu-id", (void*) id);
		g_object_set_data(G_OBJECT(w), "label", label);
		GtkScrolledWindow* scroll = (GtkScrolledWindow*)gtk_scrolled_window_new(NULL, NULL);
		gtk_scrolled_window_set_policy(scroll, GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
		gtk_scrolled_window_set_shadow_type (scroll, GTK_SHADOW_IN);
		gtk_scrolled_window_add_with_viewport(scroll, w);
		gtk_widget_set_size_request(GTK_WIDGET(scroll), 400, 300);
		gtk_widget_show(GTK_WIDGET(scroll));
		gtk_notebook_append_page(m_Book, GTK_WIDGET(scroll), label);
		gtk_notebook_set_current_page(m_Book, g_list_length(m_Book->children) - 1);
		m_pActiveDoc = (gcpDocument*) g_object_get_data(G_OBJECT(w), "doc");
	}
}

void gcpStandaloneApp::OnFileOpen()
{
	gcpFileChooser* fc = new gcpFileChooser (this, false);
}

void  gcpStandaloneApp::AddDocument(gcpDocument* pDoc)
{
	if (!pDoc) return;
	if (pDoc)
	{
		m_Docs.push_back(pDoc);
		m_pActiveDoc = pDoc;
	}
	GtkWidget* w = pDoc->GetWidget();
	if (w)
	{
		const char* tmp = pDoc->GetLabel();
		char winname[32];
		GtkWidget *label = gtk_label_new(tmp);
		g_snprintf(winname, sizeof(winname), "win%d", m_NumDoc++);
		GtkAction *action = gtk_action_new (winname, tmp, NULL, NULL);
		GtkActionGroup *group = gtk_action_group_new (winname);
		GError* error;
		gtk_action_group_add_action (group, action);
		gtk_ui_manager_insert_action_group (m_UIManager, group, 0);
		gchar* buf = g_strconcat ("<ui><menubar name='MainMenu'><menu action='WindowsMenu'><menuitem action='",
		winname, "'/></menu></menubar></ui>", NULL);
		error = NULL;
		int id = gtk_ui_manager_add_ui_from_string (m_UIManager, buf, -1, NULL);
		g_free (buf);
		buf = g_strconcat ("/MainMenu/WindowsMenu/", winname, NULL);
		GtkWidget* item = gtk_ui_manager_get_widget (m_UIManager, buf);
		g_free (buf);
		g_signal_connect(G_OBJECT(item), "activate", (GtkSignalFunc)on_select, w); 
		g_signal_connect(G_OBJECT(w), "expose_event", (GtkSignalFunc)on_page_select, NULL); 
		g_object_set_data(G_OBJECT(w), "menu-id", (void*) id);
		g_object_set_data(G_OBJECT(w), "label", label);
		GtkScrolledWindow* scroll = (GtkScrolledWindow*)gtk_scrolled_window_new(NULL, NULL);
		gtk_scrolled_window_set_policy(scroll, GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
		gtk_scrolled_window_set_shadow_type (scroll, GTK_SHADOW_IN);
		gtk_scrolled_window_add_with_viewport(scroll, w);
		gtk_widget_set_size_request(GTK_WIDGET(scroll), 400, 300);
		gtk_widget_show(GTK_WIDGET(scroll));
		gtk_notebook_append_page(m_Book, GTK_WIDGET(scroll), label);
		gtk_notebook_set_current_page(m_Book, g_list_length(m_Book->children) - 1);
		m_pActiveDoc = (gcpDocument*) g_object_get_data(G_OBJECT(w), "doc");
	}
}

void gcpStandaloneApp::RemoveDocument(gcpDocument* pDoc)
{
	GtkWidget *w = pDoc->GetWidget ();
	gtk_ui_manager_remove_ui (m_UIManager, (int) g_object_get_data (G_OBJECT (w), "menu-id"));
	m_Docs.remove (pDoc);
}

GtkWidget* gcpStandaloneApp::CreateView()
{
	gcpDocument* pDoc = new gcpDocument(this, true);
	if (pDoc)
	{
		m_Docs.push_back(pDoc);
		m_pActiveDoc = pDoc;
		return pDoc->GetWidget();
	}
	else return NULL;
}

void gcpStandaloneApp::OnSave()
{
	if (m_pActiveDoc == NULL) return;
	if (m_pActiveDoc->GetFileName()) m_pActiveDoc->Save();
	else OnSaveAs();
}

void gcpStandaloneApp::ActivateMenu(const string& menuname, bool activate)
{
	GtkWidget* w = Menus[menuname];
	if (w) gtk_widget_set_sensitive(w, activate);
}

void gcpStandaloneApp::ActivateToolItem(const string& itemname, bool activate)
{
	GtkWidget* w = ToolItems[itemname];
	if (w) gtk_widget_set_sensitive(w, activate);
}

void gcpStandaloneApp::ClearStatus()
{
	if (m_MessageId) {
		gtk_statusbar_pop (GTK_STATUSBAR (m_Bar), m_statusId);
		m_MessageId = 0;
	}
}

void gcpStandaloneApp::SetStatusText(const char* text)
{
	if (m_MessageId) gtk_statusbar_pop (GTK_STATUSBAR (m_Bar), m_statusId);
	m_MessageId = gtk_statusbar_push (GTK_STATUSBAR (m_Bar), m_statusId, text);
}

GtkWindow* gcpStandaloneApp::GetWindow()
{
	return m_Window;
}

void gcpStandaloneApp::AppendUI (BonoboUINode* Node)
{
	map<unsigned, GtkWidget*>toolbars;
	while (Node)
	{
		if (!strcmp(bonobo_ui_node_get_name(Node), "dockitem"))
		{
			AddToolbar (Node, &toolbars);
		}
		Node = bonobo_ui_node_next (Node);
	}
	map<unsigned, GtkWidget*>::iterator i, end = toolbars.end();
	for (i = toolbars.begin(); i != end; i++)
		bonobo_dock_add_item (BONOBO_DOCK (m_Dock), BONOBO_DOCK_ITEM ((*i).second), BONOBO_DOCK_TOP, (*i).first >> 8, (*i).first & 0xff, 0, 0);
	GtkToggleToolButton* button = (GtkToggleToolButton*) ToolItems["Select"];
	if (button && !gtk_toggle_tool_button_get_active (button))
		gtk_toggle_tool_button_set_active (button, true);
	InitTools();
}

void gcpStandaloneApp::AddToolbar (BonoboUINode* Node, map<unsigned, GtkWidget*> *toolbars)
{
	if (!bonobo_ui_node_has_attr (Node, "name")) return;
	if (!bonobo_ui_node_has_attr (Node, "position")) return;
	if (!bonobo_ui_node_has_attr (Node, "band_num")) return;
	char *name = NULL;
	int in_new_band = (bonobo_ui_node_has_attr (Node, "in_new_band"))?
				name = bonobo_ui_node_get_attr (Node, "in_new_band"), *name - '0': 1;
	if (name)
	{
		bonobo_ui_node_free_string (name);
		name = NULL;
	}
	name = bonobo_ui_node_get_attr (Node, "band_num");
	int row = (name)? atoi (name): 0;
	if (name)
	{
		bonobo_ui_node_free_string (name);
		name = NULL;
	}
	name = bonobo_ui_node_get_attr (Node, "position");
	int rank = (name)? atoi (name): 0;
	if (name)
	{
		bonobo_ui_node_free_string (name);
		name = NULL;
	}
	name = bonobo_ui_node_get_attr (Node, "name");
	GtkToolbar* toolbar = GTK_TOOLBAR (gtk_toolbar_new());
	Toolbars[name] = (GtkWidget*)toolbar;
	gtk_toolbar_set_show_arrow(toolbar, false);
	gtk_toolbar_set_style(GTK_TOOLBAR(toolbar), GTK_TOOLBAR_ICONS);
	gtk_toolbar_set_tooltips(GTK_TOOLBAR(toolbar), true);
	GtkWidget *item = bonobo_dock_item_new (name, BONOBO_DOCK_ITEM_BEH_NORMAL);
	gtk_container_add (GTK_CONTAINER (item), GTK_WIDGET (toolbar));
	gtk_widget_show_all(item);
	if (toolbars && !in_new_band)
		(*toolbars)[(row << 8) + rank] = item;
	else
		bonobo_dock_add_item (BONOBO_DOCK (m_Dock), BONOBO_DOCK_ITEM (item), BONOBO_DOCK_TOP, row, rank, 0, in_new_band);
	if (name)
	{
		bonobo_ui_node_free_string (name);
		name = NULL;
	}
	BonoboUINode* Child = bonobo_ui_node_children (Node);
	GtkToolItem* button;
	int pos;
	while (Child) {
		button = NULL;
		name = (char*) bonobo_ui_node_get_name (Child);
		if (!strcmp (name, "toolitem")) {
			char* type;
			if (bonobo_ui_node_has_attr (Child, "type")) {
				type = bonobo_ui_node_get_attr (Child, "type");
				if (!strcmp (type, "toggle")) {
					button = gtk_toggle_tool_button_new ();
				} else if (!strcmp (type, "radio")) {
					if (bonobo_ui_node_has_attr (Child, "group")) {
						char* group = bonobo_ui_node_get_attr (Child, "group");
						if (Groups[group])
							button = gtk_radio_tool_button_new_from_widget(Groups[group]);
						else {
							Groups[group] = (GtkRadioToolButton*) (button = gtk_radio_tool_button_new (NULL));
						}
						bonobo_ui_node_free_string (group);
					} else
						button = gtk_radio_tool_button_new (NULL);
				} else if (!strcmp (type, "widget")) {
					if (bonobo_ui_node_has_attr (Child, "build")) {
						char* build = bonobo_ui_node_get_attr (Child, "build");
						if (Callbacks[build])
							button = ((GtkToolItem* (*)(gcpApplication*)) Callbacks[build]) (this);
						bonobo_ui_node_free_string (build);
					}
				}
				bonobo_ui_node_free_string (type);
			}
			else button = gtk_tool_button_new (NULL, NULL);
			if (GTK_IS_TOOL_BUTTON (button)) {
				if (bonobo_ui_node_has_attr (Child, "pixtype")) {
					const char* filename = NULL;
					name = bonobo_ui_node_get_attr (Child, "pixtype");
					if (!strcmp (name, "filename")) {
						bonobo_ui_node_free_string (name);
						char* filename = NULL;
						if (bonobo_ui_node_has_attr (Child, "pixname")) {
							name = bonobo_ui_node_get_attr (Child, "pixname");
							filename = g_strconcat (DATADIR"/gchempaint/ui/", name, NULL);
							bonobo_ui_node_free_string (name);
						}
						if (filename) {
							gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(button), gtk_image_new_from_file (filename));
							g_free (filename);
						}
					} else if (!strcmp (name, "stock")) {
						bonobo_ui_node_free_string (name);
						if (bonobo_ui_node_has_attr (Child, "pixname")) {
							name = bonobo_ui_node_get_attr (Child, "pixname");
							gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(button), gtk_image_new_from_stock (name, GTK_ICON_SIZE_LARGE_TOOLBAR));
							bonobo_ui_node_free_string (name);
						}
					}
				}
				if (bonobo_ui_node_has_attr (Child, "label")) {
					name = bonobo_ui_node_get_attr (Child, "label");
					gtk_tool_button_set_label(GTK_TOOL_BUTTON(button), name);
					bonobo_ui_node_free_string (name);
				}
			}
			if (bonobo_ui_node_has_attr (Child, "verb")) {
				name = bonobo_ui_node_get_attr (Child, "verb");
				if (Callbacks[name])
					g_signal_connect (button, "clicked", Callbacks[name], this);
				bonobo_ui_node_free_string (name);
			}
			if (bonobo_ui_node_has_attr (Child, "name")) {
				name = bonobo_ui_node_get_attr (Child, "name");
				gtk_widget_set_name(GTK_WIDGET(button), name);
				SetToolItem (name, (GtkWidget*) button);
				bonobo_ui_node_free_string (name);
			}
			pos = -1;
			if (bonobo_ui_node_has_attr (Child, "pos")) {
				name = bonobo_ui_node_get_attr (Child, "pos");
				if (!strcmp (name, "top")) pos = 0;
				else if (!strcmp (name, "bottom"));
				else pos = atoi (name);
				bonobo_ui_node_free_string (name);
			}
			if (bonobo_ui_node_has_attr (Child, "tip")) {
				name = bonobo_ui_node_get_attr (Child, "tip");
				gtk_tool_item_set_tooltip (button, toolbar->tooltips, name, NULL);
				bonobo_ui_node_free_string (name);
			}
			if (pos > (rank = gtk_toolbar_get_n_items (toolbar)))
				pos = rank;
			if (button)
				gtk_toolbar_insert(toolbar, button, pos);
		}
		else if (!strcmp (name, "separator"))
			gtk_toolbar_insert (toolbar, gtk_separator_tool_item_new (), -1);
		Child = bonobo_ui_node_next (Child);
	}
	gtk_widget_show_all(GTK_WIDGET(toolbar));
}

void gcpStandaloneApp::OnProperties()
{
	m_pActiveDoc->OnProperties();
}

void gcpStandaloneApp::OnPrint()
{
	if (!m_pActiveDoc) return;
	GnomePrintConfig* config = gnome_print_config_default();
	GnomePrintContext *pc;// = gnome_print_context_new(config);
	GnomePrintJob *gpj = gnome_print_job_new(config);
	int do_preview, copies = 1, collate = 0;
	GnomePrintDialog *gpd;
	gpd = GNOME_PRINT_DIALOG (gnome_print_dialog_new(gpj, (const guchar*)_("Print"), GNOME_PRINT_DIALOG_COPIES));
	gnome_print_dialog_set_copies(gpd, copies, collate);
	switch (gtk_dialog_run(GTK_DIALOG(gpd)))
	{
	case GNOME_PRINT_DIALOG_RESPONSE_PRINT:
		do_preview = 0;
		break;
	case GNOME_PRINT_DIALOG_RESPONSE_PREVIEW:
		do_preview = 1;
		break;
	case GNOME_PRINT_DIALOG_RESPONSE_CANCEL:
		gtk_widget_destroy (GTK_WIDGET(gpd));
		return;
	}
	gtk_widget_destroy(GTK_WIDGET(gpd));
	pc = gnome_print_job_get_context (gpj);
	gnome_print_beginpage(pc, (const guchar*)"");
	gdouble width, height;
	gnome_print_config_get_page_size(config, &width, &height);
	m_pActiveDoc->Print(pc, width, height);
	gnome_print_showpage(pc);
	g_object_unref(pc);
	gnome_print_job_close(gpj);
	if (do_preview)
	{
		gtk_widget_show (gnome_print_job_preview_new (gpj, (const guchar*)_("Preview")));
	}
	else
	{
		gnome_print_job_print(gpj);
	}
	g_object_unref (gpj);
	gnome_print_config_unref(config);
}

void gcpStandaloneApp::SetActive (gcpDocument* pDoc, GtkWidget* w)
{
	if (pDoc != m_pActiveDoc)
	{
		if (m_pActiveDoc && !m_pActiveDoc->GetView()->PrepareUnselect()) 
		{
			w = m_pActiveDoc->GetView()->GetWidget();
		}
		else
		{
			m_pActiveDoc = pDoc;
			pDoc->SetActive();
		}
		gint i;
		i = gtk_notebook_page_num(m_Book, gtk_widget_get_parent(gtk_widget_get_parent(w)));
		if(i != gtk_notebook_get_current_page(m_Book))
			gtk_notebook_set_current_page(m_Book, i);
	}
}

void gcpStandaloneApp::OnFileClose ()
{
	gcpView* pView = m_pActiveDoc->GetView ();
	pView->PrepareUnselect (); 
	m_pActiveDoc = NULL;
	gint i;
	i = gtk_notebook_get_current_page (m_Book);
	gtk_notebook_remove_page (m_Book, i);
}

bool gcpStandaloneApp::OnKeyPressed(GtkWidget* widget, GdkEventKey* ev)
{
	gint i = gtk_notebook_get_current_page(m_Book);
	GtkWidget* w = gtk_bin_get_child(GTK_BIN(gtk_bin_get_child(GTK_BIN(gtk_notebook_get_nth_page(m_Book, i)))));
	gcpView* v = (gcpView*) g_object_get_data(G_OBJECT(w), "view");
	if (v && v->GetDoc()->GetEditable()) return v->OnKeyPress(w, ev);
	return false;
}

bool gcpStandaloneApp::OnKeyReleased(GtkWidget* widget, GdkEventKey* ev)
{
	gint i = gtk_notebook_get_current_page(m_Book);
	GtkWidget* w = gtk_bin_get_child(GTK_BIN(gtk_bin_get_child(GTK_BIN(gtk_notebook_get_nth_page(m_Book, i)))));
	gcpView* v = (gcpView*) g_object_get_data(G_OBJECT(w), "view");
	if (v && v->GetDoc()->GetEditable()) return v->OnKeyRelease(w, ev);
	return false;
}

void gcpStandaloneApp::OnUndo()
{
	m_pActiveDoc->OnUndo();
}

void gcpStandaloneApp::OnRedo()
{
	m_pActiveDoc->OnRedo();
}

void gcpStandaloneApp::OnSelectAll()
{
	gint i = gtk_notebook_get_current_page(m_Book);
	GtkWidget* w = gtk_bin_get_child(GTK_BIN(gtk_bin_get_child(GTK_BIN(gtk_notebook_get_nth_page(m_Book, i)))));
	gcpView* v = (gcpView*) g_object_get_data(G_OBJECT(w), "view");
	if (v && v->GetDoc()->GetEditable()) v->OnSelectAll();
}

void gcpStandaloneApp::OnPasteSelection()
{
	gint i = gtk_notebook_get_current_page(m_Book);
	GtkWidget* w = gtk_bin_get_child(GTK_BIN(gtk_bin_get_child(GTK_BIN(gtk_notebook_get_nth_page(m_Book, i)))));
	gcpView* v = (gcpView*) g_object_get_data(G_OBJECT(w), "view");
	GtkClipboard* clipboard = gtk_clipboard_get(GDK_SELECTION_CLIPBOARD);
	if (v && v->GetDoc()->GetEditable()) v->OnPasteSelection(w, clipboard);
}

void gcpStandaloneApp::OnCutSelection()
{
	gint i = gtk_notebook_get_current_page(m_Book);
	GtkWidget* w = gtk_bin_get_child(GTK_BIN(gtk_bin_get_child(GTK_BIN(gtk_notebook_get_nth_page(m_Book, i)))));
	gcpView* v = (gcpView*) g_object_get_data(G_OBJECT(w), "view");
	GtkClipboard* clipboard = gtk_clipboard_get(GDK_SELECTION_CLIPBOARD);
	if (v && v->GetDoc()->GetEditable()) v->OnCutSelection(w, clipboard);
}

void gcpStandaloneApp::OnCopySelection()
{
	gint i = gtk_notebook_get_current_page(m_Book);
	GtkWidget* w = gtk_bin_get_child(GTK_BIN(gtk_bin_get_child(GTK_BIN(gtk_notebook_get_nth_page(m_Book, i)))));
	gcpView* v = (gcpView*) g_object_get_data(G_OBJECT(w), "view");
	GtkClipboard* clipboard = gtk_clipboard_get(GDK_SELECTION_CLIPBOARD);
	if (v && v->GetDoc()->GetEditable()) v->OnCopySelection(w, clipboard);
}

void gcpStandaloneApp::OnDeleteSelection()
{
	gint i = gtk_notebook_get_current_page(m_Book);
	GtkWidget* w = gtk_bin_get_child(GTK_BIN(gtk_bin_get_child(GTK_BIN(gtk_notebook_get_nth_page(m_Book, i)))));
	gcpView* v = (gcpView*) g_object_get_data(G_OBJECT(w), "view");
	if (v && v->GetDoc()->GetEditable()) v->OnDeleteSelection(w);
}

void gcpStandaloneApp::OnCloseAll ()
{
	while (m_Docs.size ()) {
		OnFileClose ();
		while (gtk_events_pending())
			gtk_main_iteration();
	}
}

void gcpStandaloneApp::OnSaveAll ()
{
	gcpDocument* pDoc = m_pActiveDoc;
	list<gcpDocument*>::iterator i, end = m_Docs.end ();
	for (i = m_Docs.begin (); i != end; i++) {
		if (!(*i)->IsDirty ())
			continue;
		on_select (NULL, (*i)->GetWidget ());
		OnSave ();
	}
	if (pDoc)
		on_select (NULL, pDoc->GetWidget ());
}

void gcpStandaloneApp::ToggleMenu(const string& menuname, bool active)
{
	GtkWidget* w = Menus[menuname];
	if (!GTK_IS_CHECK_MENU_ITEM (w))
		return;
	if (gtk_check_menu_item_get_active ((GtkCheckMenuItem*) w) != active)
		((GtkCheckMenuItem*) w)->active = active;
}
