// -*- C++ -*-

/* 
 * GChemPaint
 * atom.cc
 *
 * Copyright (C) 2001-2004
 *
 * Developed by Jean Bréfort <jean.brefort@ac-dijon.fr>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <mol.h>
#undef PACKAGE
#undef PACKAGE_BUGREPORT
#undef PACKAGE_NAME
#undef PACKAGE_STRING
#undef PACKAGE_TARNAME
#undef PACKAGE_VERSION
#undef VERSION
#include "config.h"
#include "atom.h"
#include "bond.h"
#include "molecule.h"
#include "globals.h"
#include "settings.h"
#include "view.h"
#include "widgetdata.h"
#include "libgcpcanvas/gcp-canvas-group.h"
#include "libgcpcanvas/gcp-canvas-rect-ellipse.h"
#include "libgcpcanvas/gcp-canvas-line.h"
#include "libgcpcanvas/gcp-canvas-rich-text.h"
#include <math.h>
#ifdef GCU_OLD_VER
#	include <chemistry/element.h>
#else
#	include <gcu/element.h>
#endif

using namespace gcu;
using namespace OpenBabel;

extern GtkTextTagTable* TextTagTable;

gcpAtom::gcpAtom(): Atom()
{
	m_buf = gtk_text_buffer_new(TextTagTable);
	m_Valence = -1; //unspecified
	m_nve = 0;
	m_nH = 0;
	m_HPos = GetBestSide();
	m_ChargeAuto = false;
	m_ascent = 0;
	m_CHeight = 0.;
}

gcpAtom::~gcpAtom()
{
}

gcpAtom::gcpAtom(int Z, double x, double y, double z): Atom(Z, x, y, z)
{
	m_buf = gtk_text_buffer_new(TextTagTable);
	m_ChargeAuto = false;
	m_HPos = GetBestSide();
	SetZ(Z);
	m_ascent = 0;
	m_CHeight = 0.;
}

gcpAtom::gcpAtom(OBAtom* atom, double zoom): Atom()
{
	m_buf = gtk_text_buffer_new(TextTagTable);
	m_x = atom->GetX() * zoom;
	m_y = - atom->GetY() * zoom;
	m_z = atom->GetZ() * zoom;
	SetZ(atom->GetAtomicNum());
	gchar* Id = g_strdup_printf("a%d", atom->GetIdx());
	SetId(Id);
	g_free(Id);
//	m_Valence = atom->GetValence();
	m_HPos = true;
	m_ascent = 0;
	m_CHeight = 0.;
}

void gcpAtom::SetZ(int Z)
{
	Atom::SetZ(Z);
	m_Valence = Element::GetElement(m_Z)->GetDefaultValence();
	// compute valence electrons number: FIXME: should be moved elsewhere
	if (m_Valence > 0)
	{
		if (Z < 3) m_nve = Z;
		else if (Z < 11) m_nve = Z - 2;
		else if (Z < 19) m_nve = Z - 10;
		else if (Z < 37) m_nve = Z - 18;
		else if (Z < 55) m_nve = Z - 36;
		else if (Z < 87) m_nve = Z - 54;
		else m_nve = Z - 86;
		Update();
		m_HPos = GetBestSide();
	}
	else m_nH = 0;
}

int gcpAtom::GetTotalBondsNumber()
{
	std::map<Atom*, Bond*>::iterator i;
	int n = 0;
	for (i = m_Bonds.begin(); i != m_Bonds.end(); i++)
		n += (*i).second->GetOrder();
	return n;
}

void gcpAtom::AddBond(Bond* pBond)
{
	Atom::AddBond(pBond);
	Update();
}

void gcpAtom::RemoveBond(Bond* pBond)
{
	Atom::RemoveBond(pBond);
	Update();
}

bool gcpAtom::GetBestSide()
{
	if (m_Bonds.size() == 0) return Element::BestSide(m_Z);
	std::map<Atom*, Bond*>::iterator i;
	double sum = 0.0;
	for (i = m_Bonds.begin(); i != m_Bonds.end(); i++)
		sum -= cos(((gcpBond*)(*i).second)->GetAngle2DRad(this));
	if (fabs(sum) > 0.1) return (sum >= 0.0);
	else return Element::BestSide(m_Z);
}

void gcpAtom::Update()
{
	if (m_Valence <= 0) return;//FIXME:stupid
	if (m_ChargeAuto)
	{
		m_Charge = 0;
		m_ChargeAuto = false;
	}
	int nb, nlp = (m_nve - m_Valence) / 2;
	if ((m_Charge > 0) && (nlp > 0)) nlp --;
	nb = m_nve - 2 * nlp - m_Charge;
	if (nb + nlp > 4) nb -= 2; //octet rule
	m_nH = nb - GetTotalBondsNumber();
	if ((! m_Charge) && (m_nH == -1))
	{
		m_Charge = (nlp)? 1: -1;
		m_ChargeAuto = true;
	}
	if (m_nH < 0) m_nH = 0;
	m_HPos = GetBestSide();
}
	
bool gcpAtom::IsInCycle(gcpCycle* pCycle)
{
	map<Atom*, Bond*>::iterator i;
	for (i = m_Bonds.begin(); i != m_Bonds.end(); i++)
		if (((gcpBond*)(*i).second)->IsInCycle(pCycle)) return true;
	return false;
}

void gcpAtom::Add(GtkWidget* w)
{
	if (!w) return;
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	gcpView* pView = pData->View;
	if (m_ascent <= 0)
	{
		PangoContext* pc = pData->View->GetPangoContext();
		PangoLayout *pl = pango_layout_new(pc);
		pango_layout_set_text(pl, "l", 1);
		PangoLayoutIter* iter = pango_layout_get_iter(pl);
		m_ascent = pango_layout_iter_get_baseline(iter) / PANGO_SCALE;
		pango_layout_iter_free(iter);
		g_object_unref(pl);
	}
	if (m_CHeight == 0.)
	{
		PangoContext* pc = pData->View->GetPangoContext();
		PangoLayout *pl = pango_layout_new(pc);
		pango_layout_set_text(pl, "C", 1);
		PangoRectangle rect;
		pango_layout_get_extents(pl, &rect, NULL);
		m_CHeight =  double(rect.height / PANGO_SCALE) / 2.0;
		g_object_unref(G_OBJECT(pl));
	}
	const gchar* FontName = pView->GetSmallFontName();
	GtkTextTag* tag = gtk_text_tag_table_lookup(TextTagTable, FontName);
	if (!tag)
	{
		tag = gtk_text_tag_new(FontName);
		PangoFontDescription* pfd = pView->GetPangoSmallFontDesc();
		g_object_set(G_OBJECT(tag),
							"family", pango_font_description_get_family(pfd),
							"size", pango_font_description_get_size(pfd),
							NULL);
		gtk_text_tag_table_add(TextTagTable, tag);
		g_object_unref((GObject*)tag);
	}
	FontName = pView->GetFontName();
	tag = gtk_text_tag_table_lookup(TextTagTable, FontName);
	if (!tag)
	{
		tag = gtk_text_tag_new(FontName);
		PangoFontDescription* pfd = pView->GetPangoFontDesc();
		g_object_set(G_OBJECT(tag),
							"family", pango_font_description_get_family(pfd),
							"size", pango_font_description_get_size(pfd),
							NULL);
		gtk_text_tag_table_add(TextTagTable, tag);
		g_object_unref((GObject*)tag);
	}
	double x, y;
	m_width =  m_height = 2.0 * pData->Padding;
	GetCoords(&x, &y);
	x *= pData->ZoomFactor;
	y *= pData->ZoomFactor;
	double dFontHeight = pData->View->GetFontHeight();
	char * szFontName = pData->View->GetFontName();
	GnomeCanvasItem* item;
	GnomeCanvasGroup* group;
	double width, height;
	int index;
	char *hs;
	PangoContext* pc = pData->View->GetPangoContext();
	group = GNOME_CANVAS_GROUP(gnome_canvas_item_new(pData->Group, gnome_canvas_group_ext_get_type(), NULL));
	g_signal_connect(G_OBJECT(group), "event", G_CALLBACK(on_event), w);
	g_object_set_data(G_OBJECT(group), "object", this);
	if ((GetZ() != 6) || (GetBondsNumber() == 0))
	{
		GtkTextIter start, end;
		int sw, hw = 0, nw = 0;
		gtk_text_buffer_get_bounds(m_buf, &start, &end);
		gtk_text_buffer_delete(m_buf, &start, &end);
		const gchar* symbol = GetSymbol();
		gtk_text_buffer_insert_with_tags_by_name(m_buf, &start, symbol, strlen(symbol),pView->GetFontName(), NULL);
		PangoLayout *pl = pango_layout_new(pc);
		pango_layout_set_text(pl, symbol, strlen(symbol));
		PangoRectangle rect;
		pango_layout_get_extents(pl, NULL, &rect);
		m_width = sw = rect.width / PANGO_SCALE;
		m_height = rect.height / PANGO_SCALE;
		height = dFontHeight;
		int n = GetAttachedHydrogens();
		if (n > 0)
		{
			pango_layout_set_text(pl, "H", 1);
			PangoRectangle rect;
			pango_layout_get_extents(pl, NULL, &rect);
			hw = rect.width / PANGO_SCALE;
			index = (m_HPos)? strlen(symbol): 0;
			gtk_text_buffer_get_iter_at_offset(m_buf, &start, index);
			gtk_text_buffer_insert_with_tags_by_name(m_buf, &start, "H", 1,pView->GetFontName(), NULL);
		}
		g_object_unref(pl);
		if (n > 1)
		{
			pango_context_set_font_description(pc, pData->View->GetPangoSmallFontDesc());
			pl = pango_layout_new(pc);
			hs = g_strdup_printf("%d", n);
			pango_layout_set_text(pl, hs, strlen(hs));
			pango_layout_get_extents(pl, NULL, &rect);
			nw = rect.width / PANGO_SCALE;
			g_object_unref(pl);
			pango_context_set_font_description(pc, pData->View->GetPangoFontDesc());
			gtk_text_buffer_get_iter_at_offset(m_buf, &start, index + 1);
			gtk_text_buffer_insert_with_tags_by_name(m_buf, &start, hs, strlen(hs),pView->GetSmallFontName(), "subscript", NULL);
			gtk_text_buffer_get_iter_at_offset(m_buf, &start, index + 1);
			gtk_text_buffer_get_iter_at_offset(m_buf, &end, index + 1 + strlen (hs));
			gtk_text_buffer_remove_tag_by_name(m_buf, pView->GetFontName(), &start, &end);
			height += 2;
		}
		width = sw + hw + nw;
		m_lbearing = (m_HPos)? sw / 2: sw / 2 + hw + nw;

		item = gnome_canvas_item_new(
							group,
							gnome_canvas_rect_ext_get_type(),
							"x1", x - m_lbearing - pData->Padding,
							"y1", y  - m_ascent + m_CHeight - pData->Padding,
							"x2", x - m_lbearing + width + pData->Padding,
							"y2", y  - m_ascent + m_CHeight + m_height + pData->Padding,
							"fill_color", (pData->IsSelected(this))? SelectColor: "white",
							NULL);
		g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), w);
		g_object_set_data(G_OBJECT(item), "object", this);
		
		item = gnome_canvas_item_new(
							group,
							gnome_canvas_rich_text_ext_get_type(),
							"x", x - m_lbearing,
							"y", y - m_ascent + m_CHeight,
							"width", width,
							"height", m_height,
							"grow_height", false,
							"editable", false,
							"cursor_visible", false,
							NULL);
		gnome_canvas_rich_text_ext_set_buffer(GNOME_CANVAS_RICH_TEXT_EXT(item), m_buf);
		g_object_set_data(G_OBJECT(group), "symbol", item);
		gnome_canvas_rich_text_ext_set_buffer(GNOME_CANVAS_RICH_TEXT_EXT(item), m_buf);
		g_object_set_data(G_OBJECT(item), "object", this);
		g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), w);
	}
	else
	{
		item = gnome_canvas_item_new(
								group,
								gnome_canvas_rect_ext_get_type(),
								"x1", x - 3,
								"y1", y - 3,
								"x2", x + 3,
								"y2", y + 3,
								"fill_color",  (pData->IsSelected(this))? SelectColor: "white",
								NULL);
		gnome_canvas_request_redraw((GnomeCanvas*)w, (int)x-3, (int)y-3, (int)x+3, (int)y+3);
		gnome_canvas_item_lower_to_bottom(GNOME_CANVAS_ITEM(group));
		gnome_canvas_item_raise(GNOME_CANVAS_ITEM(group), 1);
		g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), w);
		g_object_set_data(G_OBJECT(item), "object", this);
	}
	pData->Items[this] = group;
	m_width /= pData->ZoomFactor;
	m_height /= pData->ZoomFactor;
	char charge = GetCharge();
	if (charge)
	{
		int align = GetChargePosition(x, y);
		x *= pData->ZoomFactor;
		y *= pData->ZoomFactor;
		switch (align)
		{
			case -2:
				y += 6.0;//FIXME: change 6.0 end other constants to variables
				break;
			case -1:
				x -= 6.0;
				break;
			case 1:
				x += 6.0;
				break;
			case 2:
				y -= 6.0;
				break;
		}
		GnomeCanvasGroup* chargegroup = GNOME_CANVAS_GROUP(gnome_canvas_item_new(group, gnome_canvas_group_ext_get_type(), NULL));
		g_object_set_data ((GObject*)group, "charge", chargegroup);
		gnome_canvas_item_new(
						chargegroup,
						gnome_canvas_ellipse_ext_get_type(),
						"width_units", 1.0,
						"outline_color", (pData->IsSelected(this))? SelectColor: Color,
						"x1", x - 4.0,
						"x2", x + 4.0,
						"y1", y - 4.0,
						"y2", y + 4.0,
						NULL);
		GnomeCanvasPoints *points = gnome_canvas_points_new (2);
		points->coords[0] = x - 2.5;
		points->coords[1] = points->coords[3] = y;
		points->coords[2] = x + 2.5;
		gnome_canvas_item_new(
						chargegroup,
						gnome_canvas_line_ext_get_type(),
						"points", points,
						"fill_color", (pData->IsSelected(this))? SelectColor: Color,
						"width_units", 1.0,
						NULL);
		if (charge > 0)
		{	
			points->coords[0] = points->coords[2] = x;
			points->coords[1] = y - 2.5;
			points->coords[3] = y + 2.5;
			gnome_canvas_item_new(
							chargegroup,
							gnome_canvas_line_ext_get_type(),
							"points", points,
							"fill_color", (pData->IsSelected(this))? SelectColor: Color,
							"width_units", 1.0,
							NULL);
		}
	}
}

void gcpAtom::Update(GtkWidget* w)
{
	if (!w) return;
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
}//FIXME: write this function

int gcpAtom::GetChargePosition(double& x, double& y)
{
	list<double> AngleList;
	list<double>::iterator n;
	double angle;
	unsigned char def;
	if (((GetZ() != 6 || m_Bonds.size() == 0)) && m_nH)
	{
		if (m_HPos)
		{
			def = 0xB6;
			AngleList.push_front(225.0);
			AngleList.push_front(135.0);
		}
		else
		{
			def = 0x6D;
			AngleList.push_front(315.0);
			AngleList.push_front(45.0);
		}
	}
	else def = 0xff;
	map<Atom*, Bond*>::iterator i = m_Bonds.begin();
	while (i != m_Bonds.end())
	{
		n = AngleList.begin();
		angle = ((gcpBond*)(*i).second)->GetAngle2D(this) + 180.0;
		while ((n != AngleList.end()) && (*n < angle)) n++;
		AngleList.insert(n, angle);
		i++;
		if ((def & CHARGE_NE) && (angle >= 180.0) && (angle <= 270.0)) def -= CHARGE_NE;
		if ((def & CHARGE_NW) && (((angle >= 270.0) && (angle <= 360.0)) || (fabs(angle) < 0.1))) def -= CHARGE_NW;
		if ((def & CHARGE_N) && (angle >= 225.0) && (angle <= 315.0)) def -= CHARGE_N;
		if ((def & CHARGE_SE) && (angle >= 90.0) && (angle <= 180.0)) def -= CHARGE_SE;
		if ((def & CHARGE_SW) && (((angle >= 0.0) && (angle <= 90.0)) || (fabs(angle - 360.0) < 0.1))) def -= CHARGE_SW;
		if ((def & CHARGE_S) && (angle >= 45.0) && (angle <= 135.0)) def -= CHARGE_S;
		if ((def & CHARGE_E) && ((angle <= 225.0) && (angle >= 135.0))) def -= CHARGE_E;
		if ((def & CHARGE_W) && (angle >= 315.0) || (angle <= 45.0)) def -= CHARGE_W;
	}
	if (def)
	{
		if (def & CHARGE_NE)
		{
			x = m_x + m_width / 2.0;
			y = m_y - m_height / 2.0;
			return 1;
		}
		if (def & CHARGE_NW)
		{
			x = m_x - m_width / 2.0;
			y = m_y - m_height / 2.0;
			return -1;
		}
		if (def & CHARGE_N)
		{
			x = m_x;
			y = m_y - m_height / 2.0;
			return 2;
		}
		if (def & CHARGE_SE)
		{
			x = m_x + m_width / 2.0;
			y = m_y + m_height / 2.0;
			return 1;
		}
		if (def & CHARGE_SW)
		{
			x = m_x - m_width / 2.0;
			y = m_y + m_height / 2.0;
			return -1;
		}
		if (def & CHARGE_S)
		{
			x = m_x;
			y = m_y + m_height / 2.0;
			return -2;
		}
		if (def & CHARGE_E)
		{
			x = m_x + m_width / 2.0;
			y = m_y;
			return 1;
		}
		if (def & CHARGE_W)
		{
			x = m_x - m_width / 2.0;
			y = m_y;
			return -1;
		}
	}
	AngleList.push_back((angle = AngleList.front()) + 360.0);
	double dir = 0.0, max = 0.0;
	//if we are there, there is at least two bonds
	for (n = AngleList.begin(), n++; n != AngleList.end(); n++)
	{
		if (*n - angle > max)
		{
			if (*n - angle - max > 0.1) x = (*n + angle) / 2;
			if (m_nH)
			{
				if (m_HPos && ((x > 225.0) || (x < 135.0))) dir = x;
				else if (m_HPos && (x > 45.0) && (x < 315.0)) dir = x;
			}
			else dir = x;
			max = *n - angle;
		}
		angle = *n;
	}
	max = sqrt(square(m_width) + square(m_height)) / 2.0 + 24;//Could do beter, should replace 24 by something more intelligent
	x = m_x + max * cos(dir / 180.0 * M_PI);
	y = m_y - max * sin(dir / 180.0 * M_PI);
	return 0;
}

bool gcpAtom::LoadNode(xmlNodePtr)
{
	SetZ(GetZ());
	return true;
}

void gcpAtom::SetSelected(GtkWidget* w, int state)
{
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	GnomeCanvasGroup* group = pData->Items[this];
	gchar* color;
	
	switch (state)
	{	
		case SelStateUnselected: color = "white"; break;
		case SelStateSelected: color = SelectColor; break;
		case SelStateUpdating: color = AddColor; break;
		case SelStateErasing: color = DeleteColor; break;
	}
	GList* il = group->item_list;
	while (il)
	{
		if (!GNOME_IS_CANVAS_RICH_TEXT_EXT(il->data))
		{
			if (GNOME_IS_CANVAS_GROUP_EXT(il->data))
			{
				GList* il1 = ((GnomeCanvasGroup*)il->data)->item_list;
				while (il1)
				{
					if (GNOME_IS_CANVAS_LINE_EXT(il1->data))
						g_object_set(G_OBJECT(il1->data), "fill_color", color, NULL);
					else g_object_set(G_OBJECT(il1->data), "outline_color", color, NULL);
					il1 = il1->next;
				}
			}
			else g_object_set(G_OBJECT(il->data), "fill_color", color, NULL);
		}
		il = il->next;
	}
}

bool gcpAtom::AcceptNewBonds(int nb)
{
	return Element::GetMaxBonds(m_Z) >= (GetTotalBondsNumber() + nb);
}

void gcpAtom::AddToMolecule(gcpMolecule* Mol)
{
	Mol->AddAtom(this);
}
