// -*- C++ -*-

/* 
 * GChemPaint
 * arrowtool.cc 
 *
 * Copyright (C) 2001-2003
 *
 * Developed by Jean Bréfort <jean.brefort@ac-dijon.fr>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "config.h"
#include "arrowtool.h"
#include "settings.h"
#include "libgcpcanvas/gcp-canvas-line.h"
#include "libgcpcanvas/gcp-canvas-group.h"
#include "globals.h"
#include "document.h"
#include <math.h>

static Object* CreateReactionArrow()
{
	return new gcpReactionArrow(NULL);
}

static Object* CreateMesomeryArrow()
{
	return new gcpMesomeryArrow(NULL);
}

gcpArrowTool sat(gcpSimpleArrow), dat(gcpReversibleArrow), mat(gcpDoubleHeadedArrow);

gcpArrowTool::gcpArrowTool(gcpArrowType ArrowType): gcpTool(gcpToolId(SimpleArrowId + ArrowType))
{
	points = gnome_canvas_points_new(2);
	m_ArrowType = ArrowType;
	switch (ArrowType)
	{
		case gcpSimpleArrow:
			Object::AddType("reaction-arrow", CreateReactionArrow, ReactionArrowType); break;
		case gcpDoubleHeadedArrow:
			Object::AddType("mesomery-arrow", CreateMesomeryArrow, MesomeryArrowType); break;
	}
}

gcpArrowTool::~gcpArrowTool()
{
	gnome_canvas_points_free(points);
}	

bool gcpArrowTool::OnClicked()
{
	if (m_pObject) return false;
	double dx, dy, x;
	m_y1 = m_y0;
	m_x1 = m_x0 + m_pData->ArrowLength;
	switch(m_ArrowType)
	{
		case gcpSimpleArrow:
			points->coords[0] = m_x0;
			points->coords[1] = m_y0;
			points->coords[2] = m_x1;
			points->coords[3] = m_y0;
			m_pItem = gnome_canvas_item_new(
										m_pGroup,
										gnome_canvas_line_ext_get_type(),
										"points", points,
										"fill_color", AddColor,
										"width_units", m_pData->BondWidth,
										"last_arrowhead", true,
										"arrow_shape_a", m_pData->ArrowHeadA,
										"arrow_shape_b", m_pData->ArrowHeadB,
										"arrow_shape_c", m_pData->ArrowHeadC,
										"last_arrowhead_style", (unsigned char)ARROW_HEAD_BOTH,
										NULL);
			break;
		case gcpReversibleArrow:
			points->coords[0] = m_x0;
			points->coords[1] = points->coords[3] = m_y0 - m_pData->ArrowDist / 2;
			points->coords[2] = m_x1;
			m_pItem = gnome_canvas_item_new(m_pGroup, gnome_canvas_group_ext_get_type(), NULL);
			gnome_canvas_item_new(
								GNOME_CANVAS_GROUP(m_pItem),
								gnome_canvas_line_ext_get_type(),
								"points", points,
								"fill_color", AddColor,
								"width_units", m_pData->BondWidth,
								"last_arrowhead", true,
								"arrow_shape_a", m_pData->ArrowHeadA,
								"arrow_shape_b", m_pData->ArrowHeadB,
								"arrow_shape_c", m_pData->ArrowHeadC,
								"last_arrowhead_style", (unsigned char)ARROW_HEAD_LEFT,
								NULL);
			points->coords[2] = m_x0;
			points->coords[1] = points->coords[3] = m_y0 + m_pData->ArrowDist / 2;
			points->coords[0] = m_x0 + m_pData->ArrowLength;
			gnome_canvas_item_new(
								GNOME_CANVAS_GROUP(m_pItem),
								gnome_canvas_line_ext_get_type(),
								"points", points,
								"fill_color", AddColor,
								"width_units", m_pData->BondWidth,
								"last_arrowhead", true,
								"arrow_shape_a", m_pData->ArrowHeadA,
								"arrow_shape_b", m_pData->ArrowHeadB,
								"arrow_shape_c", m_pData->ArrowHeadC,
								"last_arrowhead_style", (unsigned char)ARROW_HEAD_LEFT,
								NULL);
			break;
		case gcpDoubleHeadedArrow:
			points->coords[0] = m_x0;
			points->coords[1] = m_y0;
			points->coords[2] = m_x1;
			points->coords[3] = m_y0;
			m_pItem = gnome_canvas_item_new(
										m_pGroup,
										gnome_canvas_line_ext_get_type(),
										"points", points,
										"fill_color", AddColor,
										"width_units", m_pData->BondWidth,
										"first_arrowhead", true,
										"last_arrowhead", true,
										"arrow_shape_a", m_pData->ArrowHeadA,
										"arrow_shape_b", m_pData->ArrowHeadB,
										"arrow_shape_c", m_pData->ArrowHeadC,
										"first_arrowhead_style", (unsigned char)ARROW_HEAD_BOTH,
										"last_arrowhead_style", (unsigned char)ARROW_HEAD_BOTH,
										NULL);
			break;
	}
	return true;
}

void gcpArrowTool::OnDrag()
{
	double x1, y1, x2, y2;
	if (m_pItem)
	{
		gnome_canvas_item_get_bounds(GNOME_CANVAS_ITEM(m_pItem), &x1, &y1, &x2, &y2);
		gtk_object_destroy(GTK_OBJECT(GNOME_CANVAS_ITEM(m_pItem)));
		gnome_canvas_request_redraw(GNOME_CANVAS(m_pWidget), (int)x1, (int)y1, (int)x2, (int)y2);
		m_pItem = NULL;
	}
	double dAngle;
	m_x -= m_x0;
	m_y -= m_y0;
	if (m_x == 0)
	{
		if (m_y == 0) return;
		dAngle = (m_y < 0) ? 90 : 270;
	}
	else
	{
		dAngle = atan(-m_y/m_x) * 180 / M_PI;
		if (!(m_nState & GDK_CONTROL_MASK)) dAngle = rint(dAngle / 5) * 5;
		if (m_x < 0) dAngle += 180;
	}
	m_dAngle = dAngle * M_PI / 180;
	double d = (m_nState & GDK_SHIFT_MASK)? sqrt(square(m_x) + square(m_y)): m_pData->ArrowLength;
	char tmp[32];
	if (dAngle < 0) dAngle += 360;
	snprintf(tmp, sizeof(tmp) - 1, _("Orientation: %g"), dAngle);
	set_status_text(tmp);
	m_x1 = m_x0 + d * cos(m_dAngle);
	m_y1 = m_y0 - d * sin(m_dAngle);
	switch(m_ArrowType)
	{
		case gcpSimpleArrow:
			points->coords[2] = m_x1;
			points->coords[3] = m_y1;
			m_pItem = gnome_canvas_item_new(
										m_pGroup,
										gnome_canvas_line_ext_get_type(),
										"points", points,
										"fill_color", AddColor,
										"width_units", m_pData->BondWidth,
										"last_arrowhead", true,
										"arrow_shape_a", m_pData->ArrowHeadA,
										"arrow_shape_b", m_pData->ArrowHeadB,
										"arrow_shape_c", m_pData->ArrowHeadC,
										"last_arrowhead_style", (unsigned char)ARROW_HEAD_BOTH,
										NULL);
			break;
		case gcpReversibleArrow:
			points->coords[0] = m_x0 - m_pData->ArrowDist / 2 * sin(m_dAngle);
			points->coords[1] = m_y0 - m_pData->ArrowDist / 2 * cos(m_dAngle);
			points->coords[2] = m_x1 - m_pData->ArrowDist / 2 * sin(m_dAngle);
			points->coords[3] = m_y1 - m_pData->ArrowDist / 2 * cos(m_dAngle);
			m_pItem = gnome_canvas_item_new(m_pGroup, gnome_canvas_group_ext_get_type(), NULL);
			gnome_canvas_item_new(
								GNOME_CANVAS_GROUP(m_pItem),
								gnome_canvas_line_ext_get_type(),
								"points", points,
								"fill_color", AddColor,
								"width_units", m_pData->BondWidth,
								"last_arrowhead", true,
								"arrow_shape_a", m_pData->ArrowHeadA,
								"arrow_shape_b", m_pData->ArrowHeadB,
								"arrow_shape_c", m_pData->ArrowHeadC,
								"last_arrowhead_style", (unsigned char)ARROW_HEAD_LEFT,
								NULL);
			points->coords[2] = m_x0 + m_pData->ArrowDist / 2 * sin(m_dAngle);
			points->coords[3] = m_y0 + m_pData->ArrowDist / 2 * cos(m_dAngle);
			points->coords[0] = m_x1 + m_pData->ArrowDist / 2 * sin(m_dAngle);
			points->coords[1] = m_y1 + m_pData->ArrowDist / 2 * cos(m_dAngle);
			gnome_canvas_item_new(
								GNOME_CANVAS_GROUP(m_pItem),
								gnome_canvas_line_ext_get_type(),
								"points", points,
								"fill_color", AddColor,
								"width_units", m_pData->BondWidth,
								"last_arrowhead", true,
								"arrow_shape_a", m_pData->ArrowHeadA,
								"arrow_shape_b", m_pData->ArrowHeadB,
								"arrow_shape_c", m_pData->ArrowHeadC,
								"last_arrowhead_style", (unsigned char)ARROW_HEAD_LEFT,
								NULL);
			break;
		case gcpDoubleHeadedArrow:
			points->coords[2] = m_x1;
			points->coords[3] = m_y1;
			m_pItem = gnome_canvas_item_new(
										m_pGroup,
										gnome_canvas_line_ext_get_type(),
										"points", points,
										"fill_color", AddColor,
										"width_units", m_pData->BondWidth,
										"first_arrowhead", true,
										"last_arrowhead", true,
										"arrow_shape_a", m_pData->ArrowHeadA,
										"arrow_shape_b", m_pData->ArrowHeadB,
										"arrow_shape_c", m_pData->ArrowHeadC,
										"first_arrowhead_style", (unsigned char)ARROW_HEAD_BOTH,
										"last_arrowhead_style", (unsigned char)ARROW_HEAD_BOTH,
										NULL);
			break;
	}
}

void gcpArrowTool::OnRelease()
{
	double x1, y1, x2, y2;
	if (m_pItem)
	{
		gnome_canvas_item_get_bounds(GNOME_CANVAS_ITEM(m_pItem), &x1, &y1, &x2, &y2);
		gtk_object_destroy(GTK_OBJECT(GNOME_CANVAS_ITEM(m_pItem)));
		gnome_canvas_request_redraw(GNOME_CANVAS(m_pWidget), (int)x1, (int)y1, (int)x2, (int)y2);
		m_pItem = NULL;
	}
	else return;
	set_status_text("");
	gcpDocument* pDoc = m_pView->GetDoc();
	gcpArrow* a;
	if (m_ArrowType == gcpDoubleHeadedArrow)
	{
		a = new gcpMesomeryArrow(NULL);
		a->SetCoords(m_x0 / m_dZoomFactor, m_y0 / m_dZoomFactor, m_x1 / m_dZoomFactor, m_y1 / m_dZoomFactor);
		pDoc->AddObject(a);
	}
	else
	{
		a = new gcpReactionArrow(NULL, m_ArrowType);
		a->SetCoords(m_x0 / m_dZoomFactor, m_y0 / m_dZoomFactor, m_x1 / m_dZoomFactor, m_y1 / m_dZoomFactor);
		pDoc->AddObject(a);
	}
	pDoc->FinishOperation();
}
