/* board.cc -- game board for falling blocks game.  -*- C++ -*-
   Copyright (C) 2011-2012 Gerardo Ballabio

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>. */

#include <cstdlib>
#include <vector>
#include <gtkmm.h>
#include "board.h"
#include "grid.h"
#include "group.h"

bool game_board::fits(int b, int s, coords c) const
{
  std::vector<coords> bricks = get_bricks(blocks[b].shape[s]);
  for (std::vector<coords>::const_iterator i=bricks.begin();
       i!=bricks.end(); ++i)
    {
      coords n = c + *i;
      if (!cells.is_in(n) || cells.get(n))
	return false;
    }

  return true;
}

bool game_board::fits_b(int b, int s, coords c) const
{
  std::vector<coords> bricks = get_bricks(blocks[b].blocker[s]);
  for (std::vector<coords>::const_iterator i=bricks.begin();
       i!=bricks.end(); ++i)
    {
      coords n = c + *i;
      if (!cells.is_in(n) || cells.get(n))
	return false;
    }

  return true;
}

coords game_board::move_in(int b, int s, coords c) const
{
  // move down if necessary
  while (blocks[b].shape[s].get_y0() + c.y < cells.get_y0())
    c = g->down(c);

  // move right if necessary
  while (blocks[b].shape[s].get_x0() + c.x < cells.get_x0())
    c = g->right(c);

  // move left if necessary
  while (blocks[b].shape[s].get_x0() + blocks[b].shape[s].get_cols() + c.x >
	 cells.get_x0() + cells.get_cols())
    c = g->left(c);

  // move back if necessary
  while (blocks[b].shape[s].get_z0() + c.z < cells.get_z0())
    c = g->back(c);

  // move front if necessary
  while (blocks[b].shape[s].get_z0() + blocks[b].shape[s].get_layers() + c.z >
	 cells.get_z0() + cells.get_layers())
    c = g->front(c);

  return c;
}

void game_board::first_block()
{
  // initialize random numbers generator
  Glib::TimeVal timeval;
  timeval.assign_current_time();
  std::srand(static_cast<unsigned>(timeval.as_double() * G_USEC_PER_SEC));

  nsize = bmax;
  // choose block
  nblock = rand() % series[nsize];
  for (int i=0; i<nsize; ++i)
    nblock += series[i];
  // choose orientation
  nshape = rand() % blocks[nblock].shape.size();

  // move to active block
  next_block();
}

void game_board::land_block()
{
  // dump block onto cells grid
  std::vector<coords> bricks = get_bricks(blocks[iblock].shape[ishape]);
  for (std::vector<coords>::const_iterator i=bricks.begin();
       i!=bricks.end(); ++i)
    {
      coords n = pos + *i;
      cells.set(n, iblock);
    }
  score += score_block();

  // search for filled lines
  filled = g->check_lines(cells, empty);
}

void game_board::next_level()
{
  // increment level
  ++level;
  switch (mode1)
    {
    case 0:
      // increase falling block speed
      speed = level;
      break;
    case 1:
      // increase maximum block size
      ++bmax;
      next_blockset();
      switch (mode2)
	{
	case 0: if (bmin > 1) --bmin; break;
	case 1: break;
	case 2: ++bmin; break;
	}
      break;
    }
}

void game_board::next_blockset()
{
  // extract last blockset
  int first = blocks.size() - ((series.size() > 0) ? series.back() : 0);
  std::vector<block> v(blocks.begin() + first, blocks.end());

  // reduce it to a maximum size in order to speed up generation
  int max = g->blockset_max();
  if (v.size() > max)
    v = random_blocks(v, max);

  // generate and append new blockset
  v = g->augment_blocks(v, cells.get_x0(), cells.get_z0(),
			cells.get_cols(), cells.get_layers());
  blocks.insert(blocks.end(), v.begin(), v.end());
  series.push_back(v.size());
}

int game_board::score_block() const
{
  return (speed + bmax * 5) * (bmax + bmin - 5) / (1 + empty);
}

int game_board::score_lines(int lines) const
{
  return score_block() * 20 * (1 << lines);
}

grid<bool> game_board::get_block() const
{
  grid<bool> g = blocks[iblock].shape[ishape];
  g.move(pos);
  return g;
}

grid<bool> game_board::get_shadow()
{
  // move block down until it lands
  coords c = pos;
  while (fits(iblock, ishape, g->down(c)))
    c = g->down(c);

  grid<bool> g = blocks[iblock].shape[ishape];
  g.move(c);
  return g;
}

void game_board::start_game(const group *gg, int width, int depth,
			    int m1, int m2, int s, int min, int max, int e)
{
  if (game)
    return;

  // delete previous group, if any
  if (g)
    delete g;

  // set spatial group
  g = gg;
  // generate game board
  cells = g->make_board(width, depth);
  // generate block set
  std::vector<block> v;
  blocks.resize(0);
  series.resize(0);
  for (int i=0; i<=max; ++i)
    next_blockset();
  if (min > max)
    min = max;
  bmin = min;
  bmax = nsize = max;
  empty = e;
  // start game
  mode1 = m1;
  mode2 = m2;
  speed = s;
  level = (mode1 == 0) ? speed : 1;
  lines = score = 0;
  filled.resize(0);
  game = true;

  first_block();
}

bool game_board::rotate(int n)
{
  int rotations = blocks[iblock].rotate.size() / blocks[iblock].shape.size();
  if (n >= rotations)
    return false; // do nothing

  int nshape = blocks[iblock].rotate[ishape * rotations + n];
  coords m = move_in(iblock, nshape, pos);
  if (!fits(iblock, nshape, m))
    return false; // reject move

  // accept move
  pos = m;
  ishape = nshape;
  return true;
}

bool game_board::reflect()
{
  int nshape = blocks[iblock].reflect[ishape];
  coords m = move_in(iblock, nshape, pos);
  if (!fits(iblock, nshape, m))
    return false; // reject move

  // accept move
  pos = m;
  ishape = nshape;
  return true;
}

bool game_board::left()
{
  coords c = g->left(pos);
  if (!fits(iblock, ishape, c))
    return false;

  pos = c;
  return true;
}

bool game_board::right()
{
  coords c = g->right(pos);
  if (!fits(iblock, ishape, c))
    return false;

  pos = c;
  return true;
}

bool game_board::front()
{
  coords c = g->front(pos);
  if (!fits(iblock, ishape, c))
    return false;

  pos = c;
  return true;
}

bool game_board::back()
{
  coords c = g->back(pos);
  if (!fits(iblock, ishape, c))
    return false;

  pos = c;
  return true;
}

bool game_board::down()
{
  coords c = g->down(pos);
  if (fits(iblock, ishape, c) && fits_b(iblock, ishape, pos))
    {
      // move block down and continue
      pos = c;
      return true;
    }
  else
    {
      land_block();
      return false;
    }
}

bool game_board::drop()
{
  // move block down until it lands
  while (down())
    ;

  return true;
}

void game_board::color_lines(int c)
{
  // remove them
  for (std::vector<int>::const_iterator i=filled.begin();
       i!=filled.end(); ++i)
    g->color_line(cells, *i, c);
}

void game_board::remove_lines()
{
  // remove filled lines
  for (std::vector<int>::const_iterator i=filled.begin();
       i!=filled.end(); ++i)
    g->remove_line(cells, *i);

  // update lines count
  lines += filled.size();
  score += score_lines(filled.size());
  const int level_lines = 10;
  if (lines / level_lines > level)
    next_level();

  // choose new block
  next_block();
}

void game_board::next_block()
{
  // new active block
  iblock = nblock;
  ishape = nshape;
  bsize = nsize;
  // set initial position
  pos = make_coords();
  if (iblock != 0)
    pos = move_in(iblock, ishape, pos);
  // if it doesn't fit, game is over
  if (!fits(iblock, ishape, pos))
    game = false;

  // next block
  // cycle over block sizes (not in order)
  int plus = (bmax - bmin + 1) / 2, minus = plus * 2 + 1;
  nsize += plus;
  if (nsize > bmax)
    nsize -= minus;
  if (nsize < bmin)
    nsize += (plus > 0) ? plus : 1;
  // choose block
  nblock = rand() % series[nsize];
  for (int i=0; i<nsize; ++i)
    nblock += series[i];
  // choose orientation
  nshape = rand() % blocks[nblock].shape.size();
}

void game_board::stop()
{
  if (!game)
    return;

  game = false;
}
