/*  ga_service_ga_pattern.c */
/*  Copyright 2004-2006 Oswaldo Morizaki Hirakata */

/* 	This file is part of ga-nn-ag-2.

    ga-nn-ag is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    ga-nn-ag is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with ga-nn-ag; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "my_header.h"
#include "aux_prot.h"

/* 
Functions runs a genetic algorithm for patterns,
and returns pat_index.

The client index should be a copy.

- Insert errors of client_index->clients into pat_index 
- Runs the genetic algorithm for pat_index, patterns ratio aren't keep

pat_index is only required to get total num_stable and num_pattern, so it could
be called without locking it

*/

void 
* ga_service_ga_pattern(struct ga_service_client_index * index,
									  					struct ga_server_config * conf,
									  					struct ga_pat_index * pat_index,
									  					struct ga_pat_index * result_index)
{
int k,l,m,n;
int counter;
int num_cat;
int num_pat;
int num_stable;
int num_non_stable;
int num_thread;

float * fitness;

struct ga_pat_index * base_index = NULL;

struct ga_pat_ga_cat_thread_param * cat_param;

pthread_t *tid;
pthread_mutex_t cat_mutex = PTHREAD_MUTEX_INITIALIZER;
pthread_cond_t cat_cond = PTHREAD_COND_INITIALIZER;

/* Check if result index is NULL */
if (result_index != NULL)
{
	syslog(LOG_ERR,"Error, result_index not NULL in ga_service_ga_pattern, posible memory leak");
}

/* Memory reserve for base_index and result_index */
if (!(base_index = (struct ga_pat_index *)calloc(1,sizeof(struct ga_pat_index))))
{
	syslog(LOG_CRIT,"Error calloc base_index in ga_service_ga_pattern: %s",strerror(errno));
	return(NULL);
}
if (!(result_index = (struct ga_pat_index *)calloc(1,sizeof(struct ga_pat_index))))
{
	syslog(LOG_CRIT,"Error calloc result_index in ga_service_ga_pattern: %s",strerror(errno));
	free(base_index);
	return(NULL);
}

/* Memory reserve for categories in base_index and result index */
num_cat = pat_index->num_cat;
base_index->num_cat = pat_index->num_cat;
result_index->num_cat = base_index->num_cat;
if (!(base_index->cat = (struct ga_pat_cat **)calloc(
						base_index->num_cat, sizeof(struct ga_pat_cat *))))
{
	syslog(LOG_CRIT,"Error calloc base_index->cat in ga_service_ga_pattern: %s",strerror(errno));
	free(result_index);
	free(base_index);
	return(NULL);
}
if (!(result_index->cat = (struct ga_pat_cat **)calloc(
							result_index->num_cat, sizeof(struct ga_pat_cat *))))
{
	syslog(LOG_CRIT,"Error calloc result_index->cat in ga_service_ga_pattern: %s",strerror(errno));
	free(base_index->cat);
	free(result_index);
	free(base_index);
	return(NULL);
}
for (k=0; k< base_index->num_cat; k++)
{
	if (!(base_index->cat[k] = (struct ga_pat_cat *)calloc(1, sizeof(struct ga_pat_cat ))))
	{
		syslog(LOG_CRIT,"Error calloc base_index->cat[%d] in ga_service_ga_pattern: %s",k,strerror(errno));
		for (l=0 ; l< k; l++)
		{
			free(base_index->cat[l]);
		}
		free(base_index->cat);
		free(result_index->cat);
		free(result_index);
		free(base_index);
		return(NULL);
	}
	result_index->cat[k] = NULL;
}

/* Get fitness of patterns */
if (ga_pat_get_fitness(conf, index));
{
	syslog(LOG_CRIT,"Error ga_pat_get_fitness in ga_service_ga_pattern");
	for (l=0 ; l< base_index->num_cat; l++)
	{
		free(base_index->cat[l]);
	}
	free(base_index->cat);
	free(result_index->cat);
	free(result_index);
	free(base_index);
	return(NULL);
}

/* Redirect index->clients[k]->pat_index patterns to base_index */
num_pat = 0;
num_stable = 0;
for (k=0; k< base_index->num_cat; k++)
{
	/* Get number of patterns per category */
	counter = 0;
	for (l =0 ; l < index->num_client; l++)
	{
		counter += index->clients[l]->pat_index->cat[k]->num_pat; 
	}
	
	/* Get total num_pat and num_stable */
	num_pat += counter;
	num_stable += pat_index->cat[k]->num_stable;
	
	/* Memory reserve for patterns */
	if (!(base_index->cat[k]->pat = (struct ga_pattern **)calloc(
													counter, sizeof(struct ga_pattern *))))
	{
		syslog(LOG_CRIT,"Error calloc base_index->cat[%d]->pat in ga_service_ga_pattern: %s",k,strerror(errno));
		for (l=0 ; l< k; l++)
		{
			free(base_index->cat[l]->pat);
		}		
		for (l=0 ; l< base_index->num_cat; l++)
		{
			free(base_index->cat[l]);
		}
		free(base_index->cat);
		free(result_index->cat);
		free(result_index);
		free(base_index);
		return(NULL);
	}
	
	/* Redirect patterns */
	base_index->cat[k]->cat_fitness = 0;
	for (l=0; l< index->num_client; l++)
	{
		for (m = index->clients[l]->pat_index->cat[k]->num_pat-1; m+1 ; m--)
		{
			/* Pointers to the same structure */
			base_index->cat[k]->pat[n] = index->clients[l]->pat_index->cat[k]->pat[m];
		}
		base_index->cat[k]->cat_fitness += index->clients[l]->pat_index->cat[k]->cat_fitness;
	}
	base_index->cat[k]->group_mode = index->clients[l]->pat_index->cat[k]->group_mode;
	base_index->cat[k]->pat_cat = index->clients[l]->pat_index->cat[k]->pat_cat;
	base_index->cat[k]->num_pat = counter;
	base_index->cat[k]->num_stable = index->clients[l]->pat_index->cat[k]->num_stable;
}

/* Memory reserve for tid[] */
if (!(tid = (pthread_t *)malloc(num_cat*sizeof(pthread_t))))
{
	syslog(LOG_CRIT,"Error malloc tid in ga_service_ga_pattern: %s", strerror(errno));
	for (l=0 ; l< num_cat; l++)
	{
		free(base_index->cat[l]->pat);
		free(base_index->cat[l]);
	}
	free(base_index->cat);
	free(result_index->cat);
	free(result_index);
	free(base_index);
	return(NULL);
}

/* Memory reserve for cat_param */
if (!(cat_param = (struct ga_pat_ga_cat_thread_param *)malloc(num_cat*
											sizeof(struct ga_pat_ga_cat_thread_param))))
{
	syslog(LOG_CRIT,"Error malloc cat_param in ga_service_ga_pattern: %s", strerror(errno));
	for (l=0 ; l< num_cat; l++)
	{
		free(base_index->cat[l]->pat);
		free(base_index->cat[l]);
	}
	free(base_index->cat);
	free(result_index->cat);
	free(result_index);
	free(tid);
	free(base_index);
	return(NULL);
}

/* Get the number of non-stable patterns */
num_non_stable = num_pat - num_stable;

if (!(fitness = (float *)malloc(num_cat*sizeof(float))))
{
	syslog(LOG_CRIT,"Error malloc fitness in ga_service_ga_pattern: %s", strerror(errno));
	for (l=0 ; l< num_cat; l++)
	{
		free(base_index->cat[l]->pat);
		free(base_index->cat[l]);
	}
	free(tid);
	free(cat_param);
	free(base_index->cat);
	free(result_index->cat);
	free(result_index);
	free(base_index);
	return(NULL);
}

/* Get number of patterns for each category */
for (k=0; k< num_cat ; k++)
{
	fitness[k] = base_index->cat[k]->cat_fitness;
	cat_param[k].num_pat = base_index->cat[k]->num_stable;
}
for (k= 0 ; k < num_non_stable; k++)
{
	cat_param[va_roulette(num_cat, fitness)].num_pat += 1;
}

/* Set parameters for threads */
num_thread = 0;
for (k=0; k< num_cat ; k++)
{
	cat_param[k].cat = base_index->cat[k];
	cat_param[k].result_cat = result_index->cat[k];
	cat_param[k].num_thread = &num_thread;
	cat_param[k].cat_mutex = &cat_mutex;
	cat_param[k].cat_cond = &cat_cond;
	cat_param[k].ready = 0;
}

/* Start threads */
for (k=0; k< num_cat; k++)
{
	if (pthread_create(&(tid[k]), NULL, ga_pat_ga_cat_t, &(cat_param[k])) )
	{
		syslog(LOG_CRIT,"Error creating thread %d in ga_service_ga_pattern", k);
	}
}

/* Wait for ending threads */
for (k=0; k< num_cat; k++)
{
	if (pthread_mutex_lock(&cat_mutex))
	{
		syslog(LOG_CRIT,"Error pthread_mutex_lock in ga_service_ga_pattern: %s", strerror(errno));
		for (l=0 ; l< num_cat; l++)
		{
			free(base_index->cat[l]->pat);
			free(base_index->cat[l]);
		}
		free(base_index->cat);
		free(result_index->cat);
		free(result_index);
		free(tid);
		free(fitness);
		free(base_index);
		return(NULL);
	}
	while (num_thread == 0)
	{
		if (pthread_cond_wait(&cat_cond, &cat_mutex))
		{
			syslog(LOG_CRIT,"Error pthread_cond_wait in ga_service_ga_pattern: %s", strerror(errno));
			for (l=0 ; l< num_cat; l++)
			{
				free(base_index->cat[l]->pat);
				free(base_index->cat[l]);
			}
			free(base_index->cat);
			free(result_index->cat);
			free(result_index);
			free(tid);
			free(fitness);			
			free(base_index);
			return(NULL);
		}
	}
	for (l=0; l< num_cat; l++)
	{
		if (cat_param[l].ready)
		{
			if (pthread_join(tid[l], (void **) &(result_index->cat[k])) )
			{
				syslog(LOG_CRIT,"Error pthread_join %d for thread %d in ga_service_ga_pattern: %s",
								l, tid[l], strerror(errno));
								
				for (l=0 ; l< num_cat; l++)
				{
					free(base_index->cat[l]->pat);
					free(base_index->cat[l]);
				}
				free(base_index->cat);
				free(result_index->cat);
				free(result_index);
				free(tid);
				free(fitness);
				free(base_index);
				return(NULL);
			}
		}
	}
	if (pthread_mutex_unlock(&cat_mutex))
	{
		syslog(LOG_CRIT,"Error pthread_mutex_unlock in ga_service_ga_pattern: %s", strerror(errno));
		for (l=0 ; l< num_cat; l++)
		{
			free(base_index->cat[l]->pat);
			free(base_index->cat[l]);
		}
		free(base_index->cat);
		free(result_index->cat);
		free(result_index);
		free(tid);
		free(fitness);
		free(base_index);
		return(NULL);
	}
}

/* Free memory */
for (l=0 ; l< num_cat; l++)
{
	free(base_index->cat[l]->pat);
	free(base_index->cat[l]);
}
free(base_index->cat);
free(tid);
free(fitness);
free(base_index);

return (result_index);
}
