/*  ga_service_server.c */
/* 	Copyright 2004-2005 Oswaldo Morizaki Hirakata */

/* 	This file is part of ga-nn-ag-2.

    ga-nn-ag is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    ga-nn-ag is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with ga-nn-ag; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "my_header.h"
#include "aux_prot.h"

#define GA_CLIENT_NUM_PARAM 7

/* global exit_flag for sigterm */
int exit_flag = 0;

int ga_service_server(struct ga_server_config * conf)
{
int k,l;
int connfd;
int retry;
int local_population;
int status;
int listenfd;
int gen;

pid_t startup_client_pid;
pid_t temp_pid;
socklen_t addrlen;

int * num_local = NULL;
int * connptr = NULL;

char char_temp[BUFFSIZE];

char ** char_vector;

FILE * remote_config_file = NULL;

struct sigaction sig_term;

struct sockaddr_in server_addr;
struct sockaddr * client_addr = NULL;

struct io_connection remote_con;

struct ga_server_config remote_conf;

struct io_block * remote_block = NULL;
struct io_block * nn_block = NULL;

struct ga_client_info * client_info = NULL;

/* Set basic remote_con parameters */
bzero(&remote_con,sizeof(remote_con)); //set parameters to zero
remote_con.socket = 1; //connection via socket
sprintf(remote_con.port,"%d\0", conf->niche_port);

/********************* NICHE CODE ***********************/
/* ==================================================== */
/* Check if remote niches are set */
if (conf->num_niches > 0)
{
	/* Start remote slaves */
	/***********************/
	for (k=0; k< conf->num_niches; k++)
	{
		/* Calloc io_block */
		if (!(remote_block = va_calloc_io_block(0,remote_block)))
		{
			syslog(LOG_CRIT,"Error va_calloc_io_block() in ga_service_server()");
			return(-1);
		}
				
		/* Check if niches_config[k] == SAME */
		if (!strncmp(conf->niches_config[k],"SAME",4) )
		{
			strcpy(char_temp,"READFILE=SAME\0");
			remote_block = va_insert_io_block(-1,char_temp,remote_block);
		}
		/* Read new config from inet */
		else
		{
			sprintf(char_temp,"READINET=%s\0",conf->niches_config[k]);
			remote_block = va_insert_io_block(-1,char_temp,remote_block);
			
			if (!(remote_config_file = fopen(conf->niches_config[k],"r")))
			{
				syslog(LOG_CRIT,"Error couldn't open %s",conf->niches_config[k]);
				return(-2);
			}
			
			/* Append the content of file to the io_block */
			while (!feof(remote_config_file) )
			{
				fgets(char_temp,BUFFSIZE,remote_config_file);
				remote_block = va_insert_io_block(-1,char_temp,remote_block);
			}
		}
		
		/* Connect to remote servers */
		strcpy(remote_con.ip,conf->niches[k]);
		retry = 0;
		while (1)
		{
			if (va_io_connect(&remote_con) < 0)
			{
				retry += 1;
			}
			else
			{
				break;
			}
			
			if (retry == MAX_RETRY)
			{
				syslog(LOG_ERR,"Error va_io_connect(), couldn't reach %s",remote_con.ip);
				break;
			}
		}
		
		/* Now send the block to the hosts */
		remote_block->connfd = remote_con.connfd;
		
		retry = 0;
		while (1)
		{
			if (va_dwrite_io_block(remote_block) <0 )
			{
				retry += 1;
			}
			else
			{
				break;
			}
			
			if (retry >= MAX_RETRY)
			{
				syslog(LOG_ERR,"Error va_dwrite_io_block() in ga_service_server()");
				break;
			}
		}
		/* Free allocated memory */
		remote_block = va_free_io_block(remote_block);
		va_io_close(&remote_con);
	}
}

/********************* HOSTS CODE ***********************/
/* ==================================================== */
/* Check if remote hosts are used */
if (conf->num_hosts > 0)
{
	/* Start remote slaves */
	/***********************/
	for (k=0; k< conf->num_hosts; k++)
	{
		/* Calloc io_block */
		if (!(remote_block = va_calloc_io_block(0,remote_block)))
		{
			syslog(LOG_CRIT,"Error va_calloc_io_block() in ga_service_server()");
			return(-1);
		}
				
		/* Check if hosts_config[k] == SAME */
		if (!strncmp(conf->hosts_config[k],"SAME",4))
		{
			strcpy(char_temp,"READFILE=SAME\0");
			remote_block = va_insert_io_block(-1,char_temp,remote_block);
		}
		/* Read new config from inet */
		else
		{
			sprintf(char_temp,"READINET=%s\0",conf->hosts_config[k]);
			remote_block = va_insert_io_block(-1,char_temp,remote_block);
			
			if (!(remote_config_file = fopen(conf->hosts_config[k],"r")))
			{
				syslog(LOG_CRIT,"Error couldn't open %s",conf->hosts_config[k]);
				return(-2);
			}
			
			/* Append the content of file to the io_block */
			while (!feof(remote_config_file) )
			{
				fgets(char_temp,BUFFSIZE,remote_config_file);
				remote_block = va_insert_io_block(-1,char_temp,remote_block);
			}
		}
		
		/* Connect to remote servers */
		strcpy(remote_con.ip,conf->hosts[k]);
		retry = 0;
		while (1)
		{
			if (va_io_connect(&remote_con) < 0)
			{
				retry += 1;
			}
			else
			{
				break;
			}
			
			if (retry == MAX_RETRY)
			{
				syslog(LOG_ERR,"Error va_io_connect(), couldn't reach %s",remote_con.ip);
				break;
			}
		}
		
		/* Now send the block to the hosts */
		remote_block->connfd = remote_con.connfd;
		
		retry = 0;
		while (1)
		{
			if (va_dwrite_io_block(remote_block) <0 )
			{
				retry += 1;
			}
			else
			{
				break;
			}
			
			if (retry >= MAX_RETRY)
			{
				syslog(LOG_ERR,"Error va_dwrite_io_block() in ga_service_server()");
				break;
			}
		}
		/* Free allocated memory */
		remote_block = va_free_io_block(remote_block);
		va_io_close(&remote_con);
	}
}

/********************* LOCAL CODE ***********************/
/* ==================================================== */
/* Start local clients */
if ((startup_client_pid = fork()) < 0)
{
	syslog(LOG_CRIT,"Error fork() startup_pid");
	return(-10);
}
else if (startup_client_pid == 0) 
{
	/* Set exit_flag to zero */
	exit_flag = 0;
	
	/* Memory reserve for client_info vector */
	if (!(client_info = (struct ga_client_info *)calloc(conf->num_local, 
												sizeof(struct ga_client_info)) ))
	{
		syslog(LOG_CRIT,"Error calloc() client_info");
		return(-1);
	}
	
	/* Memory reserve for parameter vector */
	if (!(char_vector = (char **)calloc(GA_CLIENT_NUM_PARAM, sizeof(char *)) ))
	{
		syslog(LOG_CRIT,"Error calloc() char_vector");
		return(-1);
	}
	for (k = 0; k< GA_CLIENT_NUM_PARAM -1; k++)
	{
		if (!(char_vector[k] = (char *)calloc(BUFFSIZE, sizeof(char)) ))
		{
			syslog(LOG_CRIT,"Error calloc() char_vector[%d]",k);
			for (l = 0; l< k; l++)
			{
				free(char_vector[l]);
			}
			free(char_vector);
			return(-1);
		}
	}
	
	/* Set general parameters for all clients */
	char_vector[GA_CLIENT_NUM_PARAM -1] = NULL;
	strcpy(char_vector[0],"./ga_client\0");
	sprintf(char_vector[2],"server_ip=%s\0",conf->service_ip);	
	sprintf(char_vector[3],"server_port=%d\0",conf->service_port);
	sprintf(char_vector[4],"init_preffix=%s\0",conf->init_preffix);
	sprintf(char_vector[5],"result_preffix=%s\0",conf->result_preffix);
	
	/* Set sigaction for SIGTERM */
	sigemptyset(&sig_term.sa_mask);
	sig_term.sa_handler = ga_service_sig_term;
	sig_term.sa_flags = 0;
	sig_term.sa_flags |= SA_RESTART;
	sig_term.sa_flags |= SA_NOMASK;
	if (sigaction(SIGTERM, &sig_term, NULL) < 0)
	{
		syslog(LOG_CRIT,"Couldn't set SIGTERM handler");
		return(-1);
	}
	
	while (exit_flag == 0)
	{
		/* Spawn clients */
		for (k = 0; k< conf->num_local; k++)
		{
			if (client_info[k].is_running == 0)
			{
				/* Set the client id */
				sprintf(char_vector[1],"client_id=%d\0",k);
		
				if ((client_info[k].pid = fork()) < 0)
				{
					syslog(LOG_ERR,"Couldn't fork for client %d", k);
				}
				else if (client_info[k].pid == 0)  // exec
				{
					execvp(char_vector[0],char_vector);			
				}
				
				client_info[k].is_running = 1;
			}
		}
		
		/* Wait for clients termination, so they don't become zombies */
		for (k = 0; k < conf->num_local; k++)
		{
			temp_pid = waitpid(-1, &status, 0);	//wait for any child
		
			/* Check if normally terminated	*/
			if (WIFEXITED(status)) //Normally terminated
			{
				/* Set exit flag if exit status == 0 */
				if (WEXITSTATUS(status) == 0)
				{
					exit_flag = 1;
				}
	
				/* Else break to respawn */
				else // Exited with error
				{
					for (l = 0; l< conf->num_local; l++)
					{
						if (temp_pid == client_info[l].pid)
						{
							client_info[l].is_running = 0;
							break;
						}
					}
					break; //Stop waiting 
				}
			}
		}
		
	}
}

/******************** SERVICE CODE **********************/
/* ==================================================== */

/* Make socket and wait for connections */
if ( (listenfd = socket(PF_INET, SOCK_STREAM, 0)) < 0)
{
	syslog(LOG_CRIT,"Error socket() in ga_service_server(): %s",strerror(errno));
	return(-1);
}
bzero(&server_addr,sizeof(server_addr));
server_addr.sin_family = PF_INET;
	
if (inet_pton(PF_INET, conf->service_ip, &server_addr.sin_addr) < 1)
{
	syslog(LOG_CRIT,"Error inet_pton() in ga_service_server(): %s",strerror(errno));
	return(-1);
}
server_addr.sin_port = htons((uint16_t)(conf->service_port));

if ( (bind(listenfd, (struct sockaddr *) &server_addr, sizeof(server_addr)) ) < 0)
{
	syslog(LOG_CRIT,"Error bind() socket in ga_service_server(): %s",strerror(errno));
	return(-1);
}
if ( (listen(listenfd,LISTENQ)) <0 )
{
	syslog(LOG_CRIT,"Error listen() in ga_service_server(): %s",strerror(errno));
	return(-1);
}

/* Memory reserve */
if (!(client_addr = (struct sockaddr *)malloc(sizeof(struct sockaddr)) ))
{
	syslog(LOG_CRIT,"Error malloc client_addr");
	return(-1);
}


/* Generations loop */
for (gen = 0; gen< conf->generations; gen++)
{

	/* ACCEPT Should be in thread */
	while (1)
	{
		addrlen = sizeof(struct sockaddr);
	
		/* Malloc connptr */
		if (!(connptr = (int *)malloc(sizeof(int)) ))
		{
			syslog(LOG_CRIT,"Error malloc connptr: %s",strerror(errno));
			return(-1);
		}
	
		/* Accept connection */
		if ( (*connptr = accept(listenfd, client_addr, &addrlen)) <0 )
		{	
			if (errno == EINTR)
			{
				continue;
			}
			else
			{
				syslog(LOG_CRIT,"Accept error Inmigration Server: %s",strerror(errno));
				return(-1);
			}
		}
	}
}

}
