/**
 *  Freeverb3 Jack effect plugin Framework
 *
 *  Copyright (C) 2007-2011 Teru KAMOGASHIRA
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "fv3_config.h"

#include <cstdio>
#include <cerrno>
#include <cstdlib>
#include <cstring>
#include <cmath>

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <libgen.h>

#include <glib.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <jack/jack.h>
#include "plugin.h"
#include "CArg.hpp"
#include "freeverb/slot.hpp"
#include "freeverb/fv3_ch_tool.hpp"
#include "libsamplerate2/samplerate2.h"

#ifdef PLUGDOUBLE
typedef fv3::slot_ SLOT;
typedef double pfloat_t;
#else
typedef fv3::slot_f SLOT;
typedef float pfloat_t;
#endif

extern "C" { extern EffectPlugin *get_eplugin_info(void); }

EffectPlugin * effectPlugin;
CArg args;

jack_port_t *inputL, *inputR, *outputL, *outputR;
jack_client_t *client;
SLOT pSlot;

static void vprocess(float * inL, float * inR, float * outL, float * outR, int nframes)
{
  short int * data = new short int[nframes*2];
  if(pSlot.getsize() < nframes*2) pSlot.alloc(nframes*2, 1);
  fv3::mergeChannelsV(2, nframes, pSlot.L, inL, inR);
#ifdef PLUGDOUBLE
  src_double_to_short_array(pSlot.L, data, nframes*2);
#else
  src_float_to_short_array(pSlot.L, data, nframes*2);
#endif
  effectPlugin->mod_samples((void**)&data, nframes*2*sizeof(short int), FMT_S16_NE, (int)jack_get_sample_rate(client), 2);
#ifdef PLUGDOUBLE
  src_short_to_double_array(data, pSlot.L, nframes*2);
#else
  src_short_to_float_array(data, pSlot.L, nframes*2);
#endif
  fv3::splitChannelsV(2, nframes, pSlot.L, outL, outR);
  delete[] data;
}

static int process(jack_nframes_t nframes, void *arg)
{
  jack_default_audio_sample_t *inL, *inR, *outL, *outR;
  inL = (jack_default_audio_sample_t*)jack_port_get_buffer(inputL, nframes);
  inR = (jack_default_audio_sample_t*)jack_port_get_buffer(inputR, nframes);
  outL = (jack_default_audio_sample_t*)jack_port_get_buffer(outputL, nframes);
  outR = (jack_default_audio_sample_t*)jack_port_get_buffer(outputR, nframes);
  vprocess(inL, inR, outL, outR, nframes);
  return 0;
}

void jack_shutdown(void *arg)
{
  exit(1);
}

void help(const char * cmd)
{
  std::fprintf(stderr,
	       "Usage: %s [options]\n"
	       "[[Options]]\n"
	       "-id JACK Unique ID (default/0=pid)\n"
	       "[[Example]] %s -id 1\n"
	       "\n",
	       cmd, cmd);
}

static void destroy_window(GtkWidget *widget, gpointer data){ gtk_main_quit(); }

static void g_configure(gpointer data,guint action,GtkWidget *widget)
{
  effectPlugin->configure();
}

static void g_about(gpointer data,guint action,GtkWidget *widget)
{
  effectPlugin->about();
}

static void g_quit(gpointer data,guint action,GtkWidget *widget)
{
  destroy_window(widget, data);
}

GtkWidget *window, *menubar, *vbox;
GtkWidget *menu_bar;
GtkAccelGroup *accel_group;
GtkItemFactory *item_factory;
gint n_menu_items;
static GtkItemFactoryEntry menu_items[] =
{
  {"/_File",           NULL,         NULL,                    0, "<Branch>"},
  {"/File/_Configure", "<control>C", (void (*)())g_configure, 0, "<Item>"},
  {"/File/_About",     "<control>A", (void (*)())g_about,     0, "<Item>"},
  {"/File/-",          NULL,         NULL,                    0, "<Separator>"},
  {"/File/_Quit",      "<control>Q", (void (*)())g_quit,      0, "<Item>"},
};


static void make_directory(const gchar * path, mode_t mode)
{
  if(mkdir(path, mode) == 0) return;
  if (errno == EEXIST) return;
  g_printerr("Could not create directory (%s): %s", path, g_strerror(errno));
}

static void make_user_dir(void)
{
  gchar *dirname = g_build_filename(g_get_home_dir(), BMP_RCPATH, NULL);
  const mode_t mode755 = S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH;
  make_directory(dirname, mode755);
}

int main(int argc, char **argv)
{
  g_thread_init(NULL);
  if (!g_thread_supported())
    {
      g_printerr("Sorry, threads isn't supported on your platform.\n\n"
		 "If you're on a libc5 based linux system and installed Glib & GTK+ before you\n"
		 "installed LinuxThreads you need to recompile Glib & GTK+.\n");
      exit(EXIT_FAILURE);
    }
  gdk_threads_init();
  if (!gtk_init_check(&argc, &argv))
    {
      if (argc < 2) {
	/* GTK check failed, and no arguments passed to indicate
	   that user is intending to only remote control a running session */
	g_printerr("Unable to open display, exiting.");
	exit(EXIT_FAILURE);
      }
      exit(EXIT_SUCCESS);
    }
  
  effectPlugin = get_eplugin_info();
  effectPlugin->init();
  std::fprintf(stderr, "Freeverb3 JACK effect plugins Framework\n");
  std::fprintf(stderr, "<"PACKAGE"-"VERSION">\n");
  std::fprintf(stderr, "sizeof(pfloat_t) = %d\n", (int)sizeof(pfloat_t));

  help(basename(argv[0]));
  if(args.registerArg(argc, argv) != 0) exit(-1);
  
  make_user_dir();

  char *client_name, *server_name = NULL;
  jack_options_t options = JackNullOption;
  jack_status_t status;
  
  int uid = args.getLong("-id");
  if(uid <= 0) uid = getpid();
  char unique_name[1024];
  std::snprintf(unique_name, 1024, "%s_%d", basename(argv[0]), uid);
  client_name = unique_name;
  client = jack_client_open(client_name, options, &status, server_name);
  if (client == NULL)
    {
      std::fprintf(stderr, "jack_client_open() failed, status = 0x%2.0x\n", status);
      if (status & JackServerFailed) std::fprintf(stderr, "Unable to connect to JACK server\n");
      exit (-1);
    }
  if (status & JackServerStarted) std::fprintf(stderr, "JACK server started\n");
  if (status & JackNameNotUnique)
    {
      client_name = jack_get_client_name(client);
      std::fprintf(stderr, "unique name `%s' assigned\n", client_name);
    }
  
  jack_set_process_callback(client, process, 0);
  jack_on_shutdown(client, jack_shutdown, 0);
  inputL = jack_port_register(client, "inputL", JACK_DEFAULT_AUDIO_TYPE, JackPortIsInput, 0);
  inputR = jack_port_register(client, "inputR", JACK_DEFAULT_AUDIO_TYPE, JackPortIsInput, 0);
  outputL = jack_port_register(client, "outputL", JACK_DEFAULT_AUDIO_TYPE, JackPortIsOutput, 0);
  outputR = jack_port_register(client, "outputR", JACK_DEFAULT_AUDIO_TYPE, JackPortIsOutput, 0);
  if(inputL == NULL||inputR == NULL||outputL == NULL||outputR == NULL)
    {
      std::fprintf(stderr, "no more JACK ports available\n");
      exit(-1);
    }
  
  if(jack_activate(client))
    {
      std::fprintf(stderr, "cannot activate client\n");
      exit(-1);
    }
  
  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  char window_name[1024];
  std::snprintf(window_name, 1024, "JACK - %s", effectPlugin->description);
  gtk_window_set_title(GTK_WINDOW(window), window_name);
  gtk_widget_set_usize(GTK_WIDGET(window), 500, 100);
  gtk_signal_connect(GTK_OBJECT(window),"destroy", GTK_SIGNAL_FUNC(destroy_window), NULL);

  vbox = gtk_vbox_new(FALSE,0);
  gtk_container_add(GTK_CONTAINER(window),vbox);
  gtk_widget_show(vbox);

  accel_group = gtk_accel_group_new();
  n_menu_items = sizeof(menu_items) / sizeof(menu_items[0]);
  item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR,"<option>",accel_group);
  gtk_item_factory_create_items(item_factory,n_menu_items,menu_items,window);
  gtk_window_add_accel_group(GTK_WINDOW(window),accel_group);
  menu_bar = gtk_item_factory_get_widget(item_factory,"<option>");
  gtk_box_pack_start(GTK_BOX(vbox),menu_bar,FALSE,TRUE,0);
  gtk_widget_show(menu_bar);
  
  gtk_widget_show(window);
  gtk_main();
  jack_client_close(client);
  exit(0);
}
