/*
  (C) 2005 Vikas Gorur <vikasgp@gmail.com>

  callbacks.c: Callback functions
  
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.
    
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.
    
  You should have received a copy of the GNU General Public
  License along with this program; if not, write to the Free
  Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
  Boston, MA 02110-1301 USA
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <glib.h> 
#include <loudmouth/loudmouth.h>
#include <readline/readline.h>

#include "util.h"
#include "freetalk.h"
#include "callbacks.h"
#include "roster.h"
#include "presence.h"
#include "extensions.h"

/*
  Callback for SSL verification
*/

LmSSLResponse 
ft_ssl_response_cb (LmSSL *ssl,
     LmSSLStatus st,
     gpointer user_datta)
{
  /* dont really care */
  return LM_SSL_RESPONSE_CONTINUE;
}


/*
   Called when authentication succeeds
*/
 
void
ft_authenticate_cb (LmConnection *conn, gboolean success, gpointer user_data)
{
  do_session_init (success);
}

/*
  Called when the connection is opened
*/

void
ft_connection_open_cb (LmConnection *conn, gboolean success, gpointer u)
{
  ft_state *state = (ft_state *)u;
  if (success) {
    do_set_conn_status (FT_CONN);
    PRINTF ("Connected.");
  }
  else
    {
      do_set_conn_status (FT_DEAD);
      PRINTF ("Could not connect."); 
      return;
    }

  PRINTF ("Authenticating ..."); fflush(stdout);
  lm_connection_authenticate (conn, state->jid.node, state->password,
			      state->jid.resource, ft_authenticate_cb,
			      NULL, g_free, NULL);
}

/*
  Incoming message handlers
*/

/*
  Parse the 'stamp' element in an offline message and return a string
  representation of it. The string must be freed with g_free
*/

static char *
parse_timestamp (const char *ts)
{
  char *time = g_strdup (strchr (ts, 'T') + 1);
  return time;
}

/*
  LM_MESSAGE_TYPE_MESSAGE
*/

static LmHandlerResult
ft_msg_msg_handler (LmMessageHandler *handler, LmConnection *conn,
		    LmMessage *msg, gpointer user_data)
{
  LmMessageNode *root, *body, *x;
  const char *from, *msg_str, *type;
  char *ts = NULL;

  root = lm_message_get_node (msg);
  body = lm_message_node_get_child (root, "body");

  from = lm_message_node_get_attribute (msg->node, "from");
  msg_str = lm_message_node_get_value (body);

  type = lm_message_node_get_attribute (msg->node, "type");
  if (type && g_ascii_strcasecmp (type, "chat") != 0)
    {
      do_printf ("[message of type '%s']", type);
      return LM_HANDLER_RESULT_REMOVE_MESSAGE;
    }

  // Offline messages
  for (x = root->children; x != NULL; x = x->next)
    {
      if (!g_ascii_strcasecmp (x->name, "x"))
	{
	  const char *xmlns = lm_message_node_get_attribute (x, "xmlns");
	  if (xmlns && !g_ascii_strcasecmp (xmlns, "jabber:x:delay"))
	    {
	      ts = parse_timestamp ((char *)lm_message_node_get_attribute (x, "stamp"));
	    }
	}
    }

  /* TBD : make the below stripping of /Resoruce configurable */
  if (1) {
     if (strchr (from, '/'))
        *strchr (from, '/') = '\0';
  }

  set_hook_return (0);
  scm_run_hook (ex_message_receive_hook,
		gh_list (ts ? gh_str02scm (ts) : gh_str02scm (""),
			 gh_str02scm (from),
			 gh_str02scm (msg_str),
			 SCM_UNDEFINED));
  if (ts) g_free (ts);
  
  if (get_hook_return () == 1)
    return LM_HANDLER_RESULT_REMOVE_MESSAGE;

  PRINTF ("%s: %s", from, msg_str);

  return LM_HANDLER_RESULT_REMOVE_MESSAGE;
}

/*
  LM_MESSAGE_TYPE_PRESENCE
*/

static LmHandlerResult
ft_msg_presence_handler (LmMessageHandler *handler, LmConnection *conn,
			 LmMessage *msg, gpointer user_data)
{
  //  PRINTF ("[presence recieved]");
  ft_presence_cb (msg);
  /* scm_run_hook (ex_presence_receive_hook, ... ) */
  return LM_HANDLER_RESULT_REMOVE_MESSAGE;
}

/*
  LM_MESSAGE_TYPE_IQ
*/

static LmHandlerResult
ft_msg_iq_handler (LmMessageHandler *handler, LmConnection *conn,
		   LmMessage *msg, gpointer user_data)
{
  /* Currently the only IQ message we'll handle is the roster */
  LmMessageNode *query = lm_message_node_get_child (msg->node, "query");
  if (query)
    {
      const char *ns = lm_message_node_get_attribute (query, "xmlns");
      if (ns && !g_ascii_strcasecmp (ns, "jabber:iq:roster"))
	  ft_roster_cb (msg);
      else
	PRINTF ("[iq received: %s (unhandled yet)]", ns);
    }

  return LM_HANDLER_RESULT_REMOVE_MESSAGE;
}

/*
  Disconnection handler to cleanup mess
*/

static void
ft_disconnect_function (LmConnection *conn,
			LmDisconnectReason reason,
			gpointer user_data)
{
  char *reason_str;

  switch (reason) {
  case 0:
    reason_str = "User request";
    break;
  case 1:
    reason_str = "Network error (Timeout with server)";
    break;
  case 2:
    reason_str = "Protocol error (Hangup from server)";
    break;
  case 3: 
    if (do_get_conn_status() == FT_AUTH)
      reason_str = "Possible login from another location";
    else
      reason_str = "Invalid authentication";
    break;
  default:
    reason_str = "Unknown error";
    break;
  }
  PRINTF ("Disconnected from %s, reason(%d): %s", state.server, reason, reason_str);

  lm_connection_unref (state.conn);
  state.conn = NULL;
  do_set_conn_status (FT_DEAD);
  ft_roster_flush ();

  scm_run_hook (ex_disconnect_hook, gh_list (gh_int2scm (reason), SCM_UNDEFINED));

  return;
}

/*
  Register handlers for each type of message
*/

void
ft_register_msg_handlers (LmConnection *conn)
{
  LmMessageHandler *handler = lm_message_handler_new ((LmHandleMessageFunction) ft_msg_msg_handler,
						      NULL, NULL);
  lm_connection_register_message_handler (conn, handler, LM_MESSAGE_TYPE_MESSAGE,
					  LM_HANDLER_PRIORITY_NORMAL);
  lm_message_handler_unref (handler);

  
  handler = lm_message_handler_new ((LmHandleMessageFunction) ft_msg_presence_handler,
				    NULL, NULL);
  lm_connection_register_message_handler (conn, handler, LM_MESSAGE_TYPE_PRESENCE,
					  LM_HANDLER_PRIORITY_NORMAL);
  lm_message_handler_unref (handler);

  
  handler = lm_message_handler_new ((LmHandleMessageFunction) ft_msg_iq_handler,
				    NULL, NULL);
  lm_connection_register_message_handler (conn, handler, LM_MESSAGE_TYPE_IQ,
					  LM_HANDLER_PRIORITY_NORMAL);
  lm_message_handler_unref (handler);

  lm_connection_set_disconnect_function (conn, ft_disconnect_function, NULL, NULL);
}
