// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

//-----------------------------------------------------------------------------
// Class:
// PatchKernelNode
//-----------------------------------------------------------------------------

#ifndef POOMA_EVALUATOR_MULTIARGKERNEL_H
#define POOMA_EVALUATOR_MULTIARGKERNEL_H

//////////////////////////////////////////////////////////////////////

/** @file
 * @ingroup Evaluator
 * @brief
 * MultiArgKernel is a generalized iterate object to hand off to smarts.
 *
 * The iterate contains a MultiArg objects which can contain 1, 2, or 3
 * fields, and a Function object which is applied to the MultiArg object
 * when the iterate is run.  On construction, the user provides vectors
 * of flags that say which arguments are being written to and read from.
 *
 * This particular kernel was created to avoid the proliferation of kernels
 * generated by patch-based computations.  Previously, separate kernels were
 * defined for each number of argument passed in, and in some cases different
 * kernels were written for different forms of the function.
 */

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Engine/EngineFunctor.h"
#include "Evaluator/LoopApply.h"
#include "Functions/MultiArg.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//
// MultiArg kernel:
//
//-----------------------------------------------------------------------------

template<class MultiArg, class Function>
class MultiArgKernel
  : public Pooma::Iterate_t
{
public:
  MultiArgKernel(const MultiArg &multiArg,
		  const Function &function,
		  std::vector<bool> &write,
		  std::vector<bool> &read)
    : Pooma::Iterate_t(Pooma::scheduler()),
      multiArg_m(multiArg), function_m(function),
      write_m(write), read_m(read)
  {
    // Currently we assume that the first argument drives the computation,
    // so the affinity comes from the first argument.

    DataObjectRequest<BlockAffinity> getAffinity;
    hintAffinity(engineFunctor(multiArg.a1_m.engine(), getAffinity));

    // Request locks

    // First make the write request.
    // The write request tag remembers the data block
    // for the left hand side so we can check if there is a stencil
    // on the right.

    typedef DataObjectRequest<WriteRequest> WriteRequest_t;
    typedef DataObjectRequest<ReadRequest>  ReadRequest_t;

    WriteRequest_t writeReq(*this);

    applyMultiArgIf(multiArg, ExpressionApply<WriteRequest_t>(writeReq),
		    write_m);

    ReadRequest_t readReq(writeReq);

    applyMultiArgIf(multiArg, ExpressionApply<ReadRequest_t>(readReq),
		    read_m);
  }	      

  virtual ~MultiArgKernel()
  {
    typedef DataObjectRequest<WriteRelease> WriteRequest_t;
    typedef DataObjectRequest<ReadRelease>  ReadRequest_t;

    WriteRequest_t writeReq;

    applyMultiArgIf(multiArg_m, ExpressionApply<WriteRequest_t>(writeReq),
		    write_m);

    ReadRequest_t readReq(writeReq);

    applyMultiArgIf(multiArg_m, ExpressionApply<ReadRequest_t>(readReq),
		    read_m);
  }

  virtual void run()
  {
    function_m(multiArg_m);
  }

private:

  MultiArg multiArg_m;
  Function function_m;
  std::vector<bool> write_m;
  std::vector<bool> read_m;
};

//////////////////////////////////////////////////////////////////////

#endif     // POOMA_EVALUATOR_MULTIARGKERNEL_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: MultiArgKernel.h,v $   $Author: richard $
// $Revision: 1.4 $   $Date: 2004/11/01 18:16:40 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
