// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

//-----------------------------------------------------------------------------
// Classes: 
//   Cartesian
//-----------------------------------------------------------------------------

#ifndef POOMA_COORDINATESYSYEMS_CARTESIAN_H
#define POOMA_COORDINATESYSYEMS_CARTESIAN_H

//-----------------------------------------------------------------------------
// Overview: 
// 
// Cartesian : An N-dimensional cartesian coordinate system.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Domain/Region.h"
#include "Tiny/Vector.h"

#include <iosfwd>

//-----------------------------------------------------------------------------
// Full Description:
// 
// Cartesian is an N-dimensional cartesian coordinate system, where the
// dimensionality is specified via the single template parameter.
//
// Exported typedefs:
//
//   This_t - this class.
//
// Exported enumerations and constants:
//
//   dimension - the dimensionality of the coordinate system.
//
// Accessors:
//
//   distance(pt1, pt2) - returns the distance between two points.
//   print(stream) - prints information about this coordinate system to a
//     stream.
//   volume(Region) - returns the volume of the region.
//
//-----------------------------------------------------------------------------

template<int Dim>
class Cartesian
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.

  // This class.
  
  typedef Cartesian<Dim> This_t;
 
  // The dimensionality of this coordinate system.
  
  enum { dimensions = Dim };

  //---------------------------------------------------------------------------
  // Volume calculation functions.

  // Compute the volume of a region, interpreted as a range of coordinate
  // values for each coordinate. In 1D, this is a length, in 2D an area, in 3D
  // a volume, in 4D a hypervolume, etc.

  template<class T>
  T volume(const Region<Dim, T> &r) const
  {
    return r.size();
  }

  // Volume of a general solid (dense set of points in space). Defer
  // calculation of volume to the class representing the solid.
  
  /*
  template<class Solid>
  typename Solid::PointComponentType_t volume(const Solid &solid) const
  {
    return solid.volume();
  }
  */
  
  //---------------------------------------------------------------------------
  // Distance calculation function.

  template<class T>
  T distance(const Vector<Dim, T> &pt1, const Vector<Dim, T> &pt2)
  {
    T dist = 0.0;
    for (int d = 0; d < Dim; d++) {
      dist += (pt2(d) - pt1(d))*(pt2(d) - pt1(d));
    }
    return sqrt(dist);
  }

  // --------------------------------------------------------------------------
  // Print a Cartesian<Dim> on an output stream.
  
  template <class Ostream>
  void print(Ostream &ostr) const
  {
    ostr << "Cartesian<" << Dim << ">";
  }

};

//-----------------------------------------------------------------------------
//
// ostream inserter for Cartesian<Dim>.
//
//-----------------------------------------------------------------------------

template<int Dim>
std::ostream &operator<<(std::ostream &ostr, 
  const Cartesian<Dim> &cs)
{
  cs.print(ostr);
  return ostr;
}

#endif // POOMA_COORDINATESYSYEMS_CARTESIAN_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Cartesian.h,v $   $Author: richard $
// $Revision: 1.11 $   $Date: 2004/11/01 18:16:25 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
