// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// UMPJacobi.cpp is used in Tutorial 4 to show how to declare and use
// MultiPatch arrays to support parallel calculations.
//-----------------------------------------------------------------------------

#include "Pooma/Arrays.h"

#include <iostream>

// The size of each side of the domain. 
// N+2 must be a multiple of 4.
const int N = 18;

// Apply a Jacobi iteration on the given domain.
template<class T1, class E1, class T2, class E2>
void
ApplyJacobi(
    const Array<2, T1, E1>      & V,	// to be relaxed
    const Array<2, T2, E2>      & b,	// fixed term
    const Range<2>              & IJ	// region of calculation
){
    V(IJ) = 0.25 * (V(IJ+Loc<2>(1,0)) + V(IJ+Loc<2>(-1,0)) +
                    V(IJ+Loc<2>(0,1)) + V(IJ+Loc<2>(0,-1)) - b.read(IJ));
}

// Apply periodic boundary conditions by copying.
template<class T1, class E1>
void
ApplyPeriodic(
    const Array<2, T1, E1>      & V  // to be wrapped
){
    // Get the horizontal and vertical extents of the domain.
    Interval<1> I = V.domain()[0],
                J = V.domain()[1];

    // Copy each of the four slices in turn.
    V(I.first(), J) = V(I.last()-1, J);
    V(I.last(),  J) = V(I.first()+1,J);
    V(I, J.first()) = V(I, J.last()-1);
    V(I, J.last())  = V(I, J.first()+1);
}

int main(
    int                 argc,           // argument count
    char *              argv[]          // argument vector
){
    // Initialize Pooma.
    Pooma::initialize(argc,argv);

    // The domain with guard cells on the boundary.
    Interval<2> guarded( Interval<1>(0, N+1), Interval<1>(0, N+1) );

    // Create the layouts.
    UniformGridLayout<2> guardedLayout( guarded, Loc<2>(4, 4), ReplicatedTag() );

    // The array we'll be solving for.
    Array<2, double, MultiPatch<UniformTag,Brick> > V(guardedLayout);
    V = 0.0;

    // The right hand side of the equation.
    Array<2, double, MultiPatch<UniformTag,Brick> > b(guardedLayout);
    b = 0.0;
    
    // Must block since we're doing som scalar code here (see Tutorial 4).
    Pooma::blockAndEvaluate();
    b(3*N/4,   N/4) = -1.0;
    b(  N/4, 3*N/4) =  1.0;

    // The interior domain, now with stride 2.
    Range<2> IJ( Range<1>(1, N-1, 2), Range<1>(1, N-1, 2) );

    // Iterate 200 times.
    for (int i=0; i<200; ++i)
    {
        ApplyJacobi(V, b, IJ);
        ApplyJacobi(V, b, IJ+Loc<2>(1,1)); 
        ApplyJacobi(V, b, IJ+Loc<2>(1,0));
        ApplyJacobi(V, b, IJ+Loc<2>(0,1));
        ApplyPeriodic(V);
    }

    // Print out the result.
    std::cout << V << std::endl;

    // Clean up and report success.
    Pooma::finalize();
    return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: UMPJacobi.cpp,v $   $Author: richard $
// $Revision: 1.20 $   $Date: 2004/11/01 18:16:06 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
