// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// Contents:
//   main function for Jacobi example.
//-----------------------------------------------------------------------------
//
// The phi and theta parameters below come from decomposing an equation like:
//
//	d_t x = A x
//
// When time discretizing, it uses theta to pick the time centering of the
// right hand side:
//
//	x^{n+1} - x^n = dt A ( theta x^n + (1-theta) x^{n+1} )
//
// So, theta=0.5 is centered in time.
//
// phi is associated with the Jacobi iteration itself.  A normal Jacobi
// iteration to solve A y = b, where D is the diagonal of A and B is A-D looks
// like:
//
//	y^{n+1} = D^{-1} b - D^{-1} B y^n
//              = c + R y^n
//
// Instead of that, you mix y^n and y^{n+1} in order to adjust the 
// spectrum of R:
//
//	y^{n+1} = phi c + ( (1-phi) I + phi R ) y^n
//
// So, phi=1 is the conventional one.  The reason you do that is that R's
// spectrum kills modes in the middle of the spectrum faster than either the
// small or large wavenumbers.  By twiddling phi you can get it to kill large
// wavenumbers better (useful for multigrid), and the optimial value depends
// on R (and therefore on A). Generally you get phi<1, and 0.75 is OK for this
// operator.  Choosing 1 is fine though.
//-----------------------------------------------------------------------------

// include files

#include "Jacobi.h"
#include "StencilJac.h"
#include "stats.h"
#include <iostream>

using std::cout;
using std::cin;
using std::endl;
using std::setw;

int main(int argc, char* argv[])
{
  Pooma::initialize(argc,argv);

  double dx,dt,tfinal,theta,phi;
  int N;
  std::cout << "Final time (0.0 - 1.0): ";
  std::cin >> tfinal;
  std::cout << "Size (10 - 100): ";
  std::cin >> N;
  std::cout << "theta (0.0 - 1.0): ";
  std::cin >> theta;
  std::cout << "phi (0.75 - 1.0): ";
  std::cin >> phi;
  dx = 1.0/(N+1);
  dt = 0.1*dx;
  Interval<1> I(1,N), J(1,N);
  Interval<2> IJ(I,J);

  Array<2> rho0(N+2,N+2), rho1(N+2,N+2);
  int total_iters, iters;
  double t;

  std::cout << "Beginning Jacobi iterations..." << std::endl;
  rho0 = 0;
  rho0(N/2,N/2) = 1;

  stats(rho0,dx,0.0);
  total_iters = 0;
  for (t=0; t<tfinal; t+=2*dt)
    {
      iters = 2*Jacobi(rho0,dx,dt,theta,phi,IJ);
      total_iters += iters;
      std::cout << std::setw(6) << iters;
      stats(rho0,dx,t+dt);
    }
  std::cout << "total iters = " << total_iters << std::endl;

  std::cout << "Beginning StencilJac iterations..." << std::endl;
  rho1 = 0;
  rho1(N/2,N/2) = 1;

  stats(rho1,dx,0.0);
  total_iters = 0;
  for (t=0; t<tfinal; t+=2*dt)
    {
      iters = 2*StencilJac(rho1,dx,dt,theta,phi,IJ);
      total_iters += iters;
      std::cout << std::setw(6) << iters;
      stats(rho1,dx,t+dt);
    }
  std::cout << "total iters = " << total_iters << std::endl;

  Pooma::finalize();
  return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: driver.cpp,v $   $Author: richard $
// $Revision: 1.11 $   $Date: 2004/11/01 18:15:53 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
