// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

#ifndef POOMA_EXAMPLES_SOLVERS_KRYLOV_CGSOLVE_H
#define POOMA_EXAMPLES_SOLVERS_KRYLOV_CGSOLVE_H

//-----------------------------------------------------------------------------

// Contents:
//   PCGSolve - preconditioned conjugate gradient solver
//-----------------------------------------------------------------------------

// Templated preconditioned conjugate gradient solver taken from the book
// by Ortega.

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
//
//  Templated conjugate gradient solver based on the templated solver in the
//  book: Templates for the Solution of Linear Systems by R. Barrett, M. Berry, 
//  T.F. Chan, J. Demmel, J. Donato, J. Dongarra, V. Eijkhout, R. Pozo, 
//  C. Romine, and H. van der Vorst.
//
//  The major differences between the current version and the one in the book:
//  2) takes a dot product class that defines the dot product
//  3) takes an init class that tells you how to initialize arrays.
//  4) uses dot(a, a) for norm(a)
//  5) matrix operation y=Ax is represented by A(x, y) instead of y=A(x)
//
//  PCGSolve(A, M, x, b, dot, init, max_iter, tol);
//    -solves Ax=b for x
//    -iterates until max_iter iterations or until
//     dot(Ax-b, Ax-b) < tol*dot(b, b)
//    -A must be self-adjoint with respect to dot and positive definite.
//
//-----------------------------------------------------------------------------

// currently does the convergence test in the preconditioned dot product
// (rhat, r); A better implementation would check the true residual
// (r, r) after passing the preconditioned convergence test

template<class Matrix, class Prec, class V1, class V2, class Dot, class Init>
void
PCGSolve (const Matrix &A, const Prec &M, V1& x, const V2& b, 
   const Dot &dot, const Init &init, int &max_iter, double &tol)
{
    double resid, r2, r2new;
    double alpha, beta;

    bool notConverged = true;

    V1 r, rhat, d, q;
    init(r);
    r = 0.0;
    init(rhat);
    rhat = 0.0;
    init(d);
    d = 0.0;
    init(q);
    q = 0.0;

/*
    double residTrue;
    V1 rTrue;
    init (rTrue);
*/

    double normb = dot(b, b);

    // r = b - Ax is the first residual
    A(x, q);
    r = b - q;

    M(r, rhat);
    d = rhat;

    if (normb == 0.0)  normb = 1;
  
    r2 = dot(rhat, r);
    resid = r2 / normb;

    int i;
    for (i = 1; (i <= max_iter)&&(notConverged); i++) {
      A(d, q);
      alpha = r2 / dot(d, q);
      x += alpha * d;
      r -= alpha * q;

      M(r, rhat);
      r2new = dot(rhat, r);
      beta = r2new / r2;
      d = rhat + beta * d;

      r2 = r2new;
      resid = r2 / normb;

//    notConverged = (resid > tol);
      notConverged = (r2 > tol);

/*

      if (!notConverged) {
         A(x, q);
         rTrue = b - q;
         residTrue = dot(rTrue, rTrue) / normb; 
         notConverged = (residTrue > tol);
      }
*/

// delong edits here

      std::cout << "PCG: iter = " << i << " sqrt resid = " <<
         sqrt(resid) << " log sqrt resid = " << log(sqrt(resid))/log(10.0)
         << std::endl;

// delong edits end

    }
    tol = resid;
    max_iter = i;
}

//////////////////////////////////////////////////////////////////////

#endif
