// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

//-----------------------------------------------------------------------------
// Class:
// GuardedPatchEvaluator
//-----------------------------------------------------------------------------

#ifndef POOMA_EVALUATOR_GUARDEDPATCHEVALUATOR_H
#define POOMA_EVALUATOR_GUARDEDPATCHEVALUATOR_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
//
// GuardedPatchEvaluator is an evaluator that takes up to three array
// arguments to perform a user-defined function on those arrays.
//
// We assume that we're writing to the first array and reading from the
// others.
//
// For now, the parallel decomposition is based on the array being written to.
// (Except for the three-argument version that decomposes based on the
// last argument.)
// (If the other arrays don't conform, the INode view will fail.)
//
// The user-defined function can specify a number of guard layers that are
// required.
//
// We also make the assumption that all the arrays are aligned, so there
// is no zero-based view to take.
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "PETE/PETE.h"
#include "Evaluator/EvaluatorTags.h"
#include "Evaluator/Evaluator.h"
#include "Evaluator/PatchKernel.h"

//-----------------------------------------------------------------------------
//
// Full Description:
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------

template<int D, class Function>
inline Interval<D> extendDomain(const Interval<D> &domain, const Function &f)
{
  Interval<D> ret;
  int d;
  for (d = 0; d < D; ++d)
  {
    ret[d] =
      Interval<1>(
		  domain[d].first() - f.lowerExtent(d),
		  domain[d].last() + f.upperExtent(d)
		  );
  }
  return ret;
}

template<int D, class Function>
inline INode<D> extendDomain(const INode<D> &inode, const Function &f)
{
  return INode<D>(inode, extendDomain(inode.domain(), f));
}

template<class EvalTag>
class GuardedPatchEvaluator
{
};

template<>
class GuardedPatchEvaluator<MainEvaluatorTag>
{
public:

  // Default ctor.

  GuardedPatchEvaluator() {}

  // Destructor

  ~GuardedPatchEvaluator() {}

  template<class A1, class Function, int Dim>
  static void
  evaluate(const A1 &a1,
	   const Function &function, const Interval<Dim> &domain)
  {
    typedef typename EvaluatorTag1<A1>::Evaluator_t Evaluator_t;

    Pooma::beginExpression();
    GuardedPatchEvaluator<Evaluator_t>::evaluate(a1, function, domain);
    notifyEngineWrite(a1.engine());
    Pooma::endExpression();
  }

  template<class A1, class A2, class Function, int Dim>
  static void
  evaluate(const A1 &a1, const A2 &a2,
	    const Function &function, const Interval<Dim> &domain)
  {
    typedef typename EvaluatorTag<A1,A2>::Evaluator_t Evaluator_t;

    Pooma::beginExpression();
    GuardedPatchEvaluator<Evaluator_t>::evaluate(a1, a2, function, domain);
    notifyEngineWrite(a1.engine());
    Pooma::endExpression();
  }

  template<class A1, class A2, class A3, class Function, int Dim>
  static void
  evaluate(const A1& a1, const A2& a2, const A3 &a3,
	   const Function& function, const Interval<Dim> &domain)
  {
    typedef typename EvaluatorTag1<A2>::Evaluator_t Eval2_t;
    typedef typename EvaluatorTag1<A3>::Evaluator_t Eval3_t;
    typedef typename EvaluatorCombine<Eval2_t,Eval3_t>::Evaluator_t Eval23_t;
    typedef typename EvaluatorTag1<A1>::Evaluator_t Eval1_t;
    typedef typename EvaluatorCombine<Eval1_t,Eval23_t>::Evaluator_t Evaluator_t;

    Pooma::beginExpression();
    GuardedPatchEvaluator<Evaluator_t>::evaluate(a1, a2, a3, function, domain);
    notifyEngineWrite(a1.engine());
    Pooma::endExpression();
  }

  template<class A1, class Function>
  static void
  createIterate(const A1& a1,
		const Function& function)
  {
    Pooma::Iterate_t *iterate = new PatchKernel<A1,Function>(a1,function);
    Pooma::scheduler().handOff(iterate);
  }

  template<class A1, class A2, class Function>
  static void
  createIterate(const A1 &a1, const A2 &a2,
		const Function &function)
  {
    Pooma::Iterate_t *iterate =
      new PatchKernel2<A1, A2, Function>(a1, a2, function);

    Pooma::scheduler().handOff(iterate);
  }

  template<class A1, class A2, class A3, class Function>
  static void
  createIterate(const A1 &a1, const A2 &a2, const A3 &a3,
		const Function &function)
  {
    Pooma::Iterate_t *iterate =
      new PatchKernel3<A1, A2, A3, Function>(a1, a2, a3, function);

    Pooma::scheduler().handOff(iterate);
  }

private:
};

// The single patch version just passes the tag on to generate
// an expression kernel.

template<>
class GuardedPatchEvaluator<SinglePatchEvaluatorTag>
{
public:

  //
  // Default ctor.
  // The only member data can construct itself, so we
  // don't need to specify anything.
  //
  GuardedPatchEvaluator() {}

  //
  // Destructor
  //
  ~GuardedPatchEvaluator() {}

  template<class A1, class Function, int Dim>
  static void
  evaluate(const A1& a1,
	   const Function& function, Interval<Dim> domain)
  {
    Interval<Dim> newDom = extendDomain(domain, function);
    GuardedPatchEvaluator<MainEvaluatorTag>::createIterate(a1(newDom),
							   function);
  }

  template<class A1, class A2, class Function, int Dim>
  static void
  evaluate(const A1 &a1, const A2 &a2,
	    const Function& function, Interval<Dim> domain)
  {
    Interval<Dim> newDom = extendDomain(domain, function);
    GuardedPatchEvaluator<MainEvaluatorTag>::createIterate(a1(newDom),
							   a2(newDom),
							   function);
  }

  template<class A1, class A2, class A3, class Function, int Dim>
  static void
  evaluate(const A1 &a1, const A2 &a2, const A3 &a3,
	    const Function& function, Interval<Dim> domain)
  {
    Interval<Dim> newDom = extendDomain(domain, function);
    GuardedPatchEvaluator<MainEvaluatorTag>::createIterate(a1(newDom),
							   a2(newDom),
							   a3(newDom),
							   function);
  }

private:
};



// The multiple patch version makes patches and sends them out to
// the single patch evaluator.

template<>
class GuardedPatchEvaluator<MultiPatchEvaluatorTag>
{
public:

  //
  // Default ctor.
  // The only member data can construct itself, so we
  // don't need to specify anything.
  //
  GuardedPatchEvaluator() {}

  //
  // Destructor
  //
  ~GuardedPatchEvaluator() {}

  template<class A1, class Function, int Dim>
  static void
  evaluate(const A1 &a1,
	   const Function &function, const Interval<Dim> &domain)
  {
    typedef std::vector<INode<Dim> > INodeContainer_t;
    INodeContainer_t inodes;
    GlobalIDDataBase gidStore;
    int id = a1.engine().layout().ID();
    int key = GlobalIDDataBase::nullNodeKey();
    a1.engine().layout().touches(domain, std::back_inserter(inodes),
				 TouchesConstructINode<Dim>(id, key,
							    &gidStore));

    typename INodeContainer_t::const_iterator i = inodes.begin();
    while (i != inodes.end())
    {
      INode<Dim> node = extendDomain(*i, function);
      GuardedPatchEvaluator<MainEvaluatorTag>::createIterate(a1(node),
							     function);
      ++i;
    }
  }

  template<class A1, class A2, class Function, int Dim>
  static void
  evaluate(const A1 &a1, const A2 &a2,
	   const Function &function, const Interval<Dim> &domain)
  {
    typedef std::vector<INode<Dim> > INodeContainer_t;
    INodeContainer_t inodes;
    GlobalIDDataBase gidStore;
    int id1 = a1.engine().layout().ID();
    int id2 = a2.engine().layout().ID();
    int key = GlobalIDDataBase::nullNodeKey();
    a1.engine().layout().touches(domain, std::back_inserter(inodes),
				 TouchesConstructINode<Dim>(id1, key,
							    &gidStore));
    gidStore.shared(id2, id1);
    a2.engine().fillGuards();

    typename INodeContainer_t::const_iterator i = inodes.begin();
    while (i != inodes.end())
    {
      INode<Dim> node = extendDomain(*i, function);
      GuardedPatchEvaluator<MainEvaluatorTag>::createIterate(a1(node),
							     a2(node),
							     function);
      ++i;
    }
  }

  template<class A1, class A2, class A3, class Function, int Dim>
  static void
  evaluate(const A1 &a1, const A2 &a2, const A3 &a3,
	   const Function &function, Interval<Dim> domain)
  {
    typedef std::vector<INode<Dim> > INodeContainer_t;
    INodeContainer_t inodes;
    GlobalIDDataBase gidStore;
    int id1 = a1.engine().layout().ID();
    int id2 = a2.engine().layout().ID();
    int id3 = a3.engine().layout().ID();
    int key = GlobalIDDataBase::nullNodeKey();
    a3.engine().layout().touches(domain, std::back_inserter(inodes),
				 TouchesConstructINode<Dim>(id3, key,
							    &gidStore));
    gidStore.shared(id1, id3);
    gidStore.shared(id2, id3);

    a2.engine().fillGuards();
    a3.engine().fillGuards();

    typename INodeContainer_t::const_iterator i = inodes.begin();
    while (i != inodes.end())
    {
      INode<Dim> node = extendDomain(*i, function);
      GuardedPatchEvaluator<MainEvaluatorTag>::createIterate(a1(node),
							     a2(node),
							     a3(node),
							     function);
      ++i;
    }
  }
};


//////////////////////////////////////////////////////////////////////

#endif     // POOMA_EVALUATOR_GUARDEDPATCHEVALUATOR_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ExtendedPatchEvaluator.h,v $   $Author: richard $
// $Revision: 1.3 $   $Date: 2004/11/01 18:15:51 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
