// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// Contents:
// Example POOMA II program using PAWS: Sender.cpp
//-----------------------------------------------------------------------------

/***************************************************************************
 *
 * Sender will send a scalar and a 2D Array to another program (or set
 * of programs) that wish to receive them.  The scalar that is send stores
 * the number of iterations that will be performed.
 *
 ***************************************************************************/

#include "Pooma/Pooma.h"
#include "Pooma/Arrays.h"

#if POOMA_PAWS
#include "Pooma/Paws.h"
#endif // POOMA_PAWS

int main(int argc, char *argv[])
{
  // Initialize POOMA and create an output stream.

  Pooma::initialize(argc, argv);
  Inform msg("Sender");

#if POOMA_PAWS

  // Initialize PAWS.

  msg << "Starting PAWS ..." << std::endl;
  Connection<Paws> paws("Sender", argc, argv);

  // Create a 2D Array A.

  Interval<2> size(10, 20);
  Array<2, double, Brick> A(size);
  A = 0.0;
  Pooma::blockAndEvaluate();

  // Register the POOMA Array with PAWS for sharing.

  msg << "Registering data ..." << std::endl;
  ConnectorBase *ap = paws.connect("A", A, ConnectionBase::out);

  // Initialize and register a scalar indicating how many iterations to do

  int iterations = 100;
  ConnectorBase *ip = paws.connectScalar("iters", iterations,
					 ConnectionBase::out);

  // Indicate we're ready to continue; we'll wait here for the Controller.

  msg << "Waiting for ready signal ..." << std::endl;
  paws.ready();

  // Let the other side know how many iterations to perform, by sending the
  // number of iterations.

  msg << "Sending number of iterations ..." << std::endl;
  paws.update(ip);

  // For several iterations, update A, and send it out.

  msg << "Sending for " << iterations << " iterations ..." << std::endl;
  for (int iter = 0 ; iter < iterations ; iter++ )
    {
      // Just add one to each element.

      A += 1.0;
      Pooma::blockAndEvaluate();
      
      // Send it out now.

      msg << "Sending on iteration " << iter << " ..." << std::endl;
      paws.update(ap);
    }

  // We're done; shut down connections and exit.

  msg << "Shutting down PAWS ..." << std::endl;
  paws.close();

#else // POOMA_PAWS

  msg << "Please configure with --paws to use this example code!"
      << std::endl;

#endif // POOMA_PAWS

  Pooma::finalize();
  return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Sender.cpp,v $   $Author: richard $
// $Revision: 1.7 $   $Date: 2004/11/01 18:15:44 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
