// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

// ----------------------------------------------------------------------
// This example illustrates the explicit use of threads to perform part
// of an algorithm written in POOMA.  In this case, Jacobi iteration is
// performed on a multi-patch array and the result is summed using
// threads.
//
// This example cannot be compiled without thread support.  Building
// POOMA on top of SMARTS provides the sufficient interface.
// (Configure with --parallel.)
// ----------------------------------------------------------------------

#include "Pooma/Arrays.h"
#include "Accumulate.h"

#include <iostream>

template<class T1, class T2, class E1, class E2>
void
ApplyJacobi(Array<2,T1,E1> x, Array<2,T2,E2> b, const Range<2>& IJ)
{
  x(IJ) = 0.25*(x(IJ+Loc<2>(1,0)) + x(IJ+Loc<2>(-1,0)) 
              + x(IJ+Loc<2>(0,1)) + x(IJ+Loc<2>(0,-1)) - b(IJ));
}

template<class T1, class E1>
void
ApplyPeriodic(Array<2,T1,E1> x)
{
  // Get the horizontal and vertical extents of the domain.
  Interval<1> I = x.domain()[0];
  Interval<1> J = x.domain()[1];

  // Copy each of the four slices in turn.
  x(I.first(),J) = x(I.last()-1,J);
  x(I.last(),J) = x(I.first()+1,J);
  x(I,J.first()) = x(I,J.last()-1);
  x(I,J.last()) = x(I,J.first()+1);
}

int main(int argc, char* argv[])
{
  Pooma::initialize(argc,argv);

#if POOMA_THREADS

  // The size of each side of the domain. Must be even.
  const int N=18;

  // The calculation domain.
  Interval<2> calc( Interval<1>(1,N) , Interval<1>(1,N) );

  // The domain with guard cells on the boundary.
  Interval<2> guarded( Interval<1>(0,N+1) , Interval<1>(0,N+1) );

  // The type of array.
  typedef Array<2,double,MultiPatch<UniformTag,Brick> > A2;

  // The array we'll be solving for.
  UniformGridLayout<2> guardedLayout(guarded,Loc<2>(4,4), ReplicatedTag());
  A2 x( guardedLayout );
  x = 0.0;

  // The right hand side of the equation.
  UniformGridLayout<2> calcLayout(guarded,Loc<2>(5,5), ReplicatedTag());
  A2 b( calcLayout );
  b = 0.0;
  Pooma::blockAndEvaluate();
  b(3*N/4,N/4) = -1.0;
  b(N/4,3*N/4) =  1.0;

  // The interior domain, now with stride 2.
  Range<2> IJ( Range<1>(1,N-1,2), Range<1>(1,N-1,2) );

  // Iterate 20 times.
  for (int i=0; i<20; ++i)
  {
    ApplyJacobi(x,b,IJ);
    ApplyJacobi(x,b,IJ+Loc<2>(1,0));
    ApplyJacobi(x,b,IJ+Loc<2>(0,1));
    ApplyJacobi(x,b,IJ+Loc<2>(1,1)); 
    ApplyPeriodic(x);
  }

  // Print out the sum of the result.
  std::cout << "Final sum: " << accumulate(x) << std::endl;

#else // POOMA_THREADS

  std::cout << "Please configure with --threads to use this example code!"
	    << std::endl;

#endif // POOMA_THREADS

  Pooma::finalize();
  return 0;
}


// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: UMPJacobi.cpp,v $   $Author: richard $
// $Revision: 1.15 $   $Date: 2004/11/01 18:15:43 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
