// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

//-----------------------------------------------------------------------------
// Oscillation, a simple Particles test code.  This code tracks particles
// in a 1D harmonic oscillator potential.  It uses an analytic formula to
// compute the force on each particle at each timestep, so no particle-field
// interpolation is required.  We start the particles with random positions
// and velocities within the well and advance them in time using a simple
// leapfrog scheme, since dx/dt = v and dv/dt depends only on x.
//-----------------------------------------------------------------------------

#include "Pooma/Particles.h"
#include "Pooma/DynamicArrays.h"

#include <iostream>
#include <stdlib.h>

// Dimensionality of this problem

static const int PDim = 1;

// A traits class specifying the engine and layout of a Particles class.

template <class EngineTag>
struct PTraits
{
  // The type of engine to use in the particle attributes.
  typedef EngineTag AttributeEngineTag_t;

  // The type of particle layout to use.  Here we use a UniformLayout,
  // which divides the particle data up so as to have an equal number
  // on each patch.
  typedef UniformLayout ParticleLayout_t;
};

// A Particles subclass that defines position and velocity attributes.

template <class PT>
class Quanta : public Particles<PT>
{
public:
  // Useful things to extract from the base class
  typedef Particles<PT>                         Base_t;
  typedef double                                AxisType_t;
  typedef typename Base_t::ParticleLayout_t     ParticleLayout_t;
  typedef typename Base_t::AttributeEngineTag_t AttributeEngineTag_t;
  enum { dimensions = PDim };

  // Constructor sets up layouts and registers attributes
  Quanta(const ParticleLayout_t &pl)
  : Particles<PT>(pl)
  {
    this->addAttribute(x);
    this->addAttribute(v);
  }

  // X position and velocity are attributes (would normally be
  // private, with accessor methods)
  DynamicArray< Vector<dimensions, AxisType_t>, AttributeEngineTag_t > x;
  DynamicArray< Vector<dimensions, AxisType_t>, AttributeEngineTag_t > v;
};

// Engine tag type for attributes.  Here we use a MultiPatch engine
// with the patches being Dynamic, and a DynamicTag, which allows
// the patches to change sizes dynamically.  This is important
// since we may not have the same number of particles in each patch.
#if POOMA_CHEETAH
typedef MultiPatch< DynamicTag, Remote<Dynamic> > AttrEngineTag_t;
#else
typedef MultiPatch<DynamicTag, Dynamic> AttrEngineTag_t;
#endif

// The particle traits class and layout type for this application
typedef PTraits<AttrEngineTag_t> PTraits_t;
typedef PTraits_t::ParticleLayout_t PLayout_t;

// Simulation control constants
const double omega      = 2.0;
const double dt         = 1.0 / (50.0 * omega);
const int nParticle     = 100;
const int nPatch        = 16;
const int nIter         = 500;
const int nPrintFreq    = 25;

// Main simulation routine.
int main(int argc, char *argv[])
{
  // Initialize POOMA and Inform object for output to terminal
  Pooma::initialize(argc, argv);
  Inform out(argv[0]);
  out << "Begin Oscillation example code" << std::endl;

  // Create a uniform layout object to control particle positions.
  PLayout_t layout(nPatch);

  // Create Particles, using our special subclass and the layout
  typedef Quanta<PTraits_t> Particles_t;
  Particles_t p(layout);

  // Create particles on one patch, then re-distribute (just to show off)
  if (Pooma::context() == 0)
    p.create(nParticle, 0);
  else
    p.create(0);

  int ipatch;
  int nPatchLocal = p.attributeLayout().sizeLocal();
  for (ipatch=0; ipatch<nPatchLocal; ++ipatch)
  {
    out << "Current size of patch " << ipatch << " = "
        << p.attributeLayout().patchDomain(ipatch).size()
        << std::endl;
  }

  out << "Resyncing particles object ... " << std::endl;
  p.sync(p.x);

  // Show re-balanced distribution.
  for (ipatch=0; ipatch<nPatchLocal; ++ipatch)
  {
    out << "Current size of patch " << ipatch << " = "
        << p.attributeLayout().patchDomain(ipatch).size()
        << std::endl;
  }

  // Randomize positions in domain [-1,+1], and set velocities to zero.
  // This is done with a loop because POOMA does not yet have RNGs.
  typedef Particles_t::AxisType_t Coordinate_t;
  Vector<PDim, Coordinate_t> initPos;
  srand(12345U);
  Coordinate_t ranmax = static_cast<Coordinate_t>(RAND_MAX);
  for (int ip=0; ip<nParticle; ++ip)
  {
    for (int idim=0; idim<PDim; ++idim)
    {
      initPos(idim) = 2 * (rand() / ranmax) - 1;
    }
    p.x(ip) = initPos;
    p.v(ip) = Vector<PDim, Coordinate_t>(0.0);
  }

  // print initial state
  out << "Time = 0.0:" << std::endl;
  out << "Quanta positions:" << std::endl << p.x << std::endl;
  out << "Quanta velocities:" << std::endl << p.v << std::endl;

  // Advance particles in each time step according to:
  //         dx/dt = v
  //         dv/dt = -omega^2 * x
  for (int it=1; it<=nIter; ++it)
  {
    p.x = p.x + dt * p.v;
    p.v = p.v - dt * omega * omega * p.x;
    if (it%nPrintFreq == 0) {
      out << "Time = " << it*dt << ":" << std::endl;
      out << "Quanta positions:" << std::endl << p.x << std::endl;
      out << "Quanta velocities:" << std::endl << p.v << std::endl;
    }
  }

  // Finalize POOMA
  Pooma::finalize();
  return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Oscillation.cpp,v $   $Author: richard $
// $Revision: 1.15 $   $Date: 2004/11/01 18:15:40 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
