// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

//-----------------------------------------------------------------------------
// Contents:
//   1D FFT written using indirection arrays.
//
// This example illustrates the use of indirection arrays and IndexFuncion
// Engines to compute a 1D FFT for a power of 2.
//
// (This code is not an efficient implementation.)
//-----------------------------------------------------------------------------

#include "Pooma/Arrays.h"
#include <cmath>
#include <iostream>

// Define constant value of PI.

static const double Pi = acos(-1.0);

// ShuffleMap describes the butterfly map for performing FFT's.

struct ShuffleMap
{
  ShuffleMap(int n = 0)
    : degree_m(n)
  {
    nbit_m = 1 << n;
    mask1_m = nbit_m -1;
    mask2_m = ~(nbit_m | mask1_m);
  }

  int operator()(int i) const
  {
    return
      (mask2_m & i)
      | ( (mask1_m & i) << 1 )
      | ( (nbit_m & i) ? 1 : 0 );
  }

  int nbit_m,mask1_m,mask2_m,degree_m;
};

// LeftMap and RightMap are used to construct the solution from the
// solutions at the next level down.

struct LeftMap
{
  LeftMap(int n = 0)
    : nbit_m(~(1 << n))
  { }

  int operator()(int i) const
  {
    return (nbit_m & i);
  }

  int nbit_m;
};

struct RightMap
{
  RightMap(int n = 0)
    : nbit_m(1 << n)
  { }

  int operator()(int i) const
  {
    return (nbit_m | i);
  }

  int nbit_m;
};

// The trig factors for computing the FFT.  (It would be more
// time-efficient to precompute the values, but this functor uses no
// storage.)

struct TrigFactor
{
  TrigFactor(int n = 0)
    : n_m(1 << n)
  { }

  complex<double> operator()(int i) const
  {
    return complex<double>(cos(Pi*i/n_m), sin(Pi*i/n_m));
  }

  int n_m;
};

// Perform the FFT for a given level.

void fft(const Array<1,complex<double>,Brick> &array, int level)
{
  Interval<1> domain = array.domain();

  if (level > 0)
  {
    Array<1, int,             IndexFunction<LeftMap> >    left(domain);
    Array<1, int,             IndexFunction<RightMap> >   right(domain);
    Array<1, int,             IndexFunction<ShuffleMap> > shuffle(domain);
    Array<1, complex<double>, IndexFunction<TrigFactor> > trig(domain);

    left.engine().setFunctor(LeftMap(level));
    right.engine().setFunctor(RightMap(level));
    shuffle.engine().setFunctor(ShuffleMap(level));
    trig.engine().setFunctor(TrigFactor(level));

    // Shuffle values, compute n/2 length ffts, combine results.
    array = array(shuffle);
    fft(array, level-1);
    array = array(left) + trig*array(right);
  }
  else
  {
    int size = domain.size();
    Range<1> left (0, size-2, 2),
             right(1, size-1, 2);

    array(left) += array(right);
    array(right) = array(left) - 2.0 * array(right);
  }
}

// Perform the fft on array (array size must be a power of 2).

void fft(const Array<1, complex<double>, Brick> &array)
{
  int size = array.domain().size();

  // Determine size as power of 2
  int level = -1;
  while (size > 1)
  {
    PAssert(!(size & 1));
    ++level;
    size /= 2;
  }

  if (level >= 0)
    fft(array, level);
}

//-----------------------------------------------------------------------------
// Main program.
//-----------------------------------------------------------------------------

int main(int argc, char* argv[])
{
  Pooma::initialize(argc, argv);

  int size = 16;

  Array<1, complex<double>, Brick> a(size);

  int i;
  for (i = 0; i < size; ++i)
  {
    a(i) = sin(4*i*Pi/size);
  }

  std::cout << a << std::endl;

  fft(a);

  std::cout << a << std::endl;

  Pooma::finalize();
  return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: FFT.cpp,v $   $Author: richi $
// $Revision: 1.6 $   $Date: 2004/11/10 22:28:41 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
