// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

//-----------------------------------------------------------------------------
//
// Gyrokinetic Poisson equation example
//
// Zhihong Lin's GK Poisson equation (Phys. Rev. E, V.55, p.5646)
// reduces to a 2D stencil in slab geometry if the temperature and
// magnetic field are uniform, if a single "ring" is used, and if the
// grid-spacing is chosen to be a gyroradius. This example code solves
// the resulting equation iteratively. This was primarily written to
// compare with the GTC++ PoissonSolver class, which solves the more
// general problem.
//
// The code prompts for input parameters, which are:
//
//    n        - gridsize
//    gamma0   - <\Gamma_0>, which is "approximately" phi/density;
//               This is an acceleration parameter and drops out of
//               the equation at convergence.
//    tol      - convergence tolerance.
//    maxiters - number of iterations to run for.
//
// This code performs the iteration with both straight expression 
// templates, and with a stencil operator, defined in GKPoissonStencil.h. 
// Initial tests don't show a huge speedup for the stencil operator.
// To get maximal speedup, the compiler would have to unroll the loop
// five times, which it is not doing yet.
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Include files
//-----------------------------------------------------------------------------

#include "Pooma/Arrays.h"
#include "GKPoissonStencil.h"

#include <iomanip>
#include <iostream>
#include <cmath>

//-----------------------------------------------------------------------------
// User function used in calculating residual
//-----------------------------------------------------------------------------

class SqrFunc
{
public:
  SqrFunc() {}
  double operator()(double y) const { return y*y; }
};

UserFunction<SqrFunc> sqr;

//-----------------------------------------------------------------------------
// Main program.
//-----------------------------------------------------------------------------

int main(int argc, char* argv[])
{
  // Initialize pooma.

  Pooma::initialize(argc,argv);

  // Create arrays

  Array<2> phi, density, phi_old, tmp;

  // Input problem size

  int n;
  std::cout << "Grid size (10-1000)    : ";
  std::cin >> n;
  double gamma0;
  std::cout << "Gamma_0 (0.0-1.0)      : ";
  std::cin >> gamma0;
  double tol;
  std::cout << "Convergence tolerance  : ";
  std::cin >> tol;
  int maxiters;
  std::cout << "Max iterations (10-100): ";
  std::cin >> maxiters;
  std::cout << std::endl;

  // Domain that we're solving the equations on:

  Interval<1> N(n);
  Interval<2> domain(N,N);

  // Domain with guard cells 
  // (will impose phi = 0 in boundary region).

  Interval<1> NGC(-2,n+1);
  Interval<2> gcDomain(NGC,NGC);

  density.initialize(domain);
  tmp.initialize(domain);

  phi.initialize(gcDomain);
  phi_old.initialize(gcDomain);

  // Domains for stencil expression:

  Interval<1> I(n), J(n);

  // Constants that appear in the equation:

  const double phifact = 1.0 / (2.0 - gamma0);
  const double ijfact  = 0.25 - gamma0;

  //---------------------------------------------------------------------------
  // First do the calculation with straight expression templates.
  //---------------------------------------------------------------------------

  // Reset array element values

  density = tmp = 0;

  // Initialize phi to zero. 
  // This is here to initialize the "guard cells".

  phi_old = phi = 0;

  int mp = n / 2;

  // Let evaluators catch up before doing scalar access.

  Pooma::blockAndEvaluate();

  density(mp,mp) = 1000.0;

  // Initial guess:

  phi(I,J) = gamma0 * density(I,J);

  // GK Poisson iterations

  Pooma::blockAndEvaluate();

  std::cout << "GK Poisson Solve using expression ..." << std::endl;
  std::cout << "iter =    0,    phi_mid = "
#if !POOMA_MISSING_IOMANIPS
       << std::fixed
#endif
       << std::setprecision(3) << phi(mp,mp) << std::endl;

  int i;
  for (i = 0; i < maxiters; ++i)
    {
      phi_old = phi;

      tmp(I,J) = density(I,J) 

	+ ijfact * phi(I,J)

	+ 0.125  * ( phi(I+1,J+1) + phi(I+1,J-1) +
		     phi(I-1,J+1) + phi(I-1,J-1) )

	+ 0.0625 * ( phi(I  ,J+2) + phi(I  ,J-2) +
		     phi(I+2,J  ) + phi(I-2,J  ) );

      phi(I,J) = phifact * tmp(I,J);

      double res = sqrt(sum(sqr(phi(I,J)-phi_old(I,J))));

      std::cout << "iter = " << std::setw(4) << i+1 
	   << ",    phi_mid = "
#if !POOMA_MISSING_IOMANIPS
           << std::fixed
#endif
           << std::setprecision(3) << phi(mp,mp) 
	   << ",    residual = "
#if !POOMA_MISSING_IOMANIPS
           << std::scientific
#endif
           << res << std::endl;

      if (res <= tol) break;
    }

  std::cout << std::endl;

  //---------------------------------------------------------------------------
  // Now do it with GKPoissonStencil
  //---------------------------------------------------------------------------

  // Reset array element values

  density = tmp = 0;
  phi = 0;

  // Let evaluators catch up before doing scalar access.

  Pooma::blockAndEvaluate(); 

  density(mp,mp) = 1000.0;

  // Initial guess:

  phi(I,J) = gamma0 * density(I,J);

  // GK Poisson iterations

  Pooma::blockAndEvaluate();

  std::cout << "GK Poisson Solve using stencil ..." << std::endl;
  std::cout << "iter =    0,    phi_mid = "
#if !POOMA_MISSING_IOMANIPS
       << std::fixed
#endif
       << std::setprecision(3) << phi(mp,mp) << std::endl;

  Stencil<GKPoissonStencil> stencil(gamma0);

  for (i = 0; i < maxiters; ++i)
    {
      phi_old = phi;

      tmp(I,J) = density(I,J) + stencil(phi);

      phi(I,J) = phifact * tmp(I,J);

      double res = sqrt(sum(sqr(phi(I,J)-phi_old(I,J))));

      std::cout << "iter = " << std::setw(4) << i+1 
	   << ",    phi_mid = "
#if !POOMA_MISSING_IOMANIPS
           << std::fixed
#endif
           << std::setprecision(3) << phi(mp,mp)
	   << ",    residual = "
#if !POOMA_MISSING_IOMANIPS
           << std::scientific
#endif
           << res << std::endl;

      if (res <= tol) break;
    }

  //---------------------------------------------------------------------------
  // All done. Shut down Pooma.
  //---------------------------------------------------------------------------

  Pooma::finalize();

  return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: GKPoisson.cpp,v $   $Author: richi $
// $Revision: 1.20 $   $Date: 2004/11/10 22:28:41 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
