// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

// ----------------------------------------------------------------------------
// Scalar advection example, illustrating use of Field, UniformRectilinearMesh,
// and the canned second-order divergence operator divCellToCell()
// ----------------------------------------------------------------------------

#include "Pooma/Fields.h"

#if POOMA_CHEETAH
  typedef DistributedTag LayoutTag_t;
  typedef Remote<CompressibleBrick> BrickTag_t;
#else
  typedef ReplicatedTag LayoutTag_t;
  typedef CompressibleBrick BrickTag_t;
#endif

using namespace std;

int main(int argc, char *argv[])
{
  // Set up the library
  
  Pooma::initialize(argc,argv);

  // Create the physical domains

  const int Dim    = 1;
  const int nVerts = 128;
  const int nCells = nVerts - 1;
  int d;
  Interval<Dim> vertexDomain;
  for (d = 0; d < Dim; d++) { vertexDomain[d] = Interval<1>(nVerts); }

  // Create some layouts.
  
  Loc<Dim> patches(8);
  UniformGridLayout<Dim> layout(vertexDomain, patches, 
				GuardLayers<Dim>(1), GuardLayers<Dim>(1), 
				LayoutTag_t());
  UniformGridLayout<Dim> layoutNG(vertexDomain, patches, LayoutTag_t());
  typedef MultiPatch<UniformTag, BrickTag_t> MP_t;
  
  // Create the (uniform, logically rectilinear) mesh.
  
  Vector<Dim> origin(0.0), spacings(0.2);
  typedef UniformRectilinearMesh<Dim> Mesh_t;
  Mesh_t mesh(layout, origin, spacings);
  Mesh_t meshNG(layoutNG, origin, spacings);
  
  // Create the centering.
  
  Centering<Dim> cell = canonicalCentering<Dim>(CellType, Continuous);
  
  // Create the Fields:

  // The flow Field u(x,t):
  
  Field<Mesh_t, double, MP_t> u(cell, layout, mesh);
  Field<Mesh_t, Vector<Dim>, MP_t> dtvu(cell, layout, mesh);
  
  // The same, stored at the previous timestep for staggered leapfrog
  // plus a useful temporary:
  
  Field<Mesh_t, double, MP_t> uPrev(cell, layoutNG, meshNG), 
    uTemp(cell, layoutNG, meshNG);

  // Initialize Fields to zero everywhere, even global guard layers:
  
  u.all() = 0.0;

  // Set up periodic boundary conditions on all mesh faces:
  
  Pooma::addAllPeriodicFaceBC(u);

  // Load initial condition u(x,0), a symmetric pulse centered around nCells/4
  // and decaying to zero away from nCells/4 all directions, with a height of
  // 1.0, with a half-width of nCells/8:
  
  const double pulseWidth = spacings(0)  * nCells / 8;
  Loc<Dim> pulseCenter;
  for (d = 0; d < Dim; d++) { pulseCenter[d] = Loc<1>(nCells / 4); }
  
  PositionsTraits<Mesh_t>::Type_t ux = positions(u);
  Vector<Dim> u0 = ux.read(pulseCenter);
  u = 1.0 * exp(-dot(ux - u0, ux - u0) / (2.0 * pulseWidth));

  // Output the initial field on its physical domain:

  Inform iout;
  iout << "Time = 0:\n";
  iout << u() << endl;

  const Vector<Dim> v(0.2);   // Propagation velocity
  const double dt = 0.1;      // Timestep

  // Prime the leapfrog by setting the field at the previous timestep using the
  // initial conditions:

  uPrev = u;

  // Do a preliminary timestep using forward Euler, using the canned POOMA
  // stencil-based divergence operator div() for the spatial difference:

  dtvu.all() = dt * v * u.all();
  u -= divCellToCell(dtvu);

  // Now use staggered leapfrog (second-order) for the remaining timesteps.
  // The spatial derivative is just the second-order finite difference in the
  // canned POOMA stencil-based divergence operator div():
  
  for (int timestep = 2; timestep <= 1000; timestep++)
    {
      uTemp = u;
      dtvu.all() = dt * v * u.all();
      u = uPrev - 2.0 * divCellToCell(dtvu);
      if ((timestep % 100) == 0)
	{
	  // Output the field on its physical domain at the current timestep:
	  
	  iout << "Time = " << timestep*dt << ":\n";
	  iout << u() << endl;
	}
      uPrev = uTemp;
    }
  
  Pooma::finalize();
  return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ScalarAdvection.cpp,v $   $Author: richard $
// $Revision: 1.12 $   $Date: 2004/11/01 18:15:29 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
