// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

// ----------------------------------------------------------------------------
// Jacobi Laplace solver illustrating the use of a field with a multi-patch
// engine and the implementation of custom boundary conditions. Written in
// a dimension-independent way: set Dim constant to 1, 2, or 3.
// ----------------------------------------------------------------------------

#include "Pooma/Fields.h"
#include "Utilities/Clock.h"

#include <iostream>

// Set dimension.

const int Dim = 2;

// Convenience typedefs.

typedef ConstField<
  DiscreteGeometry<Vert, UniformRectilinearMesh<Dim> >,
  double, MultiPatch<UniformTag, Brick> > ConstFieldType_t;

typedef Field<
  DiscreteGeometry<Vert, UniformRectilinearMesh<Dim> >,
  double, MultiPatch<UniformTag, Brick> > FieldType_t;

// The stencil.
  
class Laplacian
{
public:

  typedef Vert OutputCentering_t;
  typedef double OutputElement_t;

  int lowerExtent(int) const { return 1; }
  int upperExtent(int) const { return 1; }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1) const
  {
    return 0.5 * (f(i1 + 1) + f(i1 - 1));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2) const
  {
    return 0.25 * (f(i1 + 1, i2) + f(i1 - 1, i2) + 
		   f(i1, i2 + 1) + f(i1, i2 - 1));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2, int i3) const
  {
    return (1.0/6.0) * (f(i1 + 1, i2, i3) + f(i1 - 1, i2, i3) + 
		        f(i1, i2 + 1, i3) + f(i1, i2 - 1, i3) +
		        f(i1, i2, i3 + 1) + f(i1, i2, i3 - 1));
  }
};

View1<FieldStencil<Laplacian>, ConstFieldType_t>::Type_t 
laplacian(const ConstFieldType_t &f)
{
  return FieldStencil<Laplacian>()(f);
}

// The boundary condition.

class PositionFaceBC : public BCondCategory<PositionFaceBC>
{
public:

  PositionFaceBC(int face) : face_m(face) { }

  int face() const { return face_m; }

private:

  int face_m;
};

template<>
class BCond<FieldType_t, PositionFaceBC> : public FieldBCondBase<FieldType_t>
{
public:

  BCond(const FieldType_t &f, const PositionFaceBC &bc)
  : FieldBCondBase<FieldType_t>
      (f, f.totalDomain()), bc_m(bc) { }

  void applyBoundaryCondition()
  {
    int d = bc_m.face() / 2;
    int hilo = bc_m.face() & 1;
    int layer;
    Interval<FieldType_t::dimensions> domain(subject().totalDomain());
    if (hilo)
      layer = domain[d].last();
    else
      layer = domain[d].first();
      
    domain[d] = Interval<1>(layer, layer);
    subject()(domain) = 100.0 * subject().x(domain).comp(0);
    for (int i = 1; i < FieldType_t::dimensions; i++)
      subject()(domain) *= subject().x(domain).comp(i);
  }
  
  FieldBCondBase<FieldType_t> *retarget(const FieldType_t &f) const
  {
    return new BCond<FieldType_t, PositionFaceBC>(f, bc_m);
  }

private:

  PositionFaceBC bc_m;
};

class AllPositionFaceBC
{
public:

  void operator()(FieldType_t &f) const
    {
      for (int i = 0; i < 2 * FieldType_t::dimensions; i++)
        f.addBoundaryCondition(PositionFaceBC(i));
    }
};

int main(
    int argc,
    char *argv[]
){
    // Set up the library
    Pooma::initialize(argc,argv);

    // Create the physical domains:

    // Set the dimensionality:
    const int nVerts = 20;
    Loc<Dim> center, blocks;
    Interval<Dim> vertexDomain;
    Vector<Dim> origin, spacings;
    int i;
    
    Pooma::blockAndEvaluate();
    for (i = 0; i < Dim; i++)
      {
        center[i] = Loc<1>(nVerts / 2);
        blocks[i] = 2;
        vertexDomain[i] = Interval<1>(nVerts);
        origin(i) = 1.0 / (nVerts + 1);
        spacings(i) = 1.0 / (nVerts + 1);
      }

    // Create the (uniform, logically rectilinear) mesh.
    typedef UniformRectilinearMesh<Dim> Mesh_t;
    Mesh_t mesh(vertexDomain, origin, spacings);
    
    // Create a geometry object with 1 guard layer to account for
    // stencil width:
    typedef DiscreteGeometry<Vert, UniformRectilinearMesh<Dim> > Geometry_t;
    Geometry_t geom(mesh, GuardLayers<Dim>(1));

    // Create the Fields:

    // The voltage v and a temporary vTemp.
    UniformGridLayout<Dim> layout(vertexDomain, blocks, 
      GuardLayers<Dim>(1), GuardLayers<Dim>(1),ReplicatedTag());
    FieldType_t v(geom, layout), vTemp(geom, layout);

    // Set up position-dependent boundary conditions on all mesh faces:
    v.addBoundaryConditions(AllPositionFaceBC());
    vTemp.addBoundaryConditions(AllPositionFaceBC());
    
    // Construct the analytic solution:
    double analytic = 100.0;
    for (i = 0; i < Dim; i++)
      {
        analytic *= v.x(center)(i);
      }

    // Start timing:
    Pooma::Clock clock;
    double start = clock.value();

    // Load initial condition:
    v = 0.0;
    
    // Perform the Jacobi iteration. We apply the Jacobi formula twice
    // each loop:
    double error = 1000;
    int iteration = 0;
    while (error > 1e-6)
      {     
        iteration++;

        vTemp = laplacian(v);
        v = laplacian(vTemp);
        
        // Make sure calculations are done prior to scalar calculations.
        Pooma::blockAndEvaluate();
                 
        const double solution = v(center);
        error = abs(solution - analytic);
        if (iteration % 1000 == 0)
          std::cout << "Iteration: " << iteration << "; "
              << "Error: " << error << std::endl;
      }

    std::cout << "Wall clock time: " << clock.value() - start << std::endl;
    std::cout << "Iteration: " << iteration << "; " 
	      << "Error: " << error << std::endl;
    
    Pooma::finalize();
    return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Laplace.cpp,v $   $Author: richard $
// $Revision: 1.17 $   $Date: 2004/11/01 18:15:27 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
