// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

#ifndef POOMA_BENCHMARKS_SOLVERS_JACOBI_JACOBIINCPPTRAN_H
#define POOMA_BENCHMARKS_SOLVERS_JACOBI_JACOBIINCPPTRAN_H

// include files

#include "Pooma/Arrays.h"
#include "Utilities/Benchmark.h"


//-----------------------------------------------------------------------------
// JacobiInCppTran is a C++-tran implementation of the Jacobi iterative solver
//-----------------------------------------------------------------------------

class JacobiInCppTran : public Implementation {
public:
  
  typedef Array<2, double, Brick> Array2D;
  
  //---------------------------------------------------------------------------
  // We are a CppTran implementation.

  const char* type() const { return CppTranType(); }

  //---------------------------------------------------------------------------
  // We need to initialize the problem for a specific size.

  void initialize(int n) {
    // set array sizes
    Interval<1> I(1,n), J(1,n);
    Interval<2> newDomain(I,J);
    x0_m.initialize(newDomain);
    x1_m.initialize(newDomain);
    b_m.initialize(newDomain);

    // save problem size
    n_m = n;
  }

  //---------------------------------------------------------------------------
  // Runs the benchmark.

  void run() {
    int i,j;
    // reset array values
    for (j=1; j<=n_m; ++j) {
      for (i=1; i<=n_m; ++i) {
        x0_m(i,j) = 0.0;
        x1_m(i,j) = 0.0;
      }
    }
    x0_m(n_m/2,n_m/2) = 1.0;
    // set up for Jacobi iteration
    iters_m = 0;
    double dx = 1.0 / (n_m-1);
    double dt = 0.1 * dx;

    // The coefficients for the linear operator.
    double A2 = 1.0/(6.0*dx*dx);
    double A1 = 4.0*A2;
    double A0 = 4.0*(A1+A2);
    double h0 = 1.0 - dt*theta_s*A0;
    double h1 = dt*theta_s*A1;
    double h2 = dt*theta_s*A2;

    // The scaling factor for getting the constant offset part of 
    // the update from the right hand side of the equation.
    double dtth = dt*(1.0-theta_s);
    double e0 = 1.0 + dtth*A0;
    double e1 = dtth*A1;
    double e2 = dtth*A2;
    double c0 = phi_s/(1.0+dtth*A0);

    // The coefficients for the update matrix.
    double gf = phi_s*dtth/(1.0+dtth*A0);
    double g0 = 1.0-phi_s;
    double g1 = gf*A1;
    double g2 = gf*A2;
  
    for (double t=0; t<tfinal_s; t+=2*dt) {
      // Calculate the right hand side.
      for (j=2; j<=n_m-1; ++j) {
        for (i=2; i<=n_m-1; ++i) {
          b_m(i,j) = h0 * x0_m(i,j) + 
                     h1 * (x0_m(i-1,j) + x0_m(i+1,j) +
                           x0_m(i,j+1) + x0_m(i,j-1)) +
                     h2 * (x0_m(i-1,j-1) + x0_m(i+1,j-1) +
                           x0_m(i-1,j+1) + x0_m(i+1,j+1));
        }
      }
      double err;
      do {
        // Do a double iteration to use the allocated memory efficiently.
        for (j=2; j<=n_m-1; ++j) {
          for (i=2; i<=n_m-1; ++i) {
            x1_m(i,j) = c0 * b_m(i,j) + g0 * x0_m(i,j) +
	                g1 * (x0_m(i-1,j) + x0_m(i+1,j) +
                              x0_m(i,j+1) + x0_m(i,j-1)) +
	                g2 * (x0_m(i-1,j-1) + x0_m(i+1,j-1) +
                              x0_m(i-1,j+1) + x0_m(i+1,j+1));
          }
        }
        for (j=2; j<=n_m-1; ++j) {
          for (i=2; i<=n_m-1; ++i) {
            x0_m(i,j) = c0 * b_m(i,j) + g0 * x1_m(i,j) +
	                g1 * (x1_m(i-1,j) + x1_m(i+1,j) +
                              x1_m(i,j+1) + x1_m(i,j-1)) +
	                g2 * (x1_m(i-1,j-1) + x1_m(i+1,j-1) +
                              x1_m(i-1,j+1) + x1_m(i+1,j+1));
	  }
        }
        // Calculate the residual.
        // Since we don't have reductions yet, do this by hand.
        err = 0;
        for (j=2; j<=n_m-1; ++j) {
	  for (i=2; i<=n_m-1; ++i) {
	    double e = -b_m(i,j) + e0 * x0_m(i,j) -
	               e1 * (x0_m(i-1,j) + x0_m(i+1,j) +
                             x0_m(i,j+1) + x0_m(i,j-1)) -
	               e2 * (x0_m(i-1,j-1) + x0_m(i+1,j-1) +
                             x0_m(i-1,j+1) + x0_m(i+1,j+1));
	      
	    err += e*e;
	  }
        }
        ++iters_m;
      } while (err>1e-4);  
      
    }

    // save check value
    check_m = x0_m(n_m/2,n_m/2);
    // tally up the flops
    flops_m = 11 * ((double)n_m - 2) * ((double)n_m - 2) * int(tfinal_s/(2*(0.1/((double)n_m-1)))) +
              iters_m * 41 * ((double)n_m - 2) * ((double)n_m - 2);
  }    

  //---------------------------------------------------------------------------
  // Prints out the check value for this case.

  double resultCheck() const { return check_m; }
  
  //---------------------------------------------------------------------------
  // Returns the number of flops.

  double opCount() const { return flops_m; }

private:

  //---------------------------------------------------------------------------
  // Arrays.
  
  Array2D x0_m, x1_m, b_m;
  
  //---------------------------------------------------------------------------
  // Parameters.
  
  static const double tfinal_s, theta_s, phi_s;
  
  //---------------------------------------------------------------------------
  // Problem size.
  
  int n_m;
  
  //---------------------------------------------------------------------------
  // Check value.
  
  double check_m;

  //---------------------------------------------------------------------------
  // Iteration count
  
  int iters_m;

  //---------------------------------------------------------------------------
  // Flop count
  
  int flops_m;
};

// initialize static variables
const double JacobiInCppTran::tfinal_s = 0.5;
const double JacobiInCppTran::theta_s = 0.5;
const double JacobiInCppTran::phi_s = 1.0;

#endif // POOMA_BENCHMARKS_SOLVERS_JACOBI_JACOBIINCPPTRAN_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: JacobiInCppTran.h,v $   $Author: richard $
// $Revision: 1.18 $   $Date: 2004/11/01 18:15:17 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
