// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// Classes elemreadArray and elemreadCArray.
//-----------------------------------------------------------------------------

#ifndef POOMA_BENCHMARKS_SIMPLEARRAY_ELEMREAD_H
#define POOMA_BENCHMARKS_SIMPLEARRAY_ELEMREAD_H

// include files

#include "Pooma/Arrays.h"
#include "Utilities/Benchmark.h"
#include <complex>

//-----------------------------------------------------------------------------
// elemreadArray class definition.
//-----------------------------------------------------------------------------

class elemreadArray : public Implementation
{
public:

  // Typedefs for the Array types we'll be using here.

  typedef Array<1,complex<double>,Brick> Array1D;

  // This is a Pooma II benchmark

  const char* type() const { return P2Type(); }
  const char* qualification() const { return "Array"; }

  void initialize(int n) {
    // get new array domain
    Interval<1> N(1, n);

    // Reset the size of member Arrays.
    x_m.initialize(N);
    y_m.initialize(N);
    z_m.initialize(N);

    // Save the problem size.
    n_m = n;

    // Set up the initial conditions.
    setInitialConditions();
  }

  // run benchmark
  void run() {
    // computational kernel
    for (int i=1; i<=n_m; ++i)
      x_m(i) = y_m(i) + z_m(i);

    // save results for checking
    check_m = norm(x_m(n_m/2));
  }

  // Return value for checking result of benchmark run.

  double resultCheck() const { return check_m; }

  // Return number of flops in this kernel.

  double opCount() const { return (2 * (double)n_m); }

private:

  void setInitialConditions() {
    // initialize arrays
    x_m = complex<double>(0.0,0.0);
    y_m = complex<double>(3.3+n_m/1000.0,0.0);
    z_m = complex<double>(0.0,4.4+n_m/2000.0);

    Pooma::blockAndEvaluate();
  }

  // 1D Data Arrays.

  Array1D x_m, y_m, z_m;

  // Problem check value.

  double check_m;

  // Problem Size.

  int n_m;

};

//-----------------------------------------------------------------------------
// elemreadCArray class definition.
//-----------------------------------------------------------------------------

class elemreadCArray : public Implementation
{
public:

  // Typedefs for the Array types we'll be using here.

  typedef Array<1,complex<double>,Brick>      Array1D;
  typedef ConstArray<1,complex<double>,Brick> CArray1D;

  // This is a Pooma II benchmark

  const char* type() const { return P2Type(); }
  const char* qualification() const { return "ConstArray"; }

  void initialize(int n) {
    // get new array domain
    Interval<1> N(1, n);

    // Reset the size of member Arrays.
    x_m.initialize(N);

    // Save the problem size.
    n_m = n;

    // Set up the initial conditions.
    setInitialConditions();
  }

  // run benchmark
  void run() {
    // computational kernel
    for (int i=1; i<=n_m; ++i)
      x_m(i) = y_m(i) + z_m(i);

    // save results for checking
    check_m = norm(x_m(n_m/2));
  }

  // Return value for checking result of benchmark run.

  double resultCheck() const { return check_m; }

  // Return number of flops in this kernel.

  double opCount() const { return (2 * (double)n_m); }

private:

  void setInitialConditions() {
    // initialize arrays
    x_m = complex<double>(3.3+n_m/1000.0,0.0);
    Pooma::blockAndEvaluate();
    y_m.initialize(x_m.engine());
    y_m.makeOwnCopy();
    x_m = complex<double>(0.0,4.4+n_m/2000.0);
    Pooma::blockAndEvaluate();
    z_m.initialize(x_m.engine());
    z_m.makeOwnCopy();
    x_m = complex<double>(0.0,0.0);
    Pooma::blockAndEvaluate();
  }

  // 1D Data Arrays.

  Array1D x_m;
  CArray1D y_m, z_m;

  // Problem check value.

  double check_m;

  // Problem Size.

  int n_m;

};

#endif // POOMA_BENCHMARKS_SIMPLEARRAY_ELEMREAD_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: elemread.h,v $   $Author: richard $
// $Revision: 1.4 $   $Date: 2004/11/01 18:15:16 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
