// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// Class Loop18InC.  This class executes Blitz++ benchmarking loop 18
// using ANSI C arrays and elementwise expression evaluation in a for loop.
//-----------------------------------------------------------------------------

#ifndef POOMA_BENCHMARKS_BLITZLOOPS_LOOP18INC_H
#define POOMA_BENCHMARKS_BLITZLOOPS_LOOP18INC_H

// include files

#include "Utilities/PAssert.h"
#include "Utilities/Benchmark.h"

// function declarations

extern "C" {
  void
  runLoop18InC(double* x, double* a, double* b, double u, double v, int n);

  void
  runLoop18InCRestrict(double* x, double* a, double* b, double u, double v, 
		       int n);
}


//-----------------------------------------------------------------------------
// Class definition.
//-----------------------------------------------------------------------------

template<bool UseRestrict>
class Loop18InC : public Implementation {

public:

  // constructor: set data pointers to null
  Loop18InC() : x_m(NULL), a_m(NULL), b_m(NULL) {}

  // destructor
  ~Loop18InC(void) {
    // delete any existing storage
    delete [] x_m;
    delete [] a_m;
    delete [] b_m;
  }

  // this is an ANSI C benchmark (possibly using restrict)
  const char* type(void) const { return CType(); }
  const char *qualification() const
  {
    if (UseRestrict)
      return "restrict";
    else
      return "";
  }

  void initialize(int n) {
    // delete any pre-existing array storage
    delete [] x_m;
    delete [] a_m;
    delete [] b_m;

    // allocate arrays of length n
    x_m = new double[n];
    a_m = new double[n];
    b_m = new double[n];

    // check memory allocation
    PInsist(x_m != NULL, "Memory allocation failure of x_m.");
    PInsist(a_m != NULL, "Memory allocation failure of a_m.");
    PInsist(b_m != NULL, "Memory allocation failure of b_m.");

    // save problem size
    n_m = n;

    // intialize array element values
    setInitialConditions();
  }

  // run benchmark
  void run(void) {
    // run C implementation of kernel
    if (UseRestrict)
      runLoop18InCRestrict(x_m, a_m, b_m, u_m, v_m, n_m);
    else
      runLoop18InC(x_m, a_m, b_m, u_m, v_m, n_m);

    // store middle element value for checking result
    check_m = x_m[n_m/2-1];

    return;
  }

  // return value for checking result of benchmark run
  double resultCheck(void) const { return check_m; }

  // return flop count for this kernel
  double opCount(void) const { return (3 * (double)n_m); }

private:

  void setInitialConditions(void) {
    // initialize rand() random number generator
    srand(12345U);

    // loop over array elements and assign random values
    for (int i=0; i<n_m; ++i) {
      a_m[i] = rand()/(static_cast<double>(RAND_MAX)+1);
      b_m[i] = rand()/(static_cast<double>(RAND_MAX)+1);
    }

    return;
  }

  // 1D arrays
  double *x_m, *a_m, *b_m;

  // Problem check value
  double check_m;

  // Problem Size
  int n_m;

  // Scalar constants
  static const double u_m;
  static const double v_m;

};



// Assign static constants
template <bool UseRestrict>
const double Loop18InC<UseRestrict>::u_m = 0.39123982498157938742;
template <bool UseRestrict>
const double Loop18InC<UseRestrict>::v_m = 0.39123982498157938742;


#endif // POOMA_BENCHMARKS_BLITZLOOPS_LOOP18INC_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Loop18InC.h,v $   $Author: richard $
// $Revision: 1.25 $   $Date: 2004/11/01 18:15:04 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
