// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//
//-----------------------------------------------------------------------------
// Class declaration of 2D acoustic benchmark (Pooma II version).
// Based on the Blitz++ version by Todd Veldhuizen.
// See http://monet.uwaterloo.ca/blitz/benchmarks/acoustic.html.
//-----------------------------------------------------------------------------

#ifndef POOMA_BENCHMARKS_ACOUSTIC2D_ACOUSTIC2DINP2_H
#define POOMA_BENCHMARKS_ACOUSTIC2D_ACOUSTIC2DINP2_H

// Include files

#include "Pooma/Arrays.h"
#include "Utilities/Benchmark.h"

//-----------------------------------------------------------------------------
// Acoustic2dInP2 is a Pooma II implementation of the 2D acoustic benchmark.
//-----------------------------------------------------------------------------

template<class T, int Niters>
class Acoustic2dInP2 : public Implementation {
public:
  
  
  //---------------------------------------------------------------------------
  // We are a PoomaII implementation.

  const char *type() const { return P2Type(); }

  //---------------------------------------------------------------------------
  // We need to initialize the problem for a specific size.

  void initialize(int n)
  {
    Interval<2> newDomain(Interval<1>(1, n), Interval<1>(1, n));

    p1_m.initialize(newDomain);
    p2_m.initialize(newDomain);
    p3_m.initialize(newDomain);
    c_m.initialize(newDomain);
    
    n_m = n;
  }

  //---------------------------------------------------------------------------
  // Runs the benchmark.

  void run()
  {
    int iter;
    const int N = n_m;
    
    Interval<1> I(2, N - 1), J(2, N - 1);
    
    setInitialConditions();
    
    for (iter = 1; iter <= Niters; iter++)
      {
        p3_m(I, J) = (2 - 4 * c_m(I, J)) * p2_m(I, J) + c_m(I, J) *
          (p2_m(I, J - 1) + p2_m(I, J + 1) + p2_m(I - 1, J) + 
            p2_m(I + 1, J)) - p1_m(I, J);

        p1_m = p2_m;
        p2_m = p3_m;
      }            

    Pooma::blockAndEvaluate();

    check_m = p1_m(N / 2, N / 2);
  }    

  //---------------------------------------------------------------------------
  // Just run the setup.

  void runSetup()
  {
    setInitialConditions();
  }

  //---------------------------------------------------------------------------
  // Prints out the check value for this case.

  double resultCheck() const
  {
    return static_cast<double>(check_m);
  }
  
  //---------------------------------------------------------------------------
  // return flop count for this kernel.

  double opCount() const
  {
    return (9 * (n_m - 2) * (n_m - 2) * Niters);
  }


private:

  //---------------------------------------------------------------------------
  // Initialize our arrays.

  void setInitialConditions()
  {
    int i, j, blockLeft, blockRight, blockTop, blockBottom;
    int channelLeft, channelRight, channel1Height, channel2Height;
    int cr, cc;
    T s2;
    const int N = n_m;

    // Default velocity in the air.
        
    c_m = 0.2;

    // Solid block with which the pulse collides.

    blockLeft = 1;      
    blockRight = 2 * N / 5.0;
    blockTop = N / 3.0;
    blockBottom = 2 * N / 3.0;

    c_m(Interval<1>(blockTop, blockBottom), Interval<1>(blockLeft, blockRight))
      = 0.5;
 
    // Channel directing the pulse leftwards.      
    
    channelLeft = 4 * N / 5.0;
    channelRight = N;
    channel1Height = 3 * N / 8.0;
    channel2Height = 5 * N / 8.0;

    c_m(channel1Height, Interval<1>(channelLeft, channelRight)) = 0.0;
    c_m(channel2Height, Interval<1>(channelLeft, channelRight)) = 0.0;
        
    // Initial pressure distribution: a gaussian pulse inside the channel.
    
    cr = N / 2;
    cc = 7 * N / 8.0;
    s2 = 64.0 * 9.0 / ((N/2.0) * (N/2.0));

    for (j = 1; j <= N; j++)
      for (i = 1; i <= N; i++)
        {
          p1_m(i, j) = 0.0;
          p2_m(i, j) = exp(-((i-cr)*(i-cr) + (j-cc)*(j-cc)) * s2);
          p3_m(i, j) = 0.0;
        }

    Pooma::blockAndEvaluate();
  }
  
  //---------------------------------------------------------------------------
  // Arrays.
  
  Array<2, T, Brick> p1_m, p2_m, p3_m, c_m;
  
  //---------------------------------------------------------------------------
  // Problem size.
  
  int n_m;
  
  //---------------------------------------------------------------------------
  // Check value.
  
  T check_m;
};

#endif // POOMA_BENCHMARKS_ACOUSTIC2D_ACOUSTIC2DINP2_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Acoustic2dInP2.h,v $   $Author: richard $
// $Revision: 1.21 $   $Date: 2004/11/01 18:15:03 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
