// -*- C++ -*-
//
// Copyright (C) 1998, 1999, 2000, 2002  Los Alamos National Laboratory,
// Copyright (C) 1998, 1999, 2000, 2002  CodeSourcery, LLC
//
// This file is part of FreePOOMA.
//
// FreePOOMA is free software; you can redistribute it and/or modify it
// under the terms of the Expat license.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Expat
// license for more details.
//
// You should have received a copy of the Expat license along with
// FreePOOMA; see the file LICENSE.
//

#ifndef POOMA_BENCHMARKS_ABC_ABCINC_H
#define POOMA_BENCHMARKS_ABC_ABCINC_H

//-----------------------------------------------------------------------------
// Class AbcInC.
//-----------------------------------------------------------------------------

// include files

#include "Utilities/Benchmark.h"
#include "Utilities/PAssert.h"

#include <stdlib.h>

// function declarations

extern "C" {
  void runAbcSetup(double* a, double* b, double* c, int n);
  void runAbcInC(double* a, double* b, double* c, int n);
  void runAbcInCRestrict(double* a, double* b, double* c, int n);
}


//-----------------------------------------------------------------------------
// Class definitions.
//-----------------------------------------------------------------------------

template<bool UseRestrict>
class AbcInC : public Implementation {
public:

  // constructor: just set data pointers to null
  AbcInC() : a_m(NULL), b_m(NULL), c_m(NULL) {}

  // destructor: clean up allocated memory
  ~AbcInC() {
    delete [] a_m;
    delete [] b_m;
    delete [] c_m;
  }

  // This is a C benchmark.

  const char *type() const { return CType(); }
  const char *qualification() const
  {
    if (UseRestrict)
      return "restrict";
    else
      return "";
  }

  void initialize(int n) {
    // delete and reallocate the arrays
    delete [] a_m;
    delete [] b_m;
    delete [] c_m;

    a_m = new double[n * n];
    b_m = new double[n * n];
    c_m = new double[n * n];

    PInsist(a_m != NULL, "Memory allocation failure of a_m.");
    PInsist(b_m != NULL, "Memory allocation failure of b_m.");
    PInsist(c_m != NULL, "Memory allocation failure of c_m.");

    // Save problem size.
    n_m = n;
  }

  void run() {
    // run kernel
    
    if (UseRestrict)
      runAbcInCRestrict(a_m, b_m, c_m, n_m);
    else
      runAbcInC(a_m, b_m, c_m, n_m);

    // save result
    check_m =  a_m[n_m / 2 - 1 + n_m * (n_m / 2 - 1)];
  }

  void runSetup()
  {
    runAbcSetup(a_m, b_m, c_m, n_m);
  }

  // Return value for checking result of benchmark run.

  double resultCheck() const { return check_m; }

  // Return number of flops in this kernel.

  double opCount() const { return ( 40 * n_m * n_m ); }

private:

  // Data Arrays.

  double *a_m, *b_m, *c_m;

  // Problem check value.

  double check_m;

  // Problem Size.

  int n_m;

};

#endif // POOMA_BENCHMARKS_ABC_ABCINC_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ABCInC.h,v $   $Author: richard $
// $Revision: 1.3 $   $Date: 2004/11/01 18:15:02 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
