// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "mymachines.h"
#include "mymarshal.h"
#include <GL/glu.h>
#include <gtk/gtkgl.h>
#include <gdk/gdkkeysyms.h>
#include "mymachine.h"
#include "mycheckpoints.h"
#include "mytrack.h"
#include "mygadget.h"
#include "mybox.h"
#include <XL/xl.h>

#define MY_MACHINES_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE((obj), MY_TYPE_MACHINES, MyMachinesPrivate))

static
char colors[][16] = 
{
	"Red",
	"Green",
	"Blue",
	"Yellow",
	"Cyan",
	"Violet",
	"White",
	"Black"
};

static
char gadgets[3][16] =
{
	"Cube",
	"Turbo",
	"Fake"
};

typedef struct
{
	unsigned int machine;
	int keys[5];
	unsigned int checkpoint;
	float distance;
	unsigned int points;
	int item;
} Player;

typedef struct
{
	unsigned int checkpoints;
	unsigned int track;
} Race;

typedef struct
{
	unsigned int box;
	unsigned int gadget;
	int player;
	int frames;
} Item;

struct _MyMachinesPrivate
{
	GdkGLConfig *config;

	guint timeout;

	MyMachinesState state;

	dWorldID world;
	dSpaceID space;
	dJointGroupID joints;
	float aspect;

	struct
	{
		int done;
	} intro;

	struct
	{
		GLuint background;
		ALuint music;
	} main;

	struct
	{
		unsigned int frustum;
		float camera[3];
		Player players[8];
		Race race;
		Item items[20];
	} play;

	struct
	{ 
		MyMachinesLoadState state;
		unsigned int items, machines;
	} load;

	struct
	{
		char *tracks[6];
		char *machines[3];
		unsigned int nmachines;
		unsigned int nplayers;
		unsigned int nitems;
		unsigned int track;
		unsigned int machine;
		unsigned int npoints;
	} options;
};

G_DEFINE_TYPE(MyMachines, my_machines, GTK_TYPE_DRAWING_AREA);

static
void
my_machines_frame_main(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	GdkGLContext *context = gtk_widget_get_gl_context(GTK_WIDGET(self));
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(GTK_WIDGET(self));

	if(!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glBindTexture(GL_TEXTURE_2D, priv->main.background);
	glPushMatrix();
		glTranslatef((float) rand() / RAND_MAX * 0.01f - 0.005f, (float) rand() / RAND_MAX * 0.01f - 0.005f, 0.0f);
		glBegin(GL_QUADS);
			glTexCoord2i(0, 0); glVertex2f(-0.01f, -0.01f);
			glTexCoord2i(1, 0); glVertex2f(1.01f, -0.01f);
			glTexCoord2i(1, 1); glVertex2f(1.01f, 1.01f);
			glTexCoord2i(0, 1); glVertex2f(-0.01f, 1.01f);
		glEnd();
	glPopMatrix();

	gdk_gl_drawable_swap_buffers(drawable);
		
	gdk_gl_drawable_gl_end(drawable);
}

static
void
my_machines_human(MyMachines *self, unsigned int player)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);

	float position[3], target[3], direction[3], item[3];
	unsigned int i;
	int turbo = 0;

	xlPointCopy(myMachineGetPosition(priv->play.players[player].machine), position);
	xlPointCopy(myCheckPointsCheckPoint(priv->play.race.checkpoints, (priv->play.players[player].checkpoint + 1) % myCheckPointsLength(priv->play.race.checkpoints)), target);
	xlVectorSub(target, position, direction);
	xlVectorNorm(direction, &priv->play.players[player].distance);
	if(priv->play.players[player].distance < 40.0f)
	{
		priv->play.players[player].checkpoint++;
		xlPointCopy(myCheckPointsCheckPoint(priv->play.race.checkpoints, (priv->play.players[player].checkpoint + 1) % myCheckPointsLength(priv->play.race.checkpoints)), target);
		xlVectorSub(target, position, direction);
		xlVectorNorm(direction, &priv->play.players[player].distance);
	}

	if(priv->play.players[player].item == -1)
	{
		for(i = 0; i < priv->options.nitems; i++)
		{
			if(priv->play.items[i].frames == 0)
			{
				xlPointSub(position, myBoxSphere(priv->play.items[i].box), item);
				if(xlVectorNormFunc(item) <= myBoxSphere(priv->play.items[i].box)[3])
				{
					priv->play.players[player].item = i;
					priv->play.items[i].player = player;
					switch(myGadgetType(priv->play.items[i].gadget))
					{
						case MY_GADGET_CUBE: priv->play.items[i].frames = 800; break;
						case MY_GADGET_TURBO: priv->play.items[i].frames = 100; break;
						case MY_GADGET_FAKE: priv->play.items[i].frames = 200; break;
					}
					g_signal_emit_by_name(self, "my-gadget-captured", player, myGadgetType(priv->play.items[priv->play.players[player].item].gadget));
				}
			}
		}
	}
	else
	{
		if(priv->play.players[player].keys[4])
		{
			switch(myGadgetType(priv->play.items[priv->play.players[player].item].gadget))
			{
				case MY_GADGET_CUBE:
				{
					const float *r;
					r = myMachineGetRotation(priv->play.players[player].machine);
					direction[0] = -r[2]; direction[1] = -r[6]; direction[2] = -r[10];
					xlVectorScalarProductInPlace(direction, 3.0f);
					xlPointSub(position, direction, item);
					myGadgetPlace(priv->play.items[priv->play.players[player].item].gadget, item);
					myGadgetEnable(priv->play.items[priv->play.players[player].item].gadget);
					unsigned int gadget = myGadgetType(priv->play.items[priv->play.players[player].item].gadget);
					priv->play.items[priv->play.players[player].item].player = -1;
					priv->play.players[player].item = -1;
					g_signal_emit_by_name(self, "my-gadget-released", player, gadget);
				}
					break;
				case MY_GADGET_TURBO:
					turbo = 1;
					break;
				case MY_GADGET_FAKE:
					break;
			}
		}
	}

	myMachineDrive(priv->play.players[player].machine, priv->play.players[player].keys[0], priv->play.players[player].keys[1], priv->play.players[player].keys[2], priv->play.players[player].keys[3], turbo);
}

static
void
my_machines_computer(MyMachines *self, unsigned int player)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);

	float position[3], target[3], direction[3], orientation[3], item[3], plane[4], angle, variation, distance;
	const float *r;
	int steer, gas;
	int action, turbo = 0;
	unsigned int i;

	variation = (float) rand() / RAND_MAX;

	xlPointCopy(myMachineGetPosition(priv->play.players[player].machine), position);
	xlPointCopy(myCheckPointsCheckPoint(priv->play.race.checkpoints, (priv->play.players[player].checkpoint + 1) % myCheckPointsLength(priv->play.race.checkpoints)), target);
	xlVectorSub(target, position, direction);
	xlVectorNorm(direction, &priv->play.players[player].distance);
	if(priv->play.players[player].distance < 20.0f)
	{
		priv->play.players[player].checkpoint++;
		xlPointCopy(myCheckPointsCheckPoint(priv->play.race.checkpoints, (priv->play.players[player].checkpoint + 1) % myCheckPointsLength(priv->play.race.checkpoints)), target);
		xlVectorSub(target, position, direction);
		xlVectorNorm(direction, &priv->play.players[player].distance);
	}
	xlVectorNormalInPlace(direction);
	r = myMachineGetRotation(priv->play.players[player].machine);
	orientation[0] = -r[2]; orientation[1] = -r[6]; orientation[2] = -r[10];
	xlVectorCrossProduct(xlVectorY, orientation, plane);
	plane[3] = -xlVectorDotProductFunc(plane, position);
	distance = xlVectorDotProductFunc(plane, target) + plane[3];
	xlVectorAngle(orientation, direction, &angle);
	if(distance > 10.0f * (0.25f + variation * 0.75f)) steer = 1;
	else if(distance < -10.0f * (0.25 + variation * 0.75f)) steer = -1;
	else steer = 0;
	if(angle > 75.0f)
	{
		const float *v;
		v = myMachineGetLinearVel(priv->play.players[player].machine);
		if(priv->play.players[player].distance < 20.0f * 2.0f)
		{
			if(xlVectorNormFunc(v) > 2.0f) gas = 0;
			else gas = 1;
		}
		else { gas = -1; steer *= -1; }
	}
	else gas = 1;
	action = variation < 0.05;

	if(priv->play.players[player].item == -1)
	{
		for(i = 0; i < priv->options.nitems; i++)
		{
			if(priv->play.items[i].frames == 0)
			{
				xlPointSub(position, myBoxSphere(priv->play.items[i].box), item);
				if(xlVectorNormFunc(item) <= myBoxSphere(priv->play.items[i].box)[3])
				{
					priv->play.players[player].item = i;
					priv->play.items[i].player = player;
					switch(myGadgetType(priv->play.items[i].gadget))
					{
						case MY_GADGET_CUBE: priv->play.items[i].frames = 800; break;
						case MY_GADGET_TURBO: priv->play.items[i].frames = 100; break;
						case MY_GADGET_FAKE: priv->play.items[i].frames = 200; break;
					}
					g_signal_emit_by_name(self, "my-gadget-captured", player, myGadgetType(priv->play.items[priv->play.players[player].item].gadget));
				}
			}
		}
	}
	else
	{
		if(action)
		{
			switch(myGadgetType(priv->play.items[priv->play.players[player].item].gadget))
			{
				case MY_GADGET_CUBE:
				{
					xlVectorScalarProductInPlace(orientation, 3.0f);
					xlPointSub(position, orientation, item);
					myGadgetPlace(priv->play.items[priv->play.players[player].item].gadget, item);
					myGadgetEnable(priv->play.items[priv->play.players[player].item].gadget);
					unsigned int gadget = myGadgetType(priv->play.items[priv->play.players[player].item].gadget);
					priv->play.items[priv->play.players[player].item].player = -1;
					priv->play.players[player].item = -1;
					g_signal_emit_by_name(self, "my-gadget-released", player, gadget);
				}
					break;
				case MY_GADGET_TURBO:
					turbo = 1;
					break;
				case MY_GADGET_FAKE:
					break;
			}
		}
	}

	myMachineDrive(priv->play.players[player].machine, gas == 1, gas == -1, steer == 1, steer == -1, turbo);
}

static
void
on_near(void *data, dGeomID o1, dGeomID o2)
{
	MyMachines *self = MY_MACHINES(data);
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);

	const int ncontacts = 1024;
	unsigned int i;
	int n;
	dBodyID b1, b2;
	dContact contacts[ncontacts];
	dJointID c;
	b1 = dGeomGetBody(o1);
	b2 = dGeomGetBody(o2);
	if(b1 && b2 && dAreConnectedExcluding(b1, b2, dJointTypeContact))
		return;
	n = dCollide(o1, o2, ncontacts, &contacts[0].geom, sizeof(dContact));
	if(n > 0)
	{
		for(i = 0; i < (unsigned int) n; i++)
		{
			contacts[i].surface.mode = dContactSlip1 | dContactSlip2 | dContactSoftERP | dContactSoftCFM | dContactApprox1 | dContactMu2;
			contacts[i].surface.mu = 0.75f;
			contacts[i].surface.mu2 = 0.75f;
			contacts[i].surface.slip1 = 0.01f;
			contacts[i].surface.slip2 = 0.01f;
			contacts[i].surface.soft_erp = 0.5f;
			contacts[i].surface.soft_cfm = 0.3f;
			c = dJointCreateContact(priv->world, priv->joints, &contacts[i]);
			dJointAttach(c, dGeomGetBody(contacts[i].geom.g1), dGeomGetBody(contacts[i].geom.g2));
		}
	}
}

static
void
my_machines_frame_play(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	GdkGLContext *context = gtk_widget_get_gl_context(GTK_WIDGET(self));
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(GTK_WIDGET(self));

	if(!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();

	unsigned int i, first, actives;

	for(actives = 0, i = 0; i < priv->options.nmachines; i++)
		if(myMachineIsEnabled(priv->play.players[i].machine))
		{
			actives++;
			if(i < priv->options.nplayers) my_machines_human(self, i);
			else my_machines_computer(self, i);
		}

	for(i = 0; i < 16; i++)
	{
		dSpaceCollide(priv->space, self, on_near);
		dWorldStep(priv->world, 0.01f);
		dJointGroupEmpty(priv->joints);
	}

	for(first = 0, i = 0; i < priv->options.nmachines; i++)
	{
		if(myMachineIsEnabled(priv->play.players[i].machine) && (priv->play.players[i].checkpoint > priv->play.players[first].checkpoint || (priv->play.players[i].checkpoint == priv->play.players[first].checkpoint && priv->play.players[i].distance < priv->play.players[first].distance)))
			first = i;
	}

	if(actives > 1)
	{
		const float *p;
		float a[3], b[3];
		p = myMachineGetPosition(priv->play.players[first].machine);
		xlVectorSub(p, priv->play.camera, a);
		xlVectorScalarProduct(a, 0.15f, b);
		xlVectorAddInPlace(priv->play.camera, b);
	}
	else
	{
		float position[3], rotation[16], target[3], forward[3];
		priv->play.players[first].points++;
		g_signal_emit_by_name(self, "my-win-point", first);
		xlPointCopy(myCheckPointsCheckPoint(priv->play.race.checkpoints, priv->play.players[first].checkpoint % myCheckPointsLength(priv->play.race.checkpoints)), priv->play.camera);
		xlPointCopy(myCheckPointsCheckPoint(priv->play.race.checkpoints, (priv->play.players[first].checkpoint + 1) % myCheckPointsLength(priv->play.race.checkpoints)), target);
		xlVectorSub(target, priv->play.camera, forward);
		xlVectorNormalInPlace(forward);
		xlMatrixRotation(forward, xlVectorY, rotation);
		for(i = 0; i < priv->options.nmachines; i++)
		{
			if(!myMachineIsEnabled(priv->play.players[i].machine))
				myMachineEnable(priv->play.players[i].machine);
			priv->play.players[i].checkpoint = priv->play.players[first].checkpoint;
			xlPointLoad(position, ((i % 4) < 2 ? -5.0f : 5.0f) + ((i & 1) ? 2.5f : -2.5f), 5.0f, (float) (i / 4) * 7.5f);
			xlVectorMatrixProductInPlace(position, rotation);
			xlPointAddInPlace(position, priv->play.camera);
			myMachinePlace(priv->play.players[i].machine, position, rotation);
		}
	}

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glLoadIdentity();

	gluLookAt(priv->play.camera[0], 70.0f, priv->play.camera[2], priv->play.camera[0], priv->play.camera[1], priv->play.camera[2], 0.0f, 0.0f, -1.0f);
	xlFrustumCalc(priv->play.frustum);

	myTrackFrame(priv->play.race.track, priv->play.frustum);

	for(i = 0; i < priv->options.nmachines; i++)
	{
		if(myMachineIsEnabled(priv->play.players[i].machine))
		{
			if(xlFrustumPoint(priv->play.frustum, myMachineGetPosition(priv->play.players[i].machine)))
				myMachineFrame(priv->play.players[i].machine);
			else
			{
				if(priv->play.players[i].item != -1)
				{
					unsigned int checkpoint = ((float) rand() / RAND_MAX) * myCheckPointsLength(priv->play.race.checkpoints);
					myBoxPlace(priv->play.items[priv->play.players[i].item].box, myCheckPointsCheckPoint(priv->play.race.checkpoints, checkpoint));
					unsigned int gadget = myGadgetType(priv->play.items[priv->play.players[i].item].gadget);
					priv->play.items[priv->play.players[i].item].player = -1;
					priv->play.players[i].item = -1;
					g_signal_emit_by_name(self, "my-gadget-released", i, gadget);
				}
				myMachineDisable(priv->play.players[i].machine);
				g_signal_emit_by_name(self, "my-out-point", i);
				if(actives == priv->options.nmachines && priv->play.players[i].points)
				{
					priv->play.players[i].points--;
					g_signal_emit_by_name(self, "my-loose-point", i);
				}
			}
		}
	}

	for(i = 0; i < priv->options.nitems; i++)
	{
		if(priv->play.items[i].frames > 0)
		{
			if(priv->play.items[i].player == -1)
				myGadgetFrame(priv->play.items[i].gadget);

			priv->play.items[i].frames--;

			if(priv->play.items[i].frames == 0)
			{
				if(priv->play.items[i].player != -1)
				{
					priv->play.players[priv->play.items[i].player].item = -1;
					g_signal_emit_by_name(self, "my-gadget-released", priv->play.items[i].player, myGadgetType(priv->play.items[i].gadget));
					priv->play.items[i].player = -1;
				}
				else
					myGadgetDisable(priv->play.items[i].gadget);

				unsigned int checkpoint = ((float) rand() / RAND_MAX) * myCheckPointsLength(priv->play.race.checkpoints);
				myBoxPlace(priv->play.items[i].box, myCheckPointsCheckPoint(priv->play.race.checkpoints, checkpoint));
			}
		}
		else
			myBoxFrame(priv->play.items[i].box);
	}

	alListener3f(AL_POSITION, priv->play.camera[0], 0, priv->play.camera[2]);

	gdk_gl_drawable_swap_buffers(drawable);
		
	gdk_gl_drawable_gl_end(drawable);

	if(priv->play.players[first].points == priv->options.npoints)
	{
		my_machines_set_state(self, MY_MACHINES_STATE_MAIN);
		g_signal_emit_by_name(self, "my-win", first);
	}
}

static
void
my_machines_frame_pause(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	GdkGLContext *context = gtk_widget_get_gl_context(GTK_WIDGET(self));
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(GTK_WIDGET(self));

	if(!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glBindTexture(GL_TEXTURE_2D, priv->main.background);
	glPushMatrix();
		glTranslatef((float) rand() / RAND_MAX * 0.01f - 0.005f, (float) rand() / RAND_MAX * 0.01f - 0.005f, 0.0f);
		glBegin(GL_QUADS);
			glTexCoord2i(0, 0); glVertex2f(-0.01f, -0.01f);
			glTexCoord2i(1, 0); glVertex2f(1.01f, -0.01f);
			glTexCoord2i(1, 1); glVertex2f(1.01f, 1.01f);
			glTexCoord2i(0, 1); glVertex2f(-0.01f, 1.01f);
		glEnd();
	glPopMatrix();

	gdk_gl_drawable_swap_buffers(drawable);
		
	gdk_gl_drawable_gl_end(drawable);
}

static
void
my_machines_frame(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);

	switch(priv->state)
	{
		case MY_MACHINES_STATE_MAIN: my_machines_frame_main(self); break;
		case MY_MACHINES_STATE_PLAY: my_machines_frame_play(self); break;
		case MY_MACHINES_STATE_PAUSE: my_machines_frame_pause(self); break;
	}
}

static
void
my_machines_enter_main(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	GdkGLContext *context = gtk_widget_get_gl_context(GTK_WIDGET(self));
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(GTK_WIDGET(self));

	if(!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();

	alSourcePlay(priv->main.music);

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluOrtho2D(0.0f, 1.0f, 0.0f, 1.0f);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	gdk_gl_drawable_gl_end(drawable);

	priv->state = MY_MACHINES_STATE_MAIN;
}

static
void
my_machines_leave_main(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	alSourceStop(priv->main.music);
}

static
void
my_machines_enter_play(MyMachines *self)
{
	my_machines_load(self);
}

static
void
my_machines_leave_play(MyMachines *self)
{
	my_machines_unload(self);
}

static
void
my_machines_enter_pause(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	GdkGLContext *context = gtk_widget_get_gl_context(GTK_WIDGET(self));
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(GTK_WIDGET(self));
	unsigned int i;

	if(!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();

	for(i = 0; i < priv->options.nmachines; i++)
		if(myMachineIsEnabled(priv->play.players[i].machine))
			myMachinePause(priv->play.players[i].machine);

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluOrtho2D(0.0f, 1.0f, 0.0f, 1.0f);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	gdk_gl_drawable_gl_end(drawable);

	priv->state = MY_MACHINES_STATE_PAUSE;
}

static
void
my_machines_leave_pause(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	GdkGLContext *context = gtk_widget_get_gl_context(GTK_WIDGET(self));
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(GTK_WIDGET(self));
	unsigned int i;

	if(!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();
	for(i = 0; i < priv->options.nmachines; i++)
		if(myMachineIsEnabled(priv->play.players[i].machine))
			myMachinePlay(priv->play.players[i].machine);

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(45.0f, priv->aspect, 1.0f, 250.0f);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	gdk_gl_drawable_gl_end(drawable);

	priv->state = MY_MACHINES_STATE_PLAY;
}

static
gboolean
on_load(gpointer data)
{
	MyMachines *self = MY_MACHINES(data);
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	GdkGLContext *context = gtk_widget_get_gl_context(GTK_WIDGET(self));
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(GTK_WIDGET(self));

	if (!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();

	switch(priv->load.state)
	{
		case MY_MACHINES_LOAD_BEGIN:
			priv->load.state = MY_MACHINES_LOAD_RACE;
			break;
		case MY_MACHINES_LOAD_RACE:
		{
			unsigned int strings[4];

			myGenTracks(1, &priv->play.race.track);
			myGenCheckPointss(1, &priv->play.race.checkpoints);

			xlGenStrings(4, strings);
			xlStringLoad(strings[0], priv->options.tracks[priv->options.track]);
			xlStringLoad(strings[1], "Track");
			xlStringConcat(strings[2], 2, strings);
			myTrackLoad(priv->play.race.track, strings[2], priv->space);
			xlStringsUnload(2, strings);
			xlStringLoad(strings[0], PACKAGE_DATADIR);
			xlStringLoad(strings[1], "checkpoints");
			xlStringConcatPath(strings[3], 3, strings);
			myCheckPointsLoad(priv->play.race.checkpoints, strings[3]);
			xlStringsUnload(4, strings);
			xlDeleteStrings(4, strings);

			priv->load.state = MY_MACHINES_LOAD_ITEMS;
		}
			break;
		case MY_MACHINES_LOAD_ITEMS:
		{
			float sphere[4];
			unsigned int checkpoint;

			checkpoint = ((float) rand() / RAND_MAX) * myCheckPointsLength(priv->play.race.checkpoints);
			xlPointCopy(myCheckPointsCheckPoint(priv->play.race.checkpoints, checkpoint), sphere);
			sphere[3] = 2.0f;

			myGenBoxes(1, &priv->play.items[priv->load.items].box);
			myBoxLoad(priv->play.items[priv->load.items].box, sphere);

			myGenGadgets(1, &priv->play.items[priv->load.items].gadget);

			unsigned int type = (unsigned int) (((float) rand() / RAND_MAX) * (sizeof(gadgets) / sizeof(gadgets[0])));
			switch(type)
			{
				case MY_GADGET_CUBE: myGadgetLoadCube(priv->play.items[priv->load.items].gadget, priv->world, priv->space); break;
				case MY_GADGET_TURBO: myGadgetLoadTurbo(priv->play.items[priv->load.items].gadget); break;
				case MY_GADGET_FAKE: myGadgetLoadFake(priv->play.items[priv->load.items].gadget); break;
			}

			myGadgetDisable(priv->play.items[priv->load.items].gadget);

			priv->play.items[priv->load.items].player = -1;
			priv->play.items[priv->load.items].frames = 0;

			priv->load.items++;

			if(priv->load.items == priv->options.nitems) priv->load.state = MY_MACHINES_LOAD_MACHINES;
		}
			break;
		case MY_MACHINES_LOAD_MACHINES:
		{
			float position[3];
			unsigned int strings[4];

			memset(priv->play.players[priv->load.machines].keys, 0, 5 * sizeof(int));

			priv->play.players[priv->load.machines].checkpoint = 0;
			priv->play.players[priv->load.machines].distance = 0.0f;
			priv->play.players[priv->load.machines].points = 0;
			priv->play.players[priv->load.machines].item = -1;

			myGenMachines(1, &priv->play.players[priv->load.machines].machine);

			xlPointLoad(position, ((priv->load.machines % 4) < 2 ? -5.0f : 5.0f) + ((priv->load.machines & 1) ? 2.5f : -2.5f), 5.0f, (float) (priv->load.machines / 4) * 7.5f);

			xlGenStrings(4, strings);
			xlStringLoad(strings[0], priv->options.machines[priv->options.machine]);
			xlStringLoad(strings[1], "Car");
			xlStringLoad(strings[2], colors[priv->load.machines]);
			xlStringConcat(strings[3], 3, strings);
			myMachineLoad(priv->play.players[priv->load.machines].machine, strings[3], priv->world, priv->space, position);
			xlStringsUnload(4, strings),
			xlDeleteStrings(4, strings);

			priv->load.machines++;

			if(priv->load.machines == priv->options.nmachines) priv->load.state = MY_MACHINES_LOAD_SCENE;
		}
			break;
		case MY_MACHINES_LOAD_SCENE:
		{
			float listener[6] = {0.0f, -1.0f, 0.0f, 0.0f, 0.0f, -1.0f};

			xlPointLoadZero(priv->play.camera);

			xlGenFrustums(1, &priv->play.frustum);

			alListener3f(AL_POSITION, priv->play.camera[0], 0.0f, priv->play.camera[2]);
			alListenerfv(AL_ORIENTATION, listener);

			glMatrixMode(GL_PROJECTION);
			glLoadIdentity();
			gluPerspective(45.0f, priv->aspect, 1.0f, 250.0f);
			glMatrixMode(GL_MODELVIEW);
			glLoadIdentity();

			priv->load.state = MY_MACHINES_LOAD_END;
			priv->state = MY_MACHINES_STATE_PLAY;
		}
			break;
		case MY_MACHINES_LOAD_END:
			g_assert_not_reached();
			break;
	}
	gdk_gl_drawable_gl_end(drawable);
	g_signal_emit_by_name(self, "my-load", priv->load.state);

	return priv->load.state != MY_MACHINES_LOAD_END;
}

static
gboolean
on_frame(gpointer data)
{
	MyMachines *self = (MyMachines *) data;

	my_machines_frame(self);

	return TRUE;
}

static
gboolean
on_realize(GtkWidget *widget, gpointer data)
{
	MyMachines *self = MY_MACHINES(widget);
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);

	unsigned int sound;
	unsigned int string;
	unsigned int image;
	ALuint buffer;

	GdkGLContext *context = gtk_widget_get_gl_context(widget);
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(widget);

	if (!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();

	srand(xlSystemTicks());

	glEnable(GL_COLOR_MATERIAL);
	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LEQUAL);
	glEnable(GL_TEXTURE_2D);
	glClearColor(0, 0, 0.5, 0);

	xlGenStrings(1, &string);

	glGenTextures(1, &priv->main.background);
	glBindTexture(GL_TEXTURE_2D, priv->main.background);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	xlGenImages(1, &image);
	xlStringLoad(string, PACKAGE_DATADIR XL_STRING_PATH_SEPARATOR "textures" XL_STRING_PATH_SEPARATOR "Analog-Wind-1.jpg");
	xlImageLoadJPEG(image, string);
	xlStringUnload(string);
	xlImageTexImage2D(image);
	xlImageUnload(image);
	xlDeleteImages(1, &image);

	priv->world = dWorldCreate();
	priv->space = dHashSpaceCreate(0);
	priv->joints = dJointGroupCreate(0);
	dWorldSetGravity(priv->world, 0.0f, -9.8f, 0.0f);

	alGenSources(1, &priv->main.music);
	alGenBuffers(1, &buffer);
	xlGenSounds(1, &sound);
	xlStringLoad(string, PACKAGE_DATADIR XL_STRING_PATH_SEPARATOR "sounds" XL_STRING_PATH_SEPARATOR "engine.ogg");
	xlSoundLoadVORBIS(sound, string);
	xlStringUnload(string);
	xlSoundBufferData(sound, buffer);
	xlSoundUnload(sound);
	xlDeleteSounds(1, &sound);
	alSourcei(priv->main.music, AL_BUFFER, buffer);
	alSourcei(priv->main.music, AL_LOOPING, AL_TRUE);
	alSourcei(priv->main.music, AL_SOURCE_RELATIVE, AL_TRUE);

	xlDeleteStrings(1, &string);

	gdk_gl_drawable_gl_end(drawable);

	my_machines_enter_main(self);
	g_signal_emit_by_name(self, "my-state-changed", MY_MACHINES_STATE_MAIN);

	priv->timeout = g_timeout_add(40, on_frame, MY_MACHINES(widget));

	return TRUE;
}

static
void
on_unrealize(GtkWidget *widget)
{
	MyMachines *self = MY_MACHINES(widget);
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	GdkGLContext *context = gtk_widget_get_gl_context(widget);
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(widget);
	ALuint buffer;

	g_source_remove(priv->timeout);

	if(my_machines_is_loaded(self)) my_machines_unload(self);

	if (!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();

	dJointGroupDestroy(priv->joints);
	dSpaceDestroy(priv->space);
	dWorldDestroy(priv->world);

	glDeleteTextures(1, &priv->main.background);

	alGetSourcei(priv->main.music, AL_BUFFER, (ALint*)&buffer);
	alDeleteSources(1, &priv->main.music);
	alDeleteBuffers(1, &buffer);

	gdk_gl_drawable_gl_end(drawable);
}

static
gboolean
my_machines_configure_event_main(MyMachines *self, GdkEventConfigure *event)
{
	GdkGLContext *context = gtk_widget_get_gl_context(GTK_WIDGET(self));
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(GTK_WIDGET(self));

	if (!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluOrtho2D(0.0f, 1.0f, 0.0f, 1.0f);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	
	gdk_gl_drawable_gl_end(drawable);

	return TRUE;
}

static
gboolean
my_machines_configure_event_play(MyMachines *self, GdkEventConfigure *event)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	GdkGLContext *context = gtk_widget_get_gl_context(GTK_WIDGET(self));
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(GTK_WIDGET(self));

	if (!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(45.0f, priv->aspect, 1.0f, 250.0f);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	
	gdk_gl_drawable_gl_end(drawable);

	return TRUE;
}

static
gboolean
my_machines_configure_event_pause(MyMachines *self, GdkEventConfigure *event)
{
	GdkGLContext *context = gtk_widget_get_gl_context(GTK_WIDGET(self));
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(GTK_WIDGET(self));

	if (!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluOrtho2D(0.0f, 1.0f, 0.0f, 1.0f);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	
	gdk_gl_drawable_gl_end(drawable);

	return TRUE;
}

static
gboolean
my_machines_configure_event(MyMachines *self, GdkEventConfigure *event)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	gboolean result;

	switch(priv->state)
	{
		case MY_MACHINES_STATE_MAIN: result = my_machines_configure_event_main(self, event);; break;
		case MY_MACHINES_STATE_PLAY: result = my_machines_configure_event_play(self, event); break;
		case MY_MACHINES_STATE_PAUSE: result = my_machines_configure_event_pause(self, event); break;
		default: result = FALSE; break;
	}
	return result;
}

static
gboolean
on_configure_event(GtkWidget *widget, GdkEventConfigure *event, gpointer data)
{
	MyMachines *self = MY_MACHINES(widget);
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	GdkGLContext *context = gtk_widget_get_gl_context(widget);
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(widget);

	priv->aspect = (float) widget->allocation.width / widget->allocation.height;

	if (!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();

	glViewport(0, 0, widget->allocation.width, widget->allocation.height);

	gdk_gl_drawable_gl_end(drawable);

	return my_machines_configure_event(self, event);
}

/*
static
gboolean
on_expose_event(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
	if(event->count == 0) my_machines_frame(MY_MACHINES(widget));
	return TRUE;
}
*/

static
gboolean
my_machines_key_press_event_play(MyMachines *self, GdkEventKey *event)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	switch(event->keyval)
	{
		case GDK_Up: priv->play.players[0].keys[0] = TRUE; break;
		case GDK_Down: priv->play.players[0].keys[1] = TRUE; break;
		case GDK_Left: priv->play.players[0].keys[2] = TRUE; break;
		case GDK_Right: priv->play.players[0].keys[3] = TRUE; break;
		case GDK_Return: priv->play.players[0].keys[4] = TRUE; break;
		case GDK_w: priv->play.players[1].keys[0] = TRUE; break;
		case GDK_s: priv->play.players[1].keys[1] = TRUE; break;
		case GDK_a: priv->play.players[1].keys[2] = TRUE; break;
		case GDK_d: priv->play.players[1].keys[3] = TRUE; break;
		case GDK_q: priv->play.players[1].keys[4] = TRUE; break;
		case GDK_i: priv->play.players[2].keys[0] = TRUE; break;
		case GDK_k: priv->play.players[2].keys[1] = TRUE; break;
		case GDK_j: priv->play.players[2].keys[2] = TRUE; break;
		case GDK_l: priv->play.players[2].keys[3] = TRUE; break;
		case GDK_u: priv->play.players[2].keys[4] = TRUE; break;
		case GDK_t: priv->play.players[3].keys[0] = TRUE; break;
		case GDK_g: priv->play.players[3].keys[1] = TRUE; break;
		case GDK_f: priv->play.players[3].keys[2] = TRUE; break;
		case GDK_h: priv->play.players[3].keys[3] = TRUE; break;
		case GDK_r: priv->play.players[3].keys[4] = TRUE; break;
	}
	return TRUE;
}
static
gboolean
my_machines_key_release_event_play(MyMachines *self, GdkEventKey *event)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);

	switch(event->keyval)
	{
		case GDK_Escape: my_machines_set_state(self, MY_MACHINES_STATE_MAIN); break;
		case GDK_Pause: my_machines_set_state(self, MY_MACHINES_STATE_PAUSE); break;
		case GDK_Up: priv->play.players[0].keys[0] = FALSE; break;
		case GDK_Down: priv->play.players[0].keys[1] = FALSE; break;
		case GDK_Left: priv->play.players[0].keys[2] = FALSE; break;
		case GDK_Right: priv->play.players[0].keys[3] = FALSE; break;
		case GDK_Return: priv->play.players[0].keys[4] = FALSE; break;
		case GDK_w: priv->play.players[1].keys[0] = FALSE; break;
		case GDK_s: priv->play.players[1].keys[1] = FALSE; break;
		case GDK_a: priv->play.players[1].keys[2] = FALSE; break;
		case GDK_d: priv->play.players[1].keys[3] = FALSE; break;
		case GDK_q: priv->play.players[1].keys[4] = FALSE; break;
		case GDK_i: priv->play.players[2].keys[0] = FALSE; break;
		case GDK_k: priv->play.players[2].keys[1] = FALSE; break;
		case GDK_j: priv->play.players[2].keys[2] = FALSE; break;
		case GDK_l: priv->play.players[2].keys[3] = FALSE; break;
		case GDK_u: priv->play.players[2].keys[4] = FALSE; break;
		case GDK_t: priv->play.players[3].keys[0] = FALSE; break;
		case GDK_g: priv->play.players[3].keys[1] = FALSE; break;
		case GDK_f: priv->play.players[3].keys[2] = FALSE; break;
		case GDK_h: priv->play.players[3].keys[3] = FALSE; break;
		case GDK_r: priv->play.players[3].keys[4] = FALSE; break;
	}
	return TRUE;
}

static
gboolean
my_machines_key_press_event_pause(MyMachines *self, GdkEventKey *event)
{
	return TRUE;
}

static
gboolean
my_machines_key_release_event_pause(MyMachines *self, GdkEventKey *event)
{
	switch(event->keyval)
	{
		case GDK_Escape: my_machines_set_state(self, MY_MACHINES_STATE_PLAY); break;
	}
	return TRUE;
}

static
gboolean
my_machines_key_press_event(MyMachines *self, GdkEventKey *event)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	gboolean result;

	switch(priv->state)
	{
		case MY_MACHINES_STATE_MAIN: result = FALSE;; break;
		case MY_MACHINES_STATE_PLAY: result = my_machines_key_press_event_play(self, event); break;
		case MY_MACHINES_STATE_PAUSE: result = my_machines_key_press_event_pause(self, event); break;
		default: result = FALSE; break;
	}
	return result;
}

static
gboolean
my_machines_key_release_event(MyMachines *self, GdkEventKey *event)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	gboolean result;

	switch(priv->state)
	{
		case MY_MACHINES_STATE_MAIN: result = FALSE;; break;
		case MY_MACHINES_STATE_PLAY: result = my_machines_key_release_event_play(self, event); break;
		case MY_MACHINES_STATE_PAUSE: result = my_machines_key_release_event_pause(self, event); break;
		default: result = FALSE; break;
	}
	return result;
}

static
gboolean
on_key_press_event(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	MyMachines *self = MY_MACHINES(widget);
	return my_machines_key_press_event(self, event);
}

static
gboolean
on_key_release_event(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	MyMachines *self = MY_MACHINES(widget);
	return my_machines_key_release_event(self, event);
}

static
void
my_machines_init(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);

	priv->config = gdk_gl_config_new_by_mode((GdkGLConfigMode) (GDK_GL_MODE_RGBA | GDK_GL_MODE_DEPTH | GDK_GL_MODE_DOUBLE));
	if(!priv->config) g_assert_not_reached();
	if(!gtk_widget_set_gl_capability(GTK_WIDGET(self), priv->config, NULL, TRUE, GDK_GL_RGBA_TYPE)) g_assert_not_reached();

	priv->timeout = 0;
	priv->options.tracks[0] = strdup("Crayons");
	priv->options.tracks[1] = strdup("Garden");
	priv->options.tracks[2] = strdup("Land");
	priv->options.tracks[3] = strdup("Bathtub");
	priv->options.tracks[4] = strdup("Square");
	priv->options.tracks[5] = strdup("Circle");
	priv->options.machines[0] = strdup("Generic");
	priv->options.machines[1] = strdup("Formula1");
	priv->options.machines[2] = strdup("Block");
	priv->options.nmachines = 4;
	priv->options.nplayers = 1;
	priv->options.nitems = 8;
	priv->options.track = 0;
	priv->options.machine = 0;
	priv->options.npoints = 3;

	GTK_WIDGET_SET_FLAGS(GTK_WIDGET(self), GTK_CAN_FOCUS);
	gtk_widget_set_events(GTK_WIDGET(self), GDK_EXPOSURE_MASK | GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK);
	g_signal_connect(self, "realize", G_CALLBACK(on_realize), self);
	g_signal_connect(self, "unrealize", G_CALLBACK(on_unrealize), self);
	g_signal_connect(self, "configure-event", G_CALLBACK(on_configure_event), self);
	g_signal_connect(self, "key-press-event", G_CALLBACK(on_key_press_event), self);
	g_signal_connect(self, "key-release-event", G_CALLBACK(on_key_release_event), self);
//	g_signal_connect(self, "expose-event", G_CALLBACK(on_expose_event), self);
}

static
void
my_machines_finalize(GObject *object)
{
	MyMachines *self = MY_MACHINES(object);
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);

	free(priv->options.machines[2]);
	free(priv->options.machines[1]);
	free(priv->options.machines[0]);
	free(priv->options.tracks[5]);
	free(priv->options.tracks[4]);
	free(priv->options.tracks[3]);
	free(priv->options.tracks[2]);
	free(priv->options.tracks[1]);
	free(priv->options.tracks[0]);

	G_OBJECT_CLASS(my_machines_parent_class)->finalize(object);
}

static
void
my_machines_dispose(GObject *object)
{
	G_OBJECT_CLASS(my_machines_parent_class)->dispose(object);
}

static
void
my_machines_class_init(MyMachinesClass *klass)
{
	G_OBJECT_CLASS(klass)->dispose = my_machines_dispose;
	G_OBJECT_CLASS(klass)->finalize = my_machines_finalize;

	g_type_class_add_private(klass, sizeof(MyMachinesPrivate));

	g_signal_new("my-state-changed", G_TYPE_FROM_CLASS(G_OBJECT_CLASS(klass)), G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS, 0, NULL, NULL, g_cclosure_marshal_VOID__UINT, G_TYPE_NONE, 1, G_TYPE_UINT);
	g_signal_new("my-load", G_TYPE_FROM_CLASS(G_OBJECT_CLASS(klass)), G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS, 0, NULL, NULL, g_cclosure_marshal_VOID__UINT, G_TYPE_NONE, 1, G_TYPE_UINT);
	g_signal_new("my-gadget-captured", G_TYPE_FROM_CLASS(G_OBJECT_CLASS(klass)), G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS, 0, NULL, NULL, g_cclosure_user_marshal_VOID__UINT_UINT, G_TYPE_NONE, 2, G_TYPE_UINT, G_TYPE_UINT);
	g_signal_new("my-gadget-released", G_TYPE_FROM_CLASS(G_OBJECT_CLASS(klass)), G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS, 0, NULL, NULL, g_cclosure_user_marshal_VOID__UINT_UINT, G_TYPE_NONE, 2, G_TYPE_UINT, G_TYPE_UINT);
	g_signal_new("my-win-point", G_TYPE_FROM_CLASS(G_OBJECT_CLASS(klass)), G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS, 0, NULL, NULL, g_cclosure_marshal_VOID__UINT, G_TYPE_NONE, 1, G_TYPE_UINT);
	g_signal_new("my-loose-point", G_TYPE_FROM_CLASS(G_OBJECT_CLASS(klass)), G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS, 0, NULL, NULL, g_cclosure_marshal_VOID__UINT, G_TYPE_NONE, 1, G_TYPE_UINT);
	g_signal_new("my-out-point", G_TYPE_FROM_CLASS(G_OBJECT_CLASS(klass)), G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS, 0, NULL, NULL, g_cclosure_marshal_VOID__UINT, G_TYPE_NONE, 1, G_TYPE_UINT);
	g_signal_new("my-win", G_TYPE_FROM_CLASS(G_OBJECT_CLASS(klass)), G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS, 0, NULL, NULL, g_cclosure_marshal_VOID__UINT, G_TYPE_NONE, 1, G_TYPE_UINT);
}

MyMachines *
my_machines_new(void)
{
	return MY_MACHINES(g_object_new(MY_TYPE_MACHINES, NULL));
}

void
my_machines_load(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);

	priv->load.state = MY_MACHINES_LOAD_BEGIN;
	priv->load.items = priv->load.machines = 0;
	g_signal_emit_by_name(self, "my-load", priv->load.state);

	g_idle_add(on_load, self);
}

void
my_machines_unload(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	GdkGLContext *context = gtk_widget_get_gl_context(GTK_WIDGET(self));
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(GTK_WIDGET(self));
	unsigned int i;

	if (!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();
	switch(priv->load.state)
	{
		case MY_MACHINES_LOAD_END:
		case MY_MACHINES_LOAD_SCENE:
			xlDeleteFrustums(1, &priv->play.frustum);
		case MY_MACHINES_LOAD_MACHINES:
			for(i = 0; i < priv->load.machines; i++)
			{
				myMachineUnload(priv->play.players[i].machine);
				myDeleteMachines(1, &priv->play.players[i].machine);
			}
		case MY_MACHINES_LOAD_ITEMS:
			for(i = 0; i < priv->load.items; i++)
			{
				myBoxUnload(priv->play.items[i].box);
				myDeleteBoxes(1, &priv->play.items[i].box);
				myGadgetUnload(priv->play.items[i].gadget);
				myDeleteGadgets(1, &priv->play.items[i].gadget);
			}
		case MY_MACHINES_LOAD_RACE:
			myCheckPointsUnload(priv->play.race.checkpoints);
			myDeleteCheckPointss(1, &priv->play.race.checkpoints);
			myTrackUnload(priv->play.race.track);
			myDeleteTracks(1, &priv->play.race.track);
		case MY_MACHINES_LOAD_BEGIN:
			break;
	}
	gdk_gl_drawable_gl_end(drawable);
	priv->load.state = MY_MACHINES_LOAD_BEGIN;
}

gboolean
my_machines_is_loaded(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);

	return priv->load.state != MY_MACHINES_LOAD_BEGIN;
}

void
my_machines_set_state(MyMachines *self, MyMachinesState state)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);

	switch(state)
	{
		case MY_MACHINES_STATE_MAIN:
			switch(priv->state)
			{
				case MY_MACHINES_STATE_MAIN: break;
				case MY_MACHINES_STATE_PLAY: my_machines_leave_play(self); my_machines_enter_main(self); g_signal_emit_by_name(self, "my-state-changed", state); break;
				case MY_MACHINES_STATE_PAUSE: my_machines_leave_pause(self); my_machines_leave_play(self), my_machines_enter_main(self); g_signal_emit_by_name(self, "my-state-changed", state); break;
			}
			break;
		case MY_MACHINES_STATE_PLAY:
			switch(priv->state)
			{
				case MY_MACHINES_STATE_MAIN: my_machines_leave_main(self); my_machines_enter_play(self); g_signal_emit_by_name(self, "my-state-changed", state); break;
				case MY_MACHINES_STATE_PLAY: break;
				case MY_MACHINES_STATE_PAUSE: my_machines_leave_pause(self); g_signal_emit_by_name(self, "my-state-changed", state); break;
			}
			break;
		case MY_MACHINES_STATE_PAUSE:
			switch(priv->state)
			{
				case MY_MACHINES_STATE_MAIN: break;
				case MY_MACHINES_STATE_PLAY: my_machines_enter_pause(self); g_signal_emit_by_name(self, "my-state-changed", state); break;
				case MY_MACHINES_STATE_PAUSE: break;
			}
			break;
	}
}

MyMachinesState
my_machines_get_state(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	return priv->state;
}

void
my_machines_set_machines(MyMachines *self, guint machines)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	priv->options.nmachines = machines;
}

guint
my_machines_get_machines(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	return priv->options.nmachines;
}

void
my_machines_set_players(MyMachines *self, guint players)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	priv->options.nplayers = players;
}

guint
my_machines_get_players(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	return priv->options.nplayers;
}

void
my_machines_set_items(MyMachines *self, guint items)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	priv->options.nitems = items;
}

void
my_machines_set_points(MyMachines *self, guint points)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	priv->options.npoints = points;
}

void
my_machines_set_track(MyMachines *self, guint track)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	priv->options.track = track;
}

void
my_machines_set_machine(MyMachines *self, guint machine)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	priv->options.machine = machine;
}

const gchar *
my_machines_get_gadget(MyMachines *self, unsigned int gadget)
{
	return gadgets[gadget];
}

const gchar *
my_machines_get_player_color(MyMachines *self, guint player)
{
	return colors[player];
}

guint
my_machines_get_player_points(MyMachines *self, guint player)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	return priv->play.players[player].points;
}

gint
my_machines_get_player_gadget(MyMachines *self, guint player)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	return priv->play.players[player].item >= 0 ? myGadgetType(priv->play.items[priv->play.players[player].item].gadget) : -1;
}

gfloat
my_machines_get_load_fraction(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	return (float) (priv->load.state + priv->load.items + priv->load.machines) / (MY_MACHINES_LOAD_END + priv->options.nitems + priv->options.nmachines);
}

