// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "mymachine.h"
#include <XL/xl.h>
#include <GL/gl.h>
#include <AL/al.h>
#include <ode/ode.h>
#include "id.h"

typedef struct
{
	float speed;
	float steer;
	dBodyID bodies[5];
	dGeomID geoms[5];
	dJointID joints[4];
	unsigned int textures[2];
	unsigned int meshes[2];
	unsigned int lists;
	ALuint engine;
} MACHINE;

#define STORE ID_SMALL_STORE

static MACHINE *store[STORE];

void
myMachineCreateContext(void)
{
	idClearStore(MACHINE, STORE, store);
}

static
void
Init(MACHINE *machine)
{
	xlGenMeshes(2, machine->meshes);
	xlGenGroups(2, machine->textures);
}

static
void
Term(MACHINE *machine)
{
	xlDeleteGroups(2, machine->textures);
	xlDeleteMeshes(2, machine->meshes);
}

void
myGenMachines(unsigned int n, unsigned int *machines)
{
	idGenObjects(MACHINE, STORE, store, Init, n, machines);
}

void
myDeleteMachines(unsigned int n, unsigned int *machines)
{
	idDeleteObjects(store, Term, n, machines);
}

void
myMachineLoad(unsigned int id, unsigned int name, dWorldID world, dSpaceID space, const float *position)
{
	unsigned int strings[3];
	unsigned int sound;
	unsigned int i;
	ALuint buffer;
	const float *anchor;
	dMass mass;
	dQuaternion q;
	MACHINE *machine;

	machine = store[id];

	machine->speed = 70.0f;
	machine->steer = 20.0f;

	machine->bodies[0] = dBodyCreate(world);
	dMassSetBox(&mass, 1.0f, 2.5f, 1.0f, 5.5f);
	dMassAdjust(&mass, 1.0f);
	dBodySetMass(machine->bodies[0], &mass);
	machine->geoms[0] = dCreateBox(space, 2.5f, 1.0f, 5.5f);
	dGeomSetBody(machine->geoms[0], machine->bodies[0]);
	dBodySetPosition(machine->bodies[0], position[0], position[1], position[2]);

	dQFromAxisAndAngle(q, 0.0f, 1.0f, 0.0f, M_PI * 0.5f);
	dMassSetSphere(&mass, 0.05f, 0.25f);
	dMassAdjust(&mass, 0.05f);
	for(i = 1; i < 5; i++)
	{
		machine->bodies[i] = dBodyCreate(world);
		dBodySetQuaternion(machine->bodies[i], q);
		dBodySetMass(machine->bodies[i], &mass);
		machine->geoms[i] = dCreateSphere(space, 0.25f);
		dGeomSetBody(machine->geoms[i], machine->bodies[i]);
	}
	dBodySetPosition(machine->bodies[1], position[0] - 1.5f, position[1] - 0.5f, position[2] + 1.25f);
	dBodySetPosition(machine->bodies[2], position[0] + 1.5f, position[1] - 0.5f, position[2] + 1.25f);
	dBodySetPosition(machine->bodies[3], position[0] - 1.5f, position[1] - 0.5f, position[2] - 1.5f);
	dBodySetPosition(machine->bodies[4], position[0] + 1.5f, position[1] - 0.5f, position[2] - 1.5f);

	for(i = 0; i < 4; i++)
	{
		machine->joints[i] = dJointCreateHinge2(world, NULL);
		dJointAttach(machine->joints[i], machine->bodies[0], machine->bodies[i + 1]);
		anchor = dBodyGetPosition(machine->bodies[i + 1]);
		dJointSetHinge2Anchor(machine->joints[i], anchor[0], anchor[1], anchor[2]);
		dJointSetHinge2Axis1(machine->joints[i], 0.0f, 1.0f, 0.0f);
		dJointSetHinge2Axis2(machine->joints[i], 1.0f, 0.0f, 0.0f);
		dJointSetHinge2Param(machine->joints[i], dParamSuspensionERP, 0.4f);
		dJointSetHinge2Param(machine->joints[i], dParamSuspensionCFM, 0.8f);
	}
	for(i = 0; i < 2; i++)
	{
		dJointSetHinge2Param(machine->joints[i], dParamLoStop, 0);
		dJointSetHinge2Param(machine->joints[i], dParamHiStop, 0);
	}

	xlGenStrings(3, strings);

	xlStringLoad(strings[0], PACKAGE_DATADIR);

	xlStringConcatPath2(strings[2], strings[0], name);
	xlMeshLoad(machine->meshes[0], strings[2]);

	xlStringUnload(strings[2]);

	xlStringLoad(strings[1], "wheel");
	xlStringConcatPath(strings[2], 2, strings);
	xlMeshLoad(machine->meshes[1], strings[2]);

	xlStringsUnload(2, strings + 1);

	xlStringLoad(strings[1], "textures");
	xlStringConcatPath(strings[2], 2, strings);
	xlMeshGenTextures(machine->meshes[0], strings[2], machine->textures[0]);
	xlMeshGenTextures(machine->meshes[1], strings[2], machine->textures[1]);
	xlStringsUnload(2, strings + 1);

	machine->lists = glGenLists(2);

	glNewList(machine->lists, GL_COMPILE);
	xlMeshDraw(machine->meshes[0], machine->textures[0]);
	glEndList();

	glNewList(machine->lists + 1, GL_COMPILE);
	xlMeshDraw(machine->meshes[1], machine->textures[1]);
	glEndList();

	alGenSources(1, &machine->engine);
	alGenBuffers(1, &buffer);
	xlGenSounds(1, &sound);
	xlStringLoad(strings[1], "sounds" XL_STRING_PATH_SEPARATOR "engine.ogg");
	xlStringConcatPath(strings[2], 2, strings);
	xlSoundLoadVORBIS(sound, strings[2]);
	xlSoundBufferData(sound, buffer);
	xlSoundUnload(sound);
	xlDeleteSounds(1, &sound);
	alSourcei(machine->engine, AL_BUFFER, buffer);
	alSourcei(machine->engine, AL_LOOPING, AL_TRUE);
	alSource3f(machine->engine, AL_POSITION, position[0], position[1], position[2]);
	alSourcei(machine->engine, AL_ROLLOFF_FACTOR, 0.1f);
	alSourcePlay(machine->engine);

	xlStringsUnload(3, strings);

	xlDeleteStrings(3, strings);
}

void
myMachineUnload(unsigned int id)
{
	ALuint buffer;
	unsigned int i;
	MACHINE *machine;

	machine = store[id];

	alGetSourcei(machine->engine, AL_BUFFER, (int*)&buffer);
	alDeleteSources(1, &machine->engine);
	alDeleteBuffers(1, &buffer);

	glDeleteLists(machine->lists, 2);
	glDeleteTextures(xlGroupLength(machine->textures[0]), xlGroupArray(machine->textures[0]));
	glDeleteTextures(xlGroupLength(machine->textures[1]), xlGroupArray(machine->textures[1]));
	xlGroupsUnload(2, machine->textures);
	xlMeshesUnload(2, machine->meshes);

	for(i = 0; i < 4; i++) dJointDestroy(machine->joints[i]);
	for(i = 0; i < 5; i++) { dGeomDestroy(machine->geoms[i]); dBodyDestroy(machine->bodies[i]); }
}

void
myMachinesUnload(unsigned int n, unsigned int *machines)
{
	unsigned int i;

	for(i = 0; i < n; i++) myMachineUnload(machines[i]);
}

void
myMachineFrame(unsigned int id)
{
	const float *p, *r;
	float m[16], v;
	unsigned int i;
	MACHINE *machine;

	machine = store[id];

	for(v = 0, i = 0; i < 4; v += xlVectorNormFunc(dBodyGetAngularVel(machine->bodies[i + 1])), i++);
	alSourcef(machine->engine, AL_PITCH, xlClamp(v / 80.0f, 0.0f, 1.5f) + 0.5f);

	p = dBodyGetPosition(machine->bodies[0]);
	alSource3f(machine->engine, AL_POSITION, p[0], p[1], p[2]);
	p = dBodyGetLinearVel(machine->bodies[0]);
	alSource3f(machine->engine, AL_VELOCITY, p[0], p[1], p[2]);
	r = dBodyGetRotation(machine->bodies[0]);
	alSource3f(machine->engine, AL_DIRECTION, -r[2], -r[6], -r[10]);

	for(i = 0; i < 5; i++)
	{
		p = dBodyGetPosition(machine->bodies[i]);
        	r = dBodyGetRotation(machine->bodies[i]);
		xlMatrixFromODESituation(p, r, m);
		glPushMatrix();
			glMultMatrixf(m);
			glCallList(machine->lists + (i ? 1 : 0));
		glPopMatrix();
	}
}

void
myMachineEnable(unsigned int id)
{
	unsigned int i;
	MACHINE *machine;

	machine = store[id];

	alSourcePlay(machine->engine);
	for(i = 0; i < 5; i++)
	{
		dBodyEnable(machine->bodies[i]);
		dGeomEnable(machine->geoms[i]);
	}
}

void
myMachineDisable(unsigned int id)
{
	unsigned int i;
	MACHINE *machine;

	machine = store[id];

	for(i = 0; i < 5; i++)
	{
		dGeomDisable(machine->geoms[i]);
		dBodyDisable(machine->bodies[i]);
	}
	alSourceStop(machine->engine);
}

int
myMachineIsEnabled(unsigned int id)
{
	MACHINE *machine;

	machine = store[id];

	return dBodyIsEnabled(machine->bodies[0]);
}

void
myMachinePause(unsigned int id)
{
	MACHINE *machine;

	machine = store[id];

	alSourcePause(machine->engine);
}

void
myMachinePlay(unsigned int id)
{
	MACHINE *machine;

	machine = store[id];

	alSourcePlay(machine->engine);
}

const float *
myMachineGetPosition(unsigned int id)
{
	MACHINE *machine;

	machine = store[id];

	return dBodyGetPosition(machine->bodies[0]);
}

const float *
myMachineGetRotation(unsigned int id)
{
	MACHINE *machine;

	machine = store[id];

	return dBodyGetRotation(machine->bodies[0]);
}

const float *
myMachineGetLinearVel(unsigned int id)
{
	MACHINE *machine;

	machine = store[id];

	return dBodyGetLinearVel(machine->bodies[0]);
}

void
myMachineDrive(unsigned int id, int up, int down, int left, int right, int turbo)
{
	unsigned int i;
	MACHINE *machine;

	machine = store[id];

	const float speed = machine->speed * (turbo ? 2.0f : 1.0f) * (up ? 1.0f : (down ? -1.0f : 0.0f));
	const float steer = machine->steer * (left ? -1.0f : (right ? 1.0f : 0.0f));

	if(steer)
	{
		for(i = 2; i < 4; i++)
		{
			dJointSetHinge2Param(machine->joints[i], dParamVel, steer);
			dJointSetHinge2Param(machine->joints[i], dParamFMax, 4.0f);
			dJointSetHinge2Param(machine->joints[i], dParamLoStop, -0.75f);
			dJointSetHinge2Param(machine->joints[i], dParamHiStop, 0.75f);
			dJointSetHinge2Param(machine->joints[i], dParamFudgeFactor, 0.1f);
		}
	}
	else
	{
		for(i = 2; i < 4; i++)
		{
			dJointSetHinge2Param(machine->joints[i], dParamLoStop, 0.0f);
			dJointSetHinge2Param(machine->joints[i], dParamHiStop, 0.0f);
			dJointSetHinge2Param(machine->joints[i], dParamFudgeFactor, 0.1f);
		}
	}
	for(i = 0; i < 4; i++)
	{
		dJointSetHinge2Param(machine->joints[i], dParamVel2, speed);
		dJointSetHinge2Param(machine->joints[i], dParamFMax2, 2.0f);
	}
	for(i = 0; i < 2; i++)
	{
		dJointSetHinge2Param(machine->joints[i], dParamLoStop, 0.0f);
		dJointSetHinge2Param(machine->joints[i], dParamHiStop, 0.0f);
		dJointSetHinge2Param(machine->joints[i], dParamFudgeFactor, 0.01f);
	}
}

void
myMachinePlace(unsigned int id, const float *position, const float *rotation)
{
	float matrix[12], wposition[3];
	MACHINE *machine;

	machine = store[id];

	xlMatrixToODERotation(rotation, matrix);
	dBodySetPosition(machine->bodies[0], position[0], position[1], position[2]);
	dBodySetRotation(machine->bodies[0], matrix);
	dBodySetLinearVel(machine->bodies[0], 0.0f, 0.0f, 0.0f);
	dBodySetAngularVel(machine->bodies[0], 0.0f, 0.0f, 0.0f);

	xlVectorLoad(wposition, -0.5f, - 0.5f, 1.25f);
	xlVectorMatrixProductInPlace(wposition, rotation);
	xlVectorAddInPlace(wposition, position);
	dBodySetPosition(machine->bodies[1], wposition[0], wposition[1], wposition[2]);
	dBodySetRotation(machine->bodies[1], matrix);
	dBodySetLinearVel(machine->bodies[1], 0.0f, 0.0f, 0.0f);
	dBodySetAngularVel(machine->bodies[1], 0.0f, 0.0f, 0.0f);

	xlVectorLoad(wposition, 1.5f, -0.5f, 1.25f);
	xlVectorMatrixProductInPlace(wposition, rotation);
	xlVectorAddInPlace(wposition, position);
	dBodySetPosition(machine->bodies[2], wposition[0], wposition[1], wposition[2]);
	dBodySetRotation(machine->bodies[2], matrix);
	dBodySetLinearVel(machine->bodies[2], 0.0f, 0.0f, 0.0f);
	dBodySetAngularVel(machine->bodies[2], 0.0f, 0.0f, 0.0f);

	xlVectorLoad(wposition, -1.5f, -0.5f, -1.5f);
	xlVectorMatrixProductInPlace(wposition, rotation);
	xlVectorAddInPlace(wposition, position);
	dBodySetPosition(machine->bodies[3], wposition[0], wposition[1], wposition[2]);
	dBodySetRotation(machine->bodies[3], matrix);
	dBodySetLinearVel(machine->bodies[3], 0.0f, 0.0f, 0.0f);
	dBodySetAngularVel(machine->bodies[3], 0.0f, 0.0f, 0.0f);

	xlVectorLoad(wposition, 1.5f, -0.5f, -1.5f);
	xlVectorMatrixProductInPlace(wposition, rotation);
	xlVectorAddInPlace(wposition, position);
	dBodySetPosition(machine->bodies[4], wposition[0], wposition[1], wposition[2]);
	dBodySetRotation(machine->bodies[4], matrix);
	dBodySetLinearVel(machine->bodies[4], 0.0f, 0.0f, 0.0f);
	dBodySetAngularVel(machine->bodies[4], 0.0f, 0.0f, 0.0f);
}
