// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "mygadget.h"
#include <XL/xl.h>
#include <GL/gl.h>
#include "id.h"

typedef struct
{
	unsigned int mesh;
	unsigned int textures;
	GLuint list;
	dBodyID body;
	dGeomID geom;
} GADGETcube;

typedef struct
{
	unsigned int type;
	void *data;
} GADGET;

#define STORE ID_SMALL_STORE

static GADGET *store[STORE];

void
myGadgetCreateContext(void)
{
	idClearStore(GADGET, STORE, store);
}

static
void
Init(GADGET *gadget)
{
}

static
void
Term(GADGET *gadget)
{
}

void
myGenGadgets(unsigned int n, unsigned int *gadgets)
{
	idGenObjects(GADGET, STORE, store, Init, n, gadgets);
}

void
myDeleteGadgets(unsigned int n, unsigned int *gadgets)
{
	idDeleteObjects(store, Term, n, gadgets);
}

void
myGadgetLoadCube(unsigned int id, dWorldID world, dSpaceID space)
{
	unsigned int string;
	dMass mass;
	GADGETcube *cube;
	GADGET *gadget;

	gadget = store[id];

	gadget->type = MY_GADGET_CUBE;
	gadget->data = malloc(sizeof(GADGETcube));
	cube = gadget->data;

	xlGenMeshes(1, &cube->mesh);
	xlGenGroups(1, &cube->textures);

	xlGenStrings(1, &string);
	xlStringLoad(string, PACKAGE_DATADIR XL_STRING_PATH_SEPARATOR "Cube");
	xlMeshLoad(cube->mesh, string);
	xlStringUnload(string);
	xlStringLoad(string, PACKAGE_DATADIR XL_STRING_PATH_SEPARATOR "textures");
	xlMeshGenTextures(cube->mesh, string, cube->textures);
	xlStringUnload(string);
	xlDeleteStrings(1, &string);

	cube->list = glGenLists(1);

	glNewList(cube->list, GL_COMPILE);
	xlMeshDraw(cube->mesh, cube->textures);
	glEndList();

	cube->body = dBodyCreate(world);
	dMassSetBox(&mass, 1.25f, 2.0f, 2.0f, 2.0f);
	dMassAdjust(&mass, 1.25f);
	dBodySetMass(cube->body, &mass);
	cube->geom = dCreateBox(space, 2.0f, 2.0f, 2.0f);
	dGeomSetBody(cube->geom, cube->body);
}

void
myGadgetLoadTurbo(unsigned int id)
{
	GADGET *gadget;

	gadget = store[id];

	gadget->type = MY_GADGET_TURBO;
}

void
myGadgetLoadFake(unsigned int id)
{
	GADGET *gadget;

	gadget = store[id];

	gadget->type = MY_GADGET_FAKE;
}

void
myGadgetUnload(unsigned int id)
{
	GADGET *gadget;

	gadget = store[id];
	
	switch(gadget->type)
	{
		case MY_GADGET_CUBE:
		{
			GADGETcube *cube;

			cube = gadget->data;

			dGeomDestroy(cube->geom);
			dBodyDestroy(cube->body);

			glDeleteLists(cube->list, 1);
			glDeleteTextures(xlGroupLength(cube->textures), xlGroupArray(cube->textures));
			xlGroupUnload(cube->textures);
			xlMeshUnload(cube->mesh);

			xlDeleteGroups(1, &cube->textures);
			xlDeleteMeshes(1, &cube->mesh);

			free(cube);
		}
			break;
		default:
			break;
	}
}

void
myGadgetsUnload(unsigned int n, unsigned int *gadgets)
{
	unsigned int i;
	
	for(i = 0; i < n; i++) myGadgetUnload(gadgets[i]);
}

unsigned int
myGadgetType(unsigned int id)
{
	GADGET *gadget;

	gadget = store[id];

	return gadget->type;
}

void
myGadgetFrame(unsigned int id)
{
	GADGET *gadget;

	gadget = store[id];

	switch(gadget->type)
	{
		case MY_GADGET_CUBE:
		{
			const float *p, *r;
			float m[16];
			GADGETcube *cube;

			cube = gadget->data;

			p = dBodyGetPosition(cube->body);
			r = dBodyGetRotation(cube->body);
			xlMatrixFromODESituation(p, r, m);
			glPushMatrix();
				glMultMatrixf(m);
				glCallList(cube->list);
			glPopMatrix();
		}
		default:
			break;
	};
}

void
myGadgetEnable(unsigned int id)
{
	GADGET *gadget;

	gadget = store[id];

	switch(gadget->type)
	{
		case MY_GADGET_CUBE:
		{
			GADGETcube *cube;

			cube = gadget->data;

			dBodyEnable(cube->body);
			dGeomEnable(cube->geom);
		}
		default:
			break;
	};
}


void
myGadgetDisable(unsigned int id)
{
	GADGET *gadget;

	gadget = store[id];

	switch(gadget->type)
	{
		case MY_GADGET_CUBE:
		{
			GADGETcube *cube;

			cube = gadget->data;

			dBodyDisable(cube->body);
			dGeomDisable(cube->geom);
		}
		default:
			break;
	};
}

void
myGadgetPlace(unsigned int id, const float *position)
{
	GADGET *gadget;

	gadget = store[id];

	switch(gadget->type)
	{
		case MY_GADGET_CUBE:
		{
			GADGETcube *cube;

			cube = gadget->data;

			dBodySetPosition(cube->body, position[0], position[1], position[2]);
			dBodySetLinearVel(cube->body, 0.0f, 0.0f, 0.0f);
			dBodySetAngularVel(cube->body, 0.0f, 0.0f, 0.0f);

		}
		default:
			break;
	};
}
