// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "mybox.h"
#include <XL/xl.h>
#include <GL/gl.h>
#include "id.h"

typedef struct
{
	unsigned int mesh;
	unsigned int textures;
	GLuint list;
	float sphere[4];
	float matrix[16];
} BOX;

#define STORE ID_SMALL_STORE

static BOX *store[STORE];

void
myBoxCreateContext(void)
{
	idClearStore(BOX, STORE, store);
}

static
void
Init(BOX *box)
{
	xlGenMeshes(1, &box->mesh);
	xlGenGroups(1, &box->textures);
}

static
void
Term(BOX *box)
{
	xlDeleteGroups(1, &box->textures);
	xlDeleteMeshes(1, &box->mesh);
}

void
myGenBoxes(unsigned int n, unsigned int *boxes)
{
	idGenObjects(BOX, STORE, store, Init, n, boxes);
}

void
myDeleteBoxes(unsigned int n, unsigned int *boxes)
{
	idDeleteObjects(store, Term, n, boxes);
}

void
myBoxLoad(unsigned int id, const float *sphere)
{
	unsigned int string;
	BOX *box;

	box = store[id];

	xlMatrixLoadIdentity(box->matrix);

	xlGenStrings(1, &string);
	xlStringLoad(string, PACKAGE_DATADIR XL_STRING_PATH_SEPARATOR "Box");
	xlMeshLoad(box->mesh, string);
	xlStringUnload(string);
	xlStringLoad(string, PACKAGE_DATADIR XL_STRING_PATH_SEPARATOR "textures");
	xlMeshGenTextures(box->mesh, string, box->textures);
	xlStringUnload(string);
	xlDeleteStrings(1, &string);

	box->list = glGenLists(1);

	glNewList(box->list, GL_COMPILE);
	xlMeshDraw(box->mesh, box->textures);
	glEndList();

	xlSphereCopy(sphere, box->sphere);
}

void
myBoxUnload(unsigned int id)
{
	BOX *box;

	box = store[id];

	glDeleteLists(box->list, 1);
	glDeleteTextures(xlGroupLength(box->textures), xlGroupArray(box->textures));
	xlGroupUnload(box->textures);
	xlMeshUnload(box->mesh);

	xlDeleteGroups(1, &box->textures);
	xlDeleteMeshes(1, &box->mesh);
}

void
myBoxesUnload(unsigned int n, unsigned int *boxes)
{
	unsigned int i;

	for(i = 0; i < n; i++) myBoxUnload(boxes[i]);
}

const float *
myBoxSphere(unsigned int id)
{
	BOX *box;

	box = store[id];

	return box->sphere;
}

void
myBoxPlace(unsigned int id, const float *position)
{
	BOX *box;

	box = store[id];

	xlPointCopy(position, box->sphere);
}

void
myBoxFrame(unsigned int id)
{
	BOX *box;

	box = store[id];

	xlMatrixRotateInPlace(box->matrix, xlVectorY, 18);
	glPushMatrix();
		glTranslatef(box->sphere[0], box->sphere[1], box->sphere[2]);
		glMultMatrixf(box->matrix);
		glCallList(box->list);
	glPopMatrix();
}
