// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "XL/mesh.h"
#include "XL/error.h"
#include "XL/group.h"
#include "XL/string.h"
#include "XL/image.h"
#include "XL/frustum.h"
#include "XL/xl.h"
#include <stdio.h>
#include <GL/gl.h>
#include <ode/ode.h>
#include "id.h"

typedef struct
{
	int magic;
	unsigned int vertices;
	unsigned int textures;
	unsigned int materials;
	unsigned int faces;
} MESHheader;

typedef struct
{
	float coord[3];
	float normal[3];
} MESHvertex;

typedef struct
{
	char name[64];
} MESHtexture;

typedef struct
{
	unsigned int texture;
	float alpha;
	float emit;
	float ambient[3];
	float diffuse[3];
	float specular[3];
} MESHmaterial;

typedef struct
{
	unsigned int vertices[3];
	float texcoords[3][2];
	float normal[3];
	unsigned int material;
} MESHface;

typedef struct
{
	unsigned int groups[4];
} MESH;

#define STORE ID_MEDIUM_STORE

static MESH *store[STORE];

void
xlMeshCreateContext(void)
{
	idClearStore(MESH, STORE, store);
}

static
void
Init(MESH *mesh)
{
}

static
void
Term(MESH *mesh)
{
}

void
xlGenMeshes(unsigned int n, unsigned int *meshes)
{
	idGenObjects(MESH, STORE, store, Init, n, meshes);
}

void
xlDeleteMeshes(unsigned int n, unsigned int *meshes)
{
	idDeleteObjects(store, Term, n, meshes);
}

void
xlMeshLoad(unsigned int id, unsigned int pathname)
{
	FILE *stream;

	stream = fopen(xlStringData(pathname), "rb");
	if(stream)
	{
		MESHheader header;
		MESH *mesh;

		mesh = store[id];

		fread(&header, 1, sizeof(MESHheader), stream);
		xlGenGroups(4, mesh->groups);
		xlGroupRead(mesh->groups[0], header.vertices, sizeof(MESHvertex), stream);
		xlGroupRead(mesh->groups[1], header.textures, sizeof(MESHtexture), stream);
		xlGroupRead(mesh->groups[2], header.materials, sizeof(MESHmaterial), stream);
		xlGroupRead(mesh->groups[3], header.faces, sizeof(MESHface), stream);

		fclose(stream);
	}
	else xlErrorSet(XL_INVALID);
}

void
xlMeshSave(unsigned int id, unsigned int pathname)
{
	FILE *stream;

	stream = fopen(xlStringData(pathname), "wb");
	if(stream)
	{
		MESHheader header;
		MESH *mesh;

		mesh = store[id];

		header.vertices = xlGroupLength(mesh->groups[0]);
		header.textures = xlGroupLength(mesh->groups[1]);
		header.materials = xlGroupLength(mesh->groups[2]);
		header.faces = xlGroupLength(mesh->groups[3]);

		fwrite(&header, 1, sizeof(MESHheader), stream);
		xlGroupWrite(mesh->groups[0], stream);
		xlGroupWrite(mesh->groups[1], stream);
		xlGroupWrite(mesh->groups[2], stream);
		xlGroupWrite(mesh->groups[3], stream);

		fclose(stream);
	}
	else xlErrorSet(XL_INVALID);
}

void
xlMeshCompile(unsigned int id, unsigned int pathname)
{
	FILE *stream;

	stream = fopen(xlStringData(pathname), "r");
	if(stream)
	{
		char buffer[256];
		unsigned int i;
		MESHheader header;
		MESHvertex *vertices;
		MESHtexture *textures;
		MESHmaterial *materials;
		MESHface *faces;
		MESH *mesh;

		mesh = store[id];

		xlGenGroups(4, mesh->groups);

		fgets(buffer, 256, stream);
		sscanf(buffer, "vertices %i", &header.vertices);
		xlGroupAllocate(mesh->groups[0], header.vertices, sizeof(MESHvertex));
		vertices = xlGroupArray(mesh->groups[0]);
		for(i = 0; i < header.vertices; i++)
		{
			fgets(buffer, 256, stream);
			sscanf(buffer, "coord %f %f %f", &vertices[i].coord[0], &vertices[i].coord[1], &vertices[i].coord[2]);
			fgets(buffer, 256, stream);
			sscanf(buffer, "normal %f %f %f", &vertices[i].normal[0], &vertices[i].normal[1], &vertices[i].normal[2]);
		}

		fgets(buffer, 256, stream);
		sscanf(buffer, "textures %i", &header.textures);
		xlGroupAllocate(mesh->groups[1], header.textures, sizeof(MESHtexture));
		textures = xlGroupArray(mesh->groups[1]);
		for(i = 0; i < header.textures; i++)
		{
			fgets(buffer, 256, stream);
			strncpy(textures[i].name, buffer + strlen("name "), strlen(buffer) - strlen("name ") - 1);
			textures[i].name[strlen(buffer) - strlen("name ") - 1] = '\0';
		}

		fgets(buffer, 256, stream);
		sscanf(buffer, "materials %i", &header.materials);
		xlGroupAllocate(mesh->groups[2], header.materials, sizeof(MESHmaterial));
		materials = xlGroupArray(mesh->groups[2]);
		for(i = 0; i < header.materials; i++)
		{
			fgets(buffer, 256, stream);
			sscanf(buffer, "texture %i", &materials[i].texture);
			fgets(buffer, 256, stream);
			sscanf(buffer, "alpha %f", &materials[i].alpha);
			fgets(buffer, 256, stream);
			sscanf(buffer, "emit %f", &materials[i].emit);
			fgets(buffer, 256, stream);
			sscanf(buffer, "ambient %f %f %f", &materials[i].ambient[0], &materials[i].ambient[1], &materials[i].ambient[2]);
			fgets(buffer, 256, stream);
			sscanf(buffer, "diffuse %f %f %f", &materials[i].diffuse[0], &materials[i].diffuse[1], &materials[i].diffuse[2]);
			fgets(buffer, 256, stream);
			sscanf(buffer, "specular %f %f %f", &materials[i].specular[0], &materials[i].specular[1], &materials[i].specular[2]);
		}

		fgets(buffer, 256, stream);
		sscanf(buffer, "faces %i", &header.faces);
		xlGroupAllocate(mesh->groups[3], header.faces, sizeof(MESHface));
		faces = xlGroupArray(mesh->groups[3]);
		for(i = 0; i < header.faces; i++)
		{
			fgets(buffer, 256, stream);
			sscanf(buffer, "vertices %i %i %i", &faces[i].vertices[0], &faces[i].vertices[1], &faces[i].vertices[2]);
			fgets(buffer, 256, stream);
			sscanf(buffer, "texcoords %f %f %f %f %f %f", &faces[i].texcoords[0][0], &faces[i].texcoords[0][1], &faces[i].texcoords[1][0], &faces[i].texcoords[1][1], &faces[i].texcoords[2][0], &faces[i].texcoords[2][1]);
			fgets(buffer, 256, stream);
			sscanf(buffer, "normal %f %f %f", &faces[i].normal[0], &faces[i].normal[1], &faces[i].normal[2]);
			fgets(buffer, 256, stream);
			sscanf(buffer, "material %i", &faces[i].material);
		}

		fclose(stream);
	}
	else xlErrorSet(XL_INVALID);
}

void
xlMeshDecompile(unsigned int id, unsigned int pathname)
{
	FILE *stream;

	stream = fopen(xlStringData(pathname), "w");
	if(stream)
	{
		unsigned int i;
		MESHheader header;
		MESHvertex *vertices;
		MESHtexture *textures;
		MESHmaterial *materials;
		MESHface *faces;
		MESH *mesh;

		mesh = store[id];

		header.vertices = xlGroupLength(mesh->groups[0]);
		vertices = xlGroupArray(mesh->groups[0]);
		header.textures = xlGroupLength(mesh->groups[1]);
		textures = xlGroupArray(mesh->groups[1]);
		header.materials = xlGroupLength(mesh->groups[2]);
		materials = xlGroupArray(mesh->groups[2]);
		header.faces = xlGroupLength(mesh->groups[3]);
		faces = xlGroupArray(mesh->groups[3]);

		fprintf(stream, "vertices %i\n", header.vertices);
		for(i = 0; i < header.vertices; i++)
		{
			fprintf(stream, "coord %f %f %f\n", vertices[i].coord[0], vertices[i].coord[1], vertices[i].coord[2]);
			fprintf(stream, "normal %f %f %f\n", vertices[i].normal[0], vertices[i].normal[1], vertices[i].normal[2]);
		}

		fprintf(stream, "textures %i\n", header.textures);
		for(i = 0; i < header.textures; i++)
		{
			fprintf(stream, "name %s\n", textures[i].name);
		}

		fprintf(stream, "materials %i\n", header.materials);
		for(i = 0; i < header.materials; i++)
		{
			fprintf(stream, "texture %i\n", materials[i].texture);
			fprintf(stream, "alpha %f\n", materials[i].alpha);
			fprintf(stream, "emit %f\n", materials[i].emit);
			fprintf(stream, "ambient %f %f %f\n", materials[i].ambient[0], materials[i].ambient[1], materials[i].ambient[2]);
			fprintf(stream, "diffuse %f %f %f\n", materials[i].diffuse[0], materials[i].diffuse[1], materials[i].diffuse[2]);
			fprintf(stream, "specular %f %f %f\n", materials[i].specular[0], materials[i].specular[1], materials[i].specular[2]);
		}

		fprintf(stream, "faces %i\n", header.faces);
		for(i = 0; i < header.faces; i++)
		{
			fprintf(stream, "vertices %i %i %i\n", faces[i].vertices[0], faces[i].vertices[1], faces[i].vertices[2]);
			fprintf(stream, "texcoords %f %f %f %f %f %f\n", faces[i].texcoords[0][0], faces[i].texcoords[0][1], faces[i].texcoords[1][0], faces[i].texcoords[1][1], faces[i].texcoords[2][0], faces[i].texcoords[2][1]);
			fprintf(stream, "normal %f %f %f\n", faces[i].normal[0], faces[i].normal[1], faces[i].normal[2]);
			fprintf(stream, "material %i\n", faces[i].material);
		}

		fclose(stream);
	}
	else xlErrorSet(XL_INVALID);
}

void
xlMeshUnload(unsigned int id)
{
	MESH *mesh;

	mesh = store[id];

	xlDeleteGroups(4, mesh->groups);
}

void
xlMeshesUnload(unsigned int n, unsigned int *meshes)
{
	unsigned int i;

	for(i = 0; i < n; i++) xlMeshUnload(meshes[i]);
}

void
xlMeshGenTextures(unsigned int id, unsigned int path, unsigned int textures)
{
	unsigned int i;
	unsigned int strings[2];
	unsigned int length;
	unsigned int image;
	MESHtexture *src;
	GLuint *dst;
	MESH *mesh;

	mesh = store[id];

	length = xlGroupLength(mesh->groups[1]);

	xlGroupAllocate(textures, length, sizeof(GLuint));

	src = xlGroupArray(mesh->groups[1]);
	dst = xlGroupArray(textures);
	
	glGenTextures(length, dst);

	xlGenImages(1, &image);
	xlGenStrings(2, strings);
	for(i = 0; i < length; i++)
	{
		glBindTexture(GL_TEXTURE_2D, dst[i]);

		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST_MIPMAP_NEAREST); //
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);

		xlStringLoad(strings[0], src[i].name);
		xlStringConcatPath2(strings[1], path, strings[0]);
		xlImageLoadNAME(image, strings[1]);
		xlStringsUnload(2, strings);
		xlImageBuild2DMipmaps(image);
		xlImageUnload(image);
	}
	xlDeleteStrings(2, strings);
	xlDeleteImages(1, &image);
}

void
xlMeshTriMeshDataBuild(unsigned int id, dTriMeshDataID data)
{
	MESHvertex *vertices;
	MESHface *faces;
	MESH *mesh;

	mesh = store[id];

	vertices = xlGroupArray(mesh->groups[0]);
	faces = xlGroupArray(mesh->groups[3]);

	dGeomTriMeshDataBuildSingle1(data, vertices[0].coord, sizeof(MESHvertex), xlGroupLength(mesh->groups[0]), faces[0].vertices, xlGroupLength(mesh->groups[3]) * 3, sizeof(MESHface), faces[0].normal);
}

void
xlMeshDraw(unsigned int id, unsigned int textures)
{
	unsigned int i, j;
	MESHvertex *vertices;
	GLuint *gltextures;
	MESHmaterial *materials;
	MESHface *faces;
	MESH *mesh;

	mesh = store[id];

	vertices = xlGroupArray(mesh->groups[0]);
	gltextures = xlGroupArray(textures);
	materials = xlGroupArray(mesh->groups[2]);
	faces = xlGroupArray(mesh->groups[3]);

	for(i = 0; i < xlGroupLength(mesh->groups[3]); i++)
	{
		float ambient[4] =
		{
			materials[faces[i].material].ambient[0],
			materials[faces[i].material].ambient[1],
			materials[faces[i].material].ambient[2],
			materials[faces[i].material].alpha
		};
		float specular[4] =
		{
			materials[faces[i].material].specular[0],
			materials[faces[i].material].specular[1],
			materials[faces[i].material].specular[2],
			materials[faces[i].material].alpha
		};
		float diffuse[4] =
		{
			materials[faces[i].material].diffuse[0],
			materials[faces[i].material].diffuse[1],
			materials[faces[i].material].diffuse[2],
			materials[faces[i].material].alpha
		};
		glBindTexture(GL_TEXTURE_2D, gltextures[materials[faces[i].material].texture]);
		glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, ambient);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, diffuse);
		glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, specular);
		glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, ambient);
		glBegin(GL_TRIANGLES);
		for(j = 0; j < 3; j++)
		{
			glTexCoord2fv(faces[i].texcoords[j]);
			glNormal3fv(vertices[faces[i].vertices[j]].normal);
			glVertex3fv(vertices[faces[i].vertices[j]].coord);
		}
		glEnd();
	}
}

void
xlMeshFrustumDraw(unsigned int id, unsigned int textures, unsigned int frustum)
{
	unsigned int i, j;
	MESHvertex *vertices;
	GLuint *gltextures;
	MESHmaterial *materials;
	MESHface *faces;
	MESH *mesh;

	mesh = store[id];

	vertices = xlGroupArray(mesh->groups[0]);
	gltextures = xlGroupArray(textures);
	materials = xlGroupArray(mesh->groups[2]);
	faces = xlGroupArray(mesh->groups[3]);

	for(i = 0; i < xlGroupLength(mesh->groups[3]); i++)
	{
		float triangle[9] =
		{
			vertices[faces[i].vertices[0]].coord[0],
			vertices[faces[i].vertices[0]].coord[1],
			vertices[faces[i].vertices[0]].coord[2],
			vertices[faces[i].vertices[1]].coord[0],
			vertices[faces[i].vertices[1]].coord[1],
			vertices[faces[i].vertices[1]].coord[2],
			vertices[faces[i].vertices[2]].coord[0],
			vertices[faces[i].vertices[2]].coord[1],
			vertices[faces[i].vertices[2]].coord[2],
		};
		if(xlFrustumTriangle(frustum, triangle))
		{
			float ambient[4] =
			{
				materials[faces[i].material].ambient[0],
				materials[faces[i].material].ambient[1],
				materials[faces[i].material].ambient[2],
				materials[faces[i].material].alpha
			};
			float specular[4] =
			{
				materials[faces[i].material].specular[0],
				materials[faces[i].material].specular[1],
				materials[faces[i].material].specular[2],
				materials[faces[i].material].alpha
			};
			float diffuse[4] =
			{
				materials[faces[i].material].diffuse[0],
				materials[faces[i].material].diffuse[1],
				materials[faces[i].material].diffuse[2],
				materials[faces[i].material].alpha
			};
			glBindTexture(GL_TEXTURE_2D, gltextures[materials[faces[i].material].texture]);
			glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, ambient);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, diffuse);
			glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, specular);
			glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, ambient);
			glBegin(GL_TRIANGLES);
			for(j = 0; j < 3; j++)
			{
				glTexCoord2fv(faces[i].texcoords[j]);
				glNormal3fv(vertices[faces[i].vertices[j]].normal);
				glVertex3fv(vertices[faces[i].vertices[j]].coord);
			}
			glEnd();
		}
	}
}
