// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "XL/math.h"
#include <math.h>
#include <stdio.h>
#include <string.h>

const float xlPi = 3.14159265358979323846f;

float
xlRadToDeg(float r)
{
	return r * (180.0f / xlPi);
}

float
xlDegToRad(float r)
{
	return r * (xlPi / 180.0f);
}

unsigned int
xlNextTwoPower(float x)
{
	return pow(2.0f, ceil(log(x) / log(2.0f)));
}

float
xlClamp(float x, float a, float b)
{
	return x < a ? a : (b < x ? b : x);
}

float
xlMax(float a, float b)
{
	return a > b ? a : b;
}

const float xlPointZero[3] = { 0.0f, 0.0f, 0.0f };

void
xlPointPrint(const float *a)
{
	printf("(%f, %f, %f)\n", a[0], a[1], a[2]);
}

void
xlPointCopy(const float *a, float *result)
{
	memcpy(result, a, 3 * sizeof(float));
}

void
xlPointLoad(float *a, float x, float y, float z)
{
	a[0] = x; a[1] = y; a[2] = z;
}

void
xlPointLoadZero(float *a)
{
	xlPointLoad(a, 0.0f, 0.0f, 0.0f);
}

void
xlPointAdd(const float *a, const float *b, float *result)
{
	unsigned int i;

	for(i = 0; i < 3; i++)
		result[i] = a[i] + b[i];
}

void
xlPointAddInPlace(float *a, const float *b)
{
	unsigned int i;

	for(i = 0; i < 3; i++)
		a[i] += b[i];
}

void
xlPointSub(const float *a, const float *b, float *result)
{
	unsigned int i;

	for(i = 0; i < 3; i++)
		result[i] = a[i] - b[i];
}

void
xlPointSubInPlace(float *a, const float *b)
{
	unsigned int i;

	for(i = 0; i < 3; i++)
		a[i] -= b[i];
}

const float xlVectorX[3] = { 1.0f, 0.0f, 0.0f };
const float xlVectorY[3] = { 0.0f, 1.0f, 0.0f };
const float xlVectorZ[3] = { 0.0f, 0.0f, 1.0f };
const float xlVectorNegX[3] = { -1.0f, 0.0f, 0.0f };
const float xlVectorNegY[3] = { 0.0f, -1.0f, 0.0f };
const float xlVectorNegZ[3] = { 0.0f, 0.0f, -1.0f };

void
xlVectorPrint(const float *a)
{
	printf("(%f, %f, %f)\n", a[0], a[1], a[2]);
}

void
xlVectorCopy(const float *a, float *result)
{
	memcpy(result, a, 3 * sizeof(float));
}

void
xlVectorLoad(float *a, float x, float y, float z)
{
	a[0] = x; a[1] = y; a[2] = z;
}

void
xlVectorLoadX(float *a)
{
	xlVectorLoad(a, 1.0f, 0.0f, 0.0f);
}

void
xlVectorLoadY(float *a)
{
	xlVectorLoad(a, 0.0f, 1.0f, 0.0f);
}

void
xlVectorLoadZ(float *a)
{
	xlVectorLoad(a, 0.0f, 0.0f, 1.0f);
}

void
xlVectorLoadNegX(float *a)
{
	xlVectorLoad(a, -1.0f, 0.0f, 0.0f);
}

void
xlVectorLoadNegY(float *a)
{
	xlVectorLoad(a, 0.0f, -1.0f, 0.0f);
}

void
xlVectorLoadNegZ(float *a)
{
	xlVectorLoad(a, 0.0f, 0.0f, -1.0f);
}

void
xlVectorAdd(const float *a, const float *b, float *result)
{
	unsigned int i;

	for(i = 0; i < 3; i++)
		result[i] = a[i] + b[i];
}

void
xlVectorAddInPlace(float *a, const float *b)
{
	unsigned int i;

	for(i = 0; i < 3; i++)
		a[i] += b[i];
}

void
xlVectorSub(const float *a, const float *b, float *result)
{
	unsigned int i;
	
	for(i = 0; i < 3; i++)
		result[i] = a[i] - b[i];
}

void
xlVectorSubInPlace(float *a, const float *b)
{
	unsigned int i;

	for(i = 0; i < 3; i++)
		a[i] -= b[i];
}

void
xlVectorScalarProduct(const float *a, float b, float *result)
{
	unsigned int i;

	for(i = 0; i < 3; i++)
		result[i] = a[i] * b;
}

void
xlVectorScalarProductInPlace(float *a, float b)
{
	unsigned int i;

	for(i = 0; i < 3; i++)
		a[i] *= b;
}

void
xlVectorNorm(const float *a, float *result)
{
	*result = sqrt(a[0] * a[0] + a[1] * a[1] + a[2] * a[2]);
}

float
xlVectorNormFunc(const float *a)
{
	return sqrt(a[0] * a[0] + a[1] * a[1] + a[2] * a[2]);
}

void
xlVectorDotProduct(const float *a, const float *b, float *result)
{
	*result = a[0] * b[0] + a[1] * b[1] + a[2] * b[2];
}

float
xlVectorDotProductFunc(const float *a, const float *b)
{
	return a[0] * b[0] + a[1] * b[1] + a[2] * b[2];
}

void
xlVectorCrossProduct(const float *a, const float *b, float *result)
{
	result[0] = a[1] * b[2] - a[2] * b[1];
	result[1] = a[2] * b[0] - a[0] * b[2];
	result[2] = a[0] * b[1] - a[1] * b[0];
}

void
xlVectorCrossProductInPlace(float *a, const float *b)
{
	float result[3];

	xlVectorCrossProduct(a, b, result);
	xlVectorCopy(result, a);
}

void
xlVectorNormal(const float *a, float *result)
{
	float norm;

	xlVectorNorm(a, &norm);
	xlVectorScalarProduct(a, 1.0f / norm, result);
}

void
xlVectorNormalInPlace(float *a)
{
	xlVectorScalarProductInPlace(a, 1.0f / xlVectorNormFunc(a));
}

void
xlVectorMatrixProduct(const float *a, const float *b, float *result)
{
	result[0] = b[0] * a[0] + b[4] * a[1] + b[8] * a[2] + b[12];
	result[1] = b[1] * a[0] + b[5] * a[1] + b[9] * a[2] + b[13];
	result[2] = b[2] * a[0] + b[6] * a[1] + b[10] * a[2] + b[14];
}

void
xlVectorMatrixProductInPlace(float *a, const float *b)
{
	float result[3];

	xlVectorMatrixProduct(a, b, result);
	xlVectorCopy(result, a);
}

void
xlVectorRotate(const float *a, const float *axis, float angle, float *result)
{
	const float c = cos(xlDegToRad(angle));
	const float s = sin(xlDegToRad(angle));
	const float matrix[16] =
	{
		axis[0] * axis[0] * (1.0f - c) + c, axis[1] * axis[0] * (1.0f - c) + axis[2] * s, axis[0] * axis[2] * (1.0f - c) - axis[1] * s, 0.0f,
		axis[0] * axis[1] * (1.0f - c) - axis[2] * s, axis[1] * axis[1] * (1.0f - c) + c, axis[1] * axis[2] * (1.0f - c) + axis[0] * s, 0.0f,
		axis[0] * axis[2] * (1.0f - c) + axis[1] * s, axis[1] * axis[2] * (1.0f - c) - axis[0] * s, axis[2] * axis[2] * (1.0f - c) + c, 0.0f,
		0.0f, 0.0f, 0.0f, 1.0f
	};

	xlVectorMatrixProduct(a, matrix, result);
}

void
xlVectorRotateInPlace(float *a, const float *axis, float angle)
{
	float result[3];

	xlVectorRotate(a, axis, angle, result);
	xlVectorCopy(result, a);
}

void
xlVectorAngle(const float *a, const float *b, float *angle)
{
	float dp, na, nb;

	xlVectorDotProduct(a, b, &dp);
	xlVectorNorm(a, &na);
	xlVectorNorm(b, &nb);
	*angle = xlRadToDeg(acos(dp / (na * nb)));
}

float
xlVectorAngleFunc(const float *a, const float *b)
{
	float dp, na, nb;

	xlVectorDotProduct(a, b, &dp);
	xlVectorNorm(a, &na);
	xlVectorNorm(b, &nb);
	return xlRadToDeg(acos(dp / (na * nb)));
}

const float xlMatrixIdentity[16] =
{
	1.0f, 0.0f, 0.0f, 0.0f,
	0.0f, 1.0f, 0.0f, 0.0f,
	0.0f, 0.0f, 1.0f, 0.0f,
	0.0f, 0.0f, 0.0f, 1.0f,
};

void
xlMatrixPrint(const float *a)
{
	printf("[(%f, %f, %f, %f), (%f, %f, %f, %f), (%f, %f, %f, %f), (%f, %f, %f, %f)]\n", a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8], a[9], a[10], a[11], a[12], a[13], a[14], a[15]);
}

void
xlMatrixCopy(const float *a, float *result)
{
	memcpy(result, a, 16 * sizeof(float));
}

void
xlMatrixLoad(float *x, float a, float b, float c, float d, float e, float f, float g, float h, float i, float j, float k, float l, float m, float n, float o, float p)
{
	x[0] = a; x[4] = e; x[8] = i; x[12] = m;
	x[1] = b; x[5] = f; x[9] = j; x[13] = n;
	x[2] = c; x[6] = g; x[10] = k; x[14] = o;
	x[3] = d; x[7] = h; x[11] = l; x[15] = p;
}

void
xlMatrixLoadIdentity(float *a)
{
	unsigned int i;

	for(i = 0; i < 16; i++)
		a[i] = (i % 5) == 0 ? 1.0f : 0.0f;
}

void
xlMatrixAdd(const float *a, const float *b, float *result)
{
	unsigned int i;

	for(i = 0; i < 16; i++)
		result[i] = a[i] + b[i];
}

void
xlMatrixAddInPlace(float *a, const float *b)
{
	unsigned int i;

	for(i = 0; i < 16; i++)
		a[i] += b[i];
}

void
xlMatrixSub(const float *a, const float *b, float *result)
{
	unsigned int i;

	for(i = 0; i < 16; i++)
		result[i] = a[i] - b[i];
}

void
xlMatrixSubInPlace(float *a, const float *b)
{
	unsigned int i;

	for(i = 0; i < 16; i++)
		a[i] -= b[i];
}

void
xlMatrixScalarProduct(const float *a, float b, float *result)
{
	unsigned int i;

	for(i = 0; i < 16; i++)
		result[i] = a[i] * b;
}

void
xlMatrixScalarProductInPlace(float *a, float b)
{
	unsigned int i;

	for(i = 0; i < 16; i++)
		a[i] *= b;
}

void
xlMatrixProduct(const float *a, const float *b, float *result)
{
	unsigned int i, j;

	for(j = 0; j < 4; j++)
		for(i = 0; i < 4; i++)
			result[j * 4 + i] = a[0 * 4 + i] * b[j * 4 + 0] + a[1 * 4 + i] * b[j * 4 + 1] + a[2 * 4 + i] * b[j * 4 + 2] + a[3 * 4 + i] * b[j * 4 + 3];
}

void
xlMatrixProductInPlace(float *a, const float *b)
{
	float result[16];

	xlMatrixProduct(a, b, result);
	xlMatrixCopy(result, a);
}

void
xlMatrixVectorProduct(const float *a, const float *b, float *result)
{
	result[0] = a[0] * b[0] + a[4] * b[1] + a[8] * b[2] + a[12];
	result[1] = a[1] * b[0] + a[5] * b[1] + a[9] * b[2] + a[13];
	result[2] = a[2] * b[0] + a[6] * b[1] + a[10] * b[2] + a[14];
}

void
xlMatrixVectorProductInPlace(const float *a, float *b)
{
	float result[3];

	xlMatrixVectorProduct(a, b, result);
	xlVectorCopy(result, b);
}

void
xlMatrixTranspose(const float *a, float *result)
{
	unsigned int i, j;

	for(j = 0; j < 4; j++)
		for(i = 0; i < 4; i++)
			result[j * 4 + i] = a[i * 4 + j];
}

void
xlMatrixTransposeInPlace(float *a)
{
	float result[16];

	xlMatrixTranspose(a, result);
	xlMatrixCopy(result, a);
}

void
xlMatrixScale(const float *a, const float *b, float *result)
{
	const float c[16] =
	{
		b[0], 0.0f, 0.0f, 0.0f,
		0.0f, b[1], 0.0f, 0.0f,
		0.0f, 0.0f, b[2], 0.0f,
		0.0f, 0.0f, 0.0f, 1.0f
	};

	xlMatrixProduct(a, c, result);
}

void
xlMatrixScaleInPlace(float *a, const float *b)
{
	float result[16];

	xlMatrixScale(a, b, result);
	xlMatrixCopy(result, a);
}

void
xlMatrixRotate(const float *a, const float *axis, float angle, float *result)
{
	const float c = cos(xlDegToRad(angle));
	const float s = sin(xlDegToRad(angle));
	const float b[16] = 
	{
		axis[0] * axis[0] * (1.0f - c) + c, axis[1] * axis[0] * (1.0f - c) + axis[2] * s, axis[0] * axis[2] * (1.0f - c) - axis[1] * s, 0.0f,
		axis[0] * axis[1] * (1.0f - c) - axis[2] * s, axis[1] * axis[1] * (1.0f - c) + c, axis[1] * axis[2] * (1.0f - c) + axis[0] * s, 0.0f,
		axis[0] * axis[2] * (1.0f - c) + axis[1] * s, axis[1] * axis[2] * (1.0f - c) - axis[0] * s, axis[2] * axis[2] * (1.0f - c) + c, 0.0f,
		0.0f, 0.0f, 0.0f, 1.0f
	};

	xlMatrixProduct(a, b, result);
}

void
xlMatrixRotateInPlace(float *a, const float *axis, float angle)
{
	float result[16];

	xlMatrixRotate(a, axis, angle, result);
	xlMatrixCopy(result, a);
}

void
xlMatrixTranslate(const float *a, const float *b, float *result)
{
	const float c[16] =
	{
		1.0f, 0.0f, 0.0f, 0.0f,
		0.0f, 1.0f, 0.0f, 0.0f,
		0.0f, 0.0f, 1.0f, 0.0f,
		b[0], b[1], b[2], 1.0f
	};

	xlMatrixProduct(a, c, result);
}

void
xlMatrixTranslateInPlace(float *a, const float *b)
{
	float result[16];

	xlMatrixTranslate(a, b, result);
	xlMatrixCopy(result, a);
}

void
xlMatrixLook(const float *position, const float *forward, const float *up, float *result)
{
	float right[3];

	xlVectorCrossProduct(forward, up, right);

	const float rotation[16] =
	{
		right[0], up[0], -forward[0], 0.0f,
		right[1], up[1], -forward[1], 0.0f,
		right[2], up[2], -forward[2], 0.0f,
		0.0f, 0.0f, 0.0f, 1.0f
	};

	xlVectorScalarProduct(position, -1.0f, right);
	xlMatrixTranslate(rotation, right, result);
}

void
xlMatrixLookAt(const float *eye, const float *center, const float *up, float *result)
{
	float tmp[3], forward[3];

	xlVectorSub(center, eye, tmp);
	xlVectorNormal(tmp, forward);
	xlMatrixLook(eye, forward, up, result);
}

void
xlMatrixRotation(const float *forward, const float *up, float *result)
{
	unsigned int i;
	float right[3];

	xlVectorCrossProduct(forward, up, right);
	for(i = 0; i < 3; i++)
	{
		result[0 * 4 + i] = right[i];
		result[1 * 4 + i] = up[i];
		result[2 * 4 + i] = -forward[i];
		result[3 * 4 + i] = result[i * 4 + 3] = 0.0f;
	}
	result[15] = 1.0f;
}

void
xlMatrixGetPosition(const float *a, float *result)
{
	result[0] = a[12];
	result[1] = a[13];
	result[2] = a[14];
}

void
xlMatrixGetRight(const float *a, float *result)
{
	float tmp[3] = {a[0], a[4], a[8]};

	xlVectorNormal(tmp, result);
}

void
xlMatrixGetUp(const float *a, float *result)
{
	float tmp[3] = {a[1], a[5], a[9]};
	xlVectorNormal(tmp, result);
}

void
xlMatrixGetForward(const float *a, float *result)
{
	float tmp[3] = {-a[2], -a[6], -a[10]};
	xlVectorNormal(tmp, result);
}

void
xlTrianglePrint(const float *a)
{
	printf("[(%f, %f, %f), (%f, %f, %f), (%f, %f, %f)]", a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8]);
}

void
xlTriangleCopy(const float *a, float *result)
{
	memcpy(result, a, 3 * 3 * sizeof(float));
}

void
xlTriangleGetBoundSphere(const float *a, float *result)
{
	result[0] = (a[0 * 3 + 0] + a[1 * 3 + 0] + a[2 * 3 + 0]) / 3.0f;
	result[1] = (a[0 * 3 + 1] + a[1 * 3 + 1] + a[2 * 3 + 1]) / 3.0f;
	result[2] = (a[0 * 3 + 2] + a[1 * 3 + 2] + a[2 * 3 + 2]) / 3.0f;
	result[3] = sqrt(pow(result[0] - a[0 * 3 + 0], 2) + pow(result[1] - a[0 * 3 + 1], 2) + pow(result[2] - a[0 * 3 + 2], 2));
}

void
xlSpherePrint(const float *a)
{
	printf("[(%f, %f, %f), %f]\n", a[0], a[1], a[2], a[3]);
}

void
xlSphereCopy(const float *a, float *result)
{
	memcpy(result, a, 4 * sizeof(float));
}

void
xlSphereLoad(float *a, float x, float y, float z, float r)
{
	a[0] = x; a[1] = y; a[2] = z; a[3] = r;
}

void
xlSphereMerge(const float *a, const float *b, float *result)
{
	float distancer[3];
	float distance;

	distancer[0] = b[0] - a[0];
	distancer[1] = b[1] - a[1];
	distancer[2] = b[2] - a[2];
	distance = sqrt(pow(distancer[0], 2) + pow(distancer[1], 2) + pow(distancer[2], 2));
	if(distance + b[3] > a[3])
	{
		if(distance + a[3] > b[3])
		{
			distancer[0] /= distance;
			distancer[1] /= distance;
			distancer[2] /= distance;
			result[3] = (distance + a[3] + b[3]) / 2.0f;
			result[0] = a[0] + distancer[0] * (result[3] - a[3]);
			result[1] = a[1] + distancer[1] * (result[3] - a[3]);
			result[2] = a[2] + distancer[2] * (result[3] - a[3]);
		}
		else xlSphereCopy(b, result);
	}
	else xlSphereCopy(a, result);
}

void
xlSphereMergeInPlace(float *a, const float *b)
{
	float result[4];

	xlSphereMerge(a, b, result);
	xlSphereCopy(result, a);
}
