// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "XL/image.h"
#include "XL/error.h"
#include "XL/math.h"
#include "XL/string.h"
#include "XL/xl.h"
#include <stdio.h>
#include <png.h>
#include <jpeglib.h>
#include <GL/glu.h>
#include "id.h"

typedef struct
{
	unsigned int width;
	unsigned int height;
	unsigned int bpp;
	unsigned char *pixels;
} IMAGE;

#define STORE ID_MEDIUM_STORE

static IMAGE *store[STORE];

void
xlImageCreateContext(void)
{
	idClearStore(IMAGE, STORE, store);
}

static
void
Init(IMAGE *image)
{
}

static
void
Term(IMAGE *image)
{
}

void
xlGenImages(unsigned int n, unsigned int *images)
{
	idGenObjects(IMAGE, STORE, store, Init, n, images);
}

void
xlDeleteImages(unsigned int n, unsigned int *images)
{
	idDeleteObjects(store, Term, n, images);
}

static
GLenum
Format(unsigned int bpp)
{
	GLenum format;

	switch(bpp)
	{
		case 3: format = GL_RGB; break;
		case 4: format = GL_RGBA; break;
		default: xlErrorSet(XL_INVALID); break;
	}
	return format;
}

void
xlImageLoadPNG(unsigned int id, unsigned int pathname)
{
	FILE *stream;

	stream = fopen(xlStringData(pathname), "rb");
	if(stream)
	{
		unsigned char signature[8]; 

		fread(signature, 1, 8, stream);
		if(png_check_sig(signature, 8))
		{
			png_structp png;

			png = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
			if(png)
			{
				png_infop info;

				info = png_create_info_struct(png);
				if(info)
				{
					if(!setjmp(png_jmpbuf(png)))
					{
						unsigned long width, height, i;
						int depth, type;
						png_bytepp rows;
						IMAGE *image;

						image = store[id];

						png_init_io(png, stream);
						png_set_sig_bytes(png, 8);
						png_read_info(png, info);
						png_get_IHDR(png, info, &width, &height, &depth, &type, NULL, NULL, NULL);
						image->width = xlNextTwoPower(width);
						image->height = xlNextTwoPower(height);
						image->bpp = (type & PNG_COLOR_MASK_ALPHA) ? 4 : 3;

						if(depth > 8) png_set_strip_16(png);
						if(type == PNG_COLOR_TYPE_GRAY || type == PNG_COLOR_TYPE_GRAY_ALPHA) png_set_gray_to_rgb(png);
						if(type == PNG_COLOR_TYPE_PALETTE) png_set_palette_to_rgb(png);
						png_read_update_info(png, info);
						image->pixels = malloc(image->width * image->height * image->bpp * sizeof(unsigned char));
						rows = malloc(height * sizeof(png_bytep));
						for(i = 0; i < height; i++) rows[height - 1 - i] = image->pixels + i * width * image->bpp;
						png_read_image(png, rows);
						free(rows);

						if(width != image->width || height != image->height)
							gluScaleImage(Format(image->bpp), width, height, GL_UNSIGNED_BYTE, image->pixels, image->width, image->height, GL_UNSIGNED_BYTE, image->pixels);
					}
					else xlErrorSet(XL_INVALID);
					png_destroy_read_struct(&png, &info, NULL);
				}
				else
				{
					png_destroy_read_struct(&png, (png_infopp) NULL, (png_infopp) NULL);
					xlErrorSet(XL_INVALID);
				}
			}
			else xlErrorSet(XL_INVALID);
		}
		else xlErrorSet(XL_INVALID);
		fclose(stream);
	}
	else xlErrorSet(XL_INVALID);
}

void
xlImageSavePNG(unsigned int id, unsigned int pathname)
{
	FILE *stream;

	stream = fopen(xlStringData(pathname), "wb");
	if(stream)
	{
		png_structp png;

		png = png_create_write_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
		if(png)
		{
			png_infop info;

			info = png_create_info_struct(png);
			if (info)
			{
				if(!setjmp(png_jmpbuf(png)))
				{
					unsigned long i;
					png_bytepp rows;
					IMAGE *image;

					image = store[id];

					png_init_io(png, stream);
					png_set_IHDR(png, info, image->width, image->height, 8, PNG_COLOR_TYPE_RGB_ALPHA, PNG_INTERLACE_NONE, PNG_COMPRESSION_TYPE_BASE, PNG_FILTER_TYPE_BASE);

					png_write_info(png, info);
					rows = malloc(image->height * sizeof(png_bytep));
					for(i = 0; i < image->height; i++) rows[image->height - 1 - i] = image->pixels + i * image->width * image->bpp;
					png_write_image(png, rows);
					free(rows);
					png_write_end(png, info);

				}
				else xlErrorSet(XL_INVALID);
				png_destroy_write_struct(&png, &info);
			}
			else
			{
				png_destroy_write_struct(&png, (png_infopp) NULL);
				xlErrorSet(XL_INVALID);
			}
		}
		else xlErrorSet(XL_INVALID);
		fclose(stream);
	}
	else xlErrorSet(XL_INVALID);
}

void
xlImageLoadJPEG(unsigned int id, unsigned int pathname)
{
	FILE *stream;

	stream = fopen(xlStringData(pathname), "rb");
	if(stream)
	{
		struct jpeg_decompress_struct decompress;
		struct jpeg_error_mgr error;
		unsigned char **rows;
		unsigned int i, row;
		IMAGE *image;

		image = store[id];

		jpeg_create_decompress(&decompress);
		decompress.err = jpeg_std_error(&error);
		jpeg_stdio_src(&decompress, stream);
		jpeg_read_header(&decompress, TRUE);
		jpeg_start_decompress(&decompress);
		image->width = xlNextTwoPower(decompress.image_width);
		image->height = xlNextTwoPower(decompress.image_height);
		image->bpp = decompress.output_components;
		image->pixels = malloc(image->width * image->height * image->bpp * sizeof(unsigned char));
		rows = malloc(decompress.image_height * sizeof(unsigned char *));
		for(i = 0; i < decompress.image_height; i++) rows[i] = image->pixels + i * decompress.image_width * decompress.output_components;
		for(row = 0; decompress.output_scanline < decompress.output_height; row += jpeg_read_scanlines(&decompress, &rows[row], decompress.output_height - row));
		free(rows);
		jpeg_finish_decompress(&decompress);

		if(decompress.image_width != image->width || decompress.image_height != image->height)
			gluScaleImage(Format(image->bpp), decompress.image_width, decompress.image_height, GL_UNSIGNED_BYTE, image->pixels, image->width, image->height, GL_UNSIGNED_BYTE, image->pixels);

		jpeg_destroy_decompress(&decompress);
		fclose(stream);
	}
	else xlErrorSet(XL_INVALID);
}

void
xlImageSaveJPEG(unsigned int id, unsigned int pathname)
{
	FILE *stream;

	stream = fopen(xlStringData(pathname), "wb");
	if(stream)
	{
		struct jpeg_compress_struct compress;
		struct jpeg_error_mgr error;
		unsigned char **rows;
		unsigned int i, row;
		IMAGE *image;

		image = store[id];

		jpeg_create_compress(&compress);
		compress.err = jpeg_std_error(&error);
		jpeg_stdio_dest(&compress, stream);
		compress.image_width = image->width;
		compress.image_height = image->height;
		compress.input_components = image->bpp;
		compress.in_color_space = JCS_RGB;
		jpeg_set_defaults(&compress);
		jpeg_start_compress(&compress, TRUE);
		rows = (unsigned char **) malloc(compress.image_height * sizeof(unsigned char *));
		for(i = 0; i < compress.image_height; i++) rows[compress.image_height - 1 - i] = image->pixels + i * image->width * image->bpp;
		for(row = 0; compress.next_scanline < compress.image_height; row += jpeg_write_scanlines(&compress, &rows[row], compress.image_height - row));
		free(rows);
		jpeg_finish_compress(&compress);
		jpeg_destroy_compress(&compress);
		fclose(stream);
	}
	else xlErrorSet(XL_INVALID);
}

void
xlImageLoadTGA(unsigned int id, unsigned int pathname)
{
	FILE *stream;

	stream = fopen(xlStringData(pathname), "rb");
	if(stream)
	{
		const unsigned char tgamagic[12] = {0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0};
		unsigned char magic[12];

		fread(magic, 1, sizeof(magic), stream);
		if(!memcmp(tgamagic, magic, sizeof(tgamagic)))
		{
			unsigned char header[6];

			fread(header, 1, sizeof(header), stream);
			if(header[4] == 24 || header[4] == 32)
			{
				unsigned int width, height, i;
				IMAGE *image;

				image = store[id];

				width = (unsigned int) (header[1] << 8) + header[0];
				height = (unsigned int) (header[3] << 8) + header[2];
				image->width = xlNextTwoPower(width);
				image->height = xlNextTwoPower(height);
				image->bpp = header[4] >> 3;
				image->pixels = malloc(image->width * image->height * image->bpp * sizeof(unsigned char));
				fread(image->pixels, 1, image->width * image->height * image->bpp, stream);
				for(i = 0; i < image->width * image->height * image->bpp; i += image->bpp)
				{
					GLubyte swap;
					swap = image->pixels[i];
					image->pixels[i] = image->pixels[i + 2];
					image->pixels[i + 2] = swap;
				}

				if(width != image->width || height != image->height)
					gluScaleImage(Format(image->bpp), width, height, GL_UNSIGNED_BYTE, image->pixels, image->width, image->height, GL_UNSIGNED_BYTE, image->pixels);
			}
			else xlErrorSet(XL_INVALID);
		}
		else xlErrorSet(XL_INVALID);
		fclose(stream);
	}
	else xlErrorSet(XL_INVALID);
}

void
xlImageSaveTGA(unsigned int id, unsigned int pathname)
{
	FILE *stream;

	stream = fopen(xlStringData(pathname), "wb");
	if(stream)
	{
		const unsigned char magic[12] = {0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0};
		unsigned char header[6], *pixels;
		unsigned int i;
		IMAGE *image;

		image = store[id];

		fwrite(magic, 1, sizeof(magic), stream);
		header[0] = image->width & 0xFF;
		header[1] = (image->width >> 8);
		header[2] = image->height & 0xFF;
		header[3] = (image->height >> 8);
		header[4] = image->bpp << 3;
		header[5] = 0;
		fwrite(header, 1, sizeof(header), stream);
		pixels = malloc(image->width * image->height * image->bpp * sizeof(unsigned char));
		memcpy(pixels, image->pixels, image->width * image->height * image->bpp);
		for(i = 0; i < image->width * image->height * image->bpp; i += image->bpp)
		{
			GLubyte swap;
			swap = pixels[i];
			pixels[i] = pixels[i + 2];
			pixels[i + 2] = swap;
		}
		fwrite(pixels, 1, image->width * image->height * image->bpp, stream);
		free(pixels);
		fclose(stream);
	}
	else xlErrorSet(XL_INVALID);
}

void
xlImageLoadNAME(unsigned int id, unsigned int pathname)
{
	unsigned int strings[2];
	int error;

	xlGenStrings(2, strings);

	xlStringLoad(strings[0], ".png");
	xlStringConcat2(strings[1], pathname, strings[0]);
	xlImageLoadPNG(id, strings[1]);
	error = xlErrorGet();
	xlStringsUnload(2, strings);

	if(error != XL_NONE)
	{
		xlErrorSet(XL_NONE);

		xlStringLoad(strings[0], ".jpg");
		xlStringConcat2(strings[1], pathname, strings[0]);
		xlImageLoadJPEG(id, strings[1]);
		error = xlErrorGet();
		xlStringsUnload(2, strings);

		if(error != XL_NONE)
		{
			xlErrorSet(XL_NONE);

			xlStringLoad(strings[0], ".tga");
			xlStringConcat2(strings[1], pathname, strings[0]);
			xlImageLoadTGA(id, strings[1]);
			xlStringsUnload(2, strings);
		}
	}

	xlDeleteStrings(2, strings);
}

void
xlImageCapture(unsigned int id, int x, int y, unsigned int width, unsigned int height, GLenum format)
{
	IMAGE *image;

	image = store[id];

	image->width = width;
	image->height = height;
	switch(format)
	{
		case GL_RGB: image->bpp = 3; break;
		case GL_RGBA: image->bpp = 4; break;
	}
	image->pixels = malloc(image->width * image->height * image->bpp * sizeof(unsigned char));
	glReadPixels(x, y, image->width, image->height, format, GL_UNSIGNED_BYTE, image->pixels);
}

void
xlImageUnload(unsigned int id)
{
	IMAGE *image;

	image = store[id];

	free(image->pixels);
}

void
xlImagesUnload(unsigned int n, unsigned int *images)
{
	unsigned int i;

	for(i = 0; i < n; i++) xlImageUnload(images[i]);
}

void
xlImageBuild2DMipmaps(unsigned int id)
{
	IMAGE *image;

	image = store[id];

	gluBuild2DMipmaps(GL_TEXTURE_2D, Format(image->bpp), image->width, image->height, Format(image->bpp), GL_UNSIGNED_BYTE, image->pixels);
}

void
xlImageTexImage2D(unsigned int id)
{
	IMAGE *image;

	image = store[id];

	glTexImage2D(GL_TEXTURE_2D, 0, Format(image->bpp), image->width, image->height, 0, Format(image->bpp), GL_UNSIGNED_BYTE, image->pixels);
}
