// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "XL/frustum.h"
#include "XL/group.h"
#include <math.h>
#include <GL/gl.h>
#include "id.h"

typedef unsigned int FRUSTUM;

#define STORE ID_SMALL_STORE

static FRUSTUM *store[STORE];

void
xlFrustumCreateContext(void)
{
	idClearStore(FRUSTUM, STORE, store);
}

static
void
Init(FRUSTUM *frustum)
{
	xlGenGroups(1, frustum);
	xlGroupAllocate(*frustum, 24, sizeof(float));
}

static
void
Term(FRUSTUM *frustum)
{
	xlGroupUnload(*frustum);
	xlDeleteGroups(1, frustum);
}

void
xlGenFrustums(unsigned int n, unsigned int *frustums)
{
	idGenObjects(FRUSTUM, STORE, store, Init, n, frustums);
}

void
xlDeleteFrustums(unsigned int n, unsigned int *frustums)
{
	idDeleteObjects(store, Term, n, frustums);
}

void
xlFrustumCalc(unsigned int id)
{
	float projection[16], modelview[16], clip[16], t;
	float *data;
	FRUSTUM *frustum;

	frustum = store[id];

	data = xlGroupArray(*frustum);

	glGetFloatv(GL_PROJECTION_MATRIX, projection);
	glGetFloatv(GL_MODELVIEW_MATRIX, modelview);

	clip[0] = modelview[0] * projection[0] + modelview[1] * projection[4] + modelview[2] * projection[8] + modelview[3] * projection[12];
	clip[1] = modelview[0] * projection[1] + modelview[1] * projection[5] + modelview[2] * projection[9] + modelview[3] * projection[13];
	clip[2] = modelview[0] * projection[2] + modelview[1] * projection[6] + modelview[2] * projection[10] + modelview[3] * projection[14];
	clip[3] = modelview[0] * projection[3] + modelview[1] * projection[7] + modelview[2] * projection[11] + modelview[3] * projection[15];

	clip[4] = modelview[4] * projection[0] + modelview[5] * projection[4] + modelview[6] * projection[8] + modelview[7] * projection[12];
	clip[5] = modelview[4] * projection[1] + modelview[5] * projection[5] + modelview[6] * projection[9] + modelview[7] * projection[13];
	clip[6] = modelview[4] * projection[2] + modelview[5] * projection[6] + modelview[6] * projection[10] + modelview[7] * projection[14];
	clip[7] = modelview[4] * projection[3] + modelview[5] * projection[7] + modelview[6] * projection[11] + modelview[7] * projection[15];

	clip[8] = modelview[8] * projection[0] + modelview[9] * projection[4] + modelview[10] * projection[8] + modelview[11] * projection[12];
	clip[9] = modelview[8] * projection[1] + modelview[9] * projection[5] + modelview[10] * projection[9] + modelview[11] * projection[13];
	clip[10] = modelview[8] * projection[2] + modelview[9] * projection[6] + modelview[10] * projection[10] + modelview[11] * projection[14];
	clip[11] = modelview[8] * projection[3] + modelview[9] * projection[7] + modelview[10] * projection[11] + modelview[11] * projection[15];

	clip[12] = modelview[12] * projection[0] + modelview[13] * projection[4] + modelview[14] * projection[8] + modelview[15] * projection[12];
	clip[13] = modelview[12] * projection[1] + modelview[13] * projection[5] + modelview[14] * projection[9] + modelview[15] * projection[13];
	clip[14] = modelview[12] * projection[2] + modelview[13] * projection[6] + modelview[14] * projection[10] + modelview[15] * projection[14];
	clip[15] = modelview[12] * projection[3] + modelview[13] * projection[7] + modelview[14] * projection[11] + modelview[15] * projection[15];

	data[0] = clip[3] - clip[0];
	data[1] = clip[7] - clip[4];
	data[2] = clip[11] - clip[8];
	data[3] = clip[15] - clip[12];

	t = sqrt(data[0] * data[0] + data[1] * data[1] + data[2] * data[2]);
	data[0] /= t;
	data[1] /= t;
	data[2] /= t;
	data[3] /= t;

	data[4] = clip[3] + clip[0];
	data[5] = clip[7] + clip[4];
	data[6] = clip[11] + clip[8];
	data[7] = clip[15] + clip[12];

	t = sqrt(data[4] * data[4] + data[5] * data[5] + data[6] * data[6] );
	data[4] /= t;
	data[5] /= t;
	data[6] /= t;
	data[7] /= t;

	data[8] = clip[3] + clip[1];
	data[9] = clip[7] + clip[5];
	data[10] = clip[11] + clip[9];
	data[11] = clip[15] + clip[13];

	t = sqrt(data[8] * data[8] + data[9] * data[9] + data[10] * data[10]);
	data[8] /= t;
	data[9] /= t;
	data[10] /= t;
	data[11] /= t;

	data[12] = clip[3] - clip[1];
	data[13] = clip[7] - clip[5];
	data[14] = clip[11] - clip[9];
	data[15] = clip[15] - clip[13];

	t = sqrt(data[12] * data[12] + data[13] * data[13] + data[14] * data[14]);
	data[12] /= t;
	data[13] /= t;
	data[14] /= t;
	data[15] /= t;

	data[16] = clip[3] - clip[2];
	data[17] = clip[7] - clip[6];
	data[18] = clip[11] - clip[10];
	data[19] = clip[15] - clip[14];

	t = sqrt(data[16] * data[16] + data[17] * data[17] + data[18] * data[18]);
	data[16] /= t;
	data[17] /= t;
	data[18] /= t;
	data[19] /= t;

	data[20] = clip[3] + clip[2];
	data[21] = clip[7] + clip[6];
	data[22] = clip[11] + clip[10];
	data[23] = clip[15] + clip[14];

	t = sqrt(data[20] * data[20] + data[21] * data[21] + data[22] * data[22]);
	data[20] /= t;
	data[21] /= t;
	data[22] /= t;
	data[23] /= t;
}

int
xlFrustumPoint(unsigned int id, const float *point)
{
	unsigned int f;
	float *data;
	FRUSTUM *frustum;

	frustum = store[id];

	data = xlGroupArray(*frustum);

	for(f = 0; f < 6; f++)
		if(data[f * 4 + 0] * point[0] + data[f *4 + 1] * point[1] + data[f * 4 + 2] * point[2] + data[f * 4 + 3] <= 0)
			return 0;
	return 1;
}

int
xlFrustumSphere(unsigned int id, const float *sphere)
{
	unsigned int p;
	float *data;
	FRUSTUM *frustum;

	frustum = store[id];

	data = xlGroupArray(*frustum);

	for(p = 0; p < 6; p++)
		if(data[p * 4 + 0] * sphere[0] + data[p *4 + 1] * sphere[1] + data[p *4 + 2] * sphere[2] + data[p * 4 + 3] < -sphere[3])
			return 1;
	return 0;
}

int
xlFrustumTriangle(unsigned int id, const float *triangle)
{
	unsigned int f, p;
	float *data;
	FRUSTUM *frustum;

	frustum = store[id];

	data = xlGroupArray(*frustum);

	for(f = 0; f < 6; f++)
	{
		for(p = 0; p < 3; p++)
			if(data[f * 4 + 0] * triangle[p * 3 + 0] + data[f * 4 + 1] * triangle[p * 3 + 1] + data[f * 4 + 2] * triangle[p * 3 + 2] + data[f * 4 + 3] > 0)
				break;
		if(p == 3) return 0;
	}
	return 1;
}

int
xlFrustumCube(unsigned int id, const float *cube)
{
	unsigned int p;
	float *data;
	FRUSTUM *frustum;

	frustum = store[id];

	data = xlGroupArray(*frustum);

	for(p = 0; p < 6; p++)
	{
		if(data[p * 4 + 0] * (cube[0] - cube[3]) + data[p * 4 + 1] * (cube[1] - cube[3]) + data[p * 4 + 2] * (cube[2] - cube[3]) + data[p * 4 + 3] > 0)
			continue;
		if(data[p * 4 + 0] * (cube[0] + cube[3]) + data[p * 4 + 1] * (cube[1] - cube[3]) + data[p * 4 + 2] * (cube[2] - cube[3]) + data[p * 4 + 3] > 0)
			continue;
		if(data[p * 4 + 0] * (cube[0] - cube[3]) + data[p * 4 + 1] * (cube[1] + cube[3]) + data[p * 4 + 2] * (cube[2] - cube[3]) + data[p * 4 + 3] > 0)
			continue;
		if(data[p * 4 + 0] * (cube[0] + cube[3]) + data[p * 4 + 1] * (cube[1] + cube[3]) + data[p * 4 + 2] * (cube[2] - cube[3]) + data[p * 4 + 3] > 0)
			continue;
		if(data[p * 4 + 0] * (cube[0] - cube[3]) + data[p * 4 + 1] * (cube[1] - cube[3]) + data[p * 4 + 2] * (cube[2] + cube[3]) + data[p * 4 + 3] > 0)
			continue;
		if(data[p * 4 + 0] * (cube[0] + cube[3]) + data[p * 4 + 1] * (cube[1] - cube[3]) + data[p * 4 + 2] * (cube[2] + cube[3]) + data[p * 4 + 3] > 0)
			continue;
		if(data[p * 4 + 0] * (cube[0] - cube[3]) + data[p * 4 + 1] * (cube[1] + cube[3]) + data[p * 4 + 2] * (cube[2] + cube[3]) + data[p * 4 + 3] > 0)
			continue;
		if(data[p * 4 + 0] * (cube[0] + cube[3]) + data[p * 4 + 1] * (cube[1] + cube[3]) + data[p * 4 + 2] * (cube[2] + cube[3]) + data[p * 4 + 3] > 0)
			continue;
		return 0;
	}
	return 1;
}

int
xlFrustumPolygon(unsigned int id, const float *points, unsigned int npoints)
{
	unsigned int f, p;
	float *data;
	FRUSTUM *frustum;

	frustum = store[id];

	data = xlGroupArray(*frustum);

	for(f = 0; f < 6; f++)
	{
		for(p = 0; p < npoints; p++)
			if(data[f * 4 + 0] * points[p * 3] + data[f * 4 + 1] * points[p * 3 + 1] + data[f * 4 + 2] * points[p * 3 + 2] + data[f * 4 + 3] > 0)
				break;
		if(p == npoints) return 0;
	}
	return 1;
}
