// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "mymachines.h"
#include "mymarshal.h"
#include <gtk/gtkgl.h>
#include <gdk/gdkkeysyms.h>
#include <H3D/H3D.h>
#include "MyTrack.h"
#include "MyMachine.h"
#include "MyBox.h"
#include "MyGadget.h"

#define MY_MACHINES_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE((obj), MY_TYPE_MACHINES, MyMachinesPrivate))

struct _MyMachinesPrivate
{
	GdkGLConfig *config;

	guint timeout;

	typedef struct
	{
		MyMachine *machine;
		bool keys[5];
		unsigned int checkpoint;
		float distance;
		unsigned int points;
		int item;
	} Player;

	typedef struct
	{
		MyMachinesGadget type;
		MyBox *box;
		MyGadget *gadget;
		int player;
		int frames;
	} Item;

	MyMachinesState state;

//	FT_Library freetype;
//	FT_Face face;
//	GLint font;
//	GLUtesselator *tess;
	dWorldID world;
	dSpaceID space;
	dJointGroupID joints;
	char *machines[8];
	char *gadgets[3];
	float aspect;

	struct
	{
		bool done;
	} intro;

	struct
	{
		GLuint background;
		ALuint music;
	} main;

	struct
	{
		bool pause;
		float camera[3];
		float frustum[24];
		Player players[8];
		MyTrack *track;
		Item items[20];
	} play;

	struct
	{ 
		MyMachinesLoadState state;
		guint items, machines;
	} load;

	struct
	{
		unsigned int winner;
	} score;

	struct
	{
		char *tracks[6];
		char *machines[3];
		unsigned int nmachines;
		unsigned int nplayers;
		unsigned int nitems;
		unsigned int track;
		unsigned int machine;
		unsigned int npoints;
	} options;

	struct
	{
		std::wstring entries[7];
	} help;
};

G_DEFINE_TYPE(MyMachines, my_machines, GTK_TYPE_DRAWING_AREA);

static
void
my_machines_frame_main(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	GdkGLContext *context = gtk_widget_get_gl_context(GTK_WIDGET(self));
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(GTK_WIDGET(self));

	if(!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glBindTexture(GL_TEXTURE_2D, priv->main.background);
	glPushMatrix();
		glTranslatef((float) rand() / RAND_MAX * 0.01f - 0.005f, (float) rand() / RAND_MAX * 0.01f - 0.005f, 0.0f);
		glBegin(GL_QUADS);
			glTexCoord2i(0, 0); glVertex2f(-0.01f, -0.01f);
			glTexCoord2i(1, 0); glVertex2f(1.01f, -0.01f);
			glTexCoord2i(1, 1); glVertex2f(1.01f, 1.01f);
			glTexCoord2i(0, 1); glVertex2f(-0.01f, 1.01f);
		glEnd();
	glPopMatrix();

	gdk_gl_drawable_swap_buffers(drawable);
		
	gdk_gl_drawable_gl_end(drawable);
}

static
void
my_machines_human(MyMachines *self, unsigned int player)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);

	float position[3], target[3], direction[3], item[3];
	unsigned int i;
	bool turbo = false;

	H3D::Math::Point::Copy(dBodyGetPosition(priv->play.players[player].machine->body[0]), position);
	H3D::Math::Point::Copy(&priv->play.track->content.checkpoints[(priv->play.players[player].checkpoint + 1) % (priv->play.track->info.ncheckpoints) * 3], target);
	H3D::Math::Vector::Sub(target, position, direction);
	H3D::Math::Vector::Norm(direction, &priv->play.players[player].distance);
	if(priv->play.players[player].distance < 40.0f)
	{
		priv->play.players[player].checkpoint++;
		H3D::Math::Point::Copy(&priv->play.track->content.checkpoints[(priv->play.players[player].checkpoint + 1) % (priv->play.track->info.ncheckpoints) * 3], target);
		H3D::Math::Vector::Sub(target, position, direction);
		H3D::Math::Vector::Norm(direction, &priv->play.players[player].distance);
	}

	if(priv->play.players[player].item == -1)
	{
		for(i = 0; i < priv->options.nitems; i++)
		{
			if(priv->play.items[i].frames == 0)
			{
				H3D::Math::Point::Sub(position, priv->play.items[i].box->sphere, item);
				if(H3D::Math::Vector::NormFunc(item) <= priv->play.items[i].box->sphere[3])
				{
					priv->play.players[player].item = i;
					priv->play.items[i].player = player;
					switch(priv->play.items[i].type)
					{
						case MY_MACHINES_ITEM_CUBE: priv->play.items[i].frames = 800; break;
						case MY_MACHINES_ITEM_TURBO: priv->play.items[i].frames = 100; break;
						case MY_MACHINES_ITEM_FAKE: priv->play.items[i].frames = 200; break;
					}
					g_signal_emit_by_name(self, "my-gadget-captured", player, priv->play.items[priv->play.players[player].item].type);
				}
			}
		}
	}
	else
	{
		if(priv->play.players[player].keys[4])
		{
			switch(priv->play.items[priv->play.players[player].item].type)
			{
				case MY_MACHINES_ITEM_CUBE:
				{
					const float *r;
					r = dBodyGetRotation(priv->play.players[player].machine->body[0]);
					direction[0] = -r[2]; direction[1] = -r[6]; direction[2] = -r[10];
					H3D::Math::Vector::ScalarProductInPlace(direction, 3.0f);
					H3D::Math::Point::Sub(position, direction, item);
					priv->play.items[priv->play.players[player].item].gadget->place(item);
					priv->play.items[priv->play.players[player].item].gadget->enable();
					priv->play.items[priv->play.players[player].item].player = -1;
					g_signal_emit_by_name(self, "my-gadget-released", player, priv->play.items[priv->play.players[player].item].type);
					priv->play.players[player].item = -1;
				}
					break;
				case MY_MACHINES_ITEM_TURBO:
					turbo = true;
					break;
				case MY_MACHINES_ITEM_FAKE:
					break;
			}
		}
	}

	priv->play.players[player].machine->drive(priv->play.players[player].keys[0], priv->play.players[player].keys[1], priv->play.players[player].keys[2], priv->play.players[player].keys[3], turbo);
}

static
void
my_machines_computer(MyMachines *self, unsigned int player)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);

	float position[3], target[3], direction[3], orientation[3], item[3], plane[4], angle, variation, distance;
	const float *r;
	int steer, gas;
	bool action, turbo = false;
	unsigned int i;

	variation = (float) rand() / RAND_MAX;

	H3D::Math::Point::Copy(dBodyGetPosition(priv->play.players[player].machine->body[0]), position);
	H3D::Math::Point::Copy(&priv->play.track->content.checkpoints[(priv->play.players[player].checkpoint + 1) % (priv->play.track->info.ncheckpoints) * 3], target);
	H3D::Math::Vector::Sub(target, position, direction);
	H3D::Math::Vector::Norm(direction, &priv->play.players[player].distance);
	if(priv->play.players[player].distance < 20.0f)
	{
		priv->play.players[player].checkpoint++;
		H3D::Math::Point::Copy(&priv->play.track->content.checkpoints[(priv->play.players[player].checkpoint + 1) % (priv->play.track->info.ncheckpoints) * 3], target);
		H3D::Math::Vector::Sub(target, position, direction);
		H3D::Math::Vector::Norm(direction, &priv->play.players[player].distance);
	}
	H3D::Math::Vector::NormalInPlace(direction);
	r = dBodyGetRotation(priv->play.players[player].machine->body[0]);
	orientation[0] = -r[2]; orientation[1] = -r[6]; orientation[2] = -r[10];
	H3D::Math::Vector::CrossProduct(H3D::Math::Vector::Y, orientation, plane);
	plane[3] = -H3D::Math::Vector::DotProductFunc(plane, position);
	distance = H3D::Math::Vector::DotProductFunc(plane, target) + plane[3];
	H3D::Math::Vector::Angle(orientation, direction, &angle);
	if(distance > 10.0f * (0.25f + variation * 0.75f)) steer = 1;
	else if(distance < -10.0f * (0.25 + variation * 0.75f)) steer = -1;
	else steer = 0;
	if(angle > 75.0f)
	{
		const float *v;
		v = dBodyGetLinearVel(priv->play.players[player].machine->body[0]);
		if(priv->play.players[player].distance < 20.0f * 2.0f)
		{
			if(H3D::Math::Vector::NormFunc(v) > 2.0f) gas = 0;
			else gas = 1;
		}
		else { gas = -1; steer *= -1; }
	}
	else gas = 1;
	action = variation < 0.05;

	if(priv->play.players[player].item == -1)
	{
		for(i = 0; i < priv->options.nitems; i++)
		{
			if(priv->play.items[i].frames == 0)
			{
				H3D::Math::Point::Sub(position, priv->play.items[i].box->sphere, item);
				if(H3D::Math::Vector::NormFunc(item) <= priv->play.items[i].box->sphere[3])
				{
					priv->play.players[player].item = i;
					priv->play.items[i].player = player;
					switch(priv->play.items[i].type)
					{
						case MY_MACHINES_ITEM_CUBE: priv->play.items[i].frames = 800; break;
						case MY_MACHINES_ITEM_TURBO: priv->play.items[i].frames = 100; break;
						case MY_MACHINES_ITEM_FAKE: priv->play.items[i].frames = 200; break;
					}
					g_signal_emit_by_name(self, "my-gadget-captured", player, priv->play.items[priv->play.players[player].item].type);
				}
			}
		}
	}
	else
	{
		if(action)
		{
			switch(priv->play.items[priv->play.players[player].item].type)
			{
				case MY_MACHINES_ITEM_CUBE:
				{
					H3D::Math::Vector::ScalarProductInPlace(orientation, 3.0f);
					H3D::Math::Point::Sub(position, orientation, item);
					priv->play.items[priv->play.players[player].item].gadget->place(item);
					priv->play.items[priv->play.players[player].item].gadget->enable();
					g_signal_emit_by_name(self, "my-gadget-released", player, priv->play.items[priv->play.players[player].item].type);
					priv->play.items[priv->play.players[player].item].player = -1;
					priv->play.players[player].item = -1;
				}
					break;
				case MY_MACHINES_ITEM_TURBO:
					turbo = true;
					break;
				case MY_MACHINES_ITEM_FAKE:
					break;
			}
		}
	}

	priv->play.players[player].machine->drive(gas == 1, gas == -1, steer == 1, steer == -1, turbo);
}

static
void
on_near(void *data, dGeomID o1, dGeomID o2)
{
	MyMachines *self = MY_MACHINES(data);
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);

	const int ncontacts = 1024;
	unsigned int i;
	int n;
	dBodyID b1, b2;
	dContact contacts[ncontacts];
	dJointID c;
	b1 = dGeomGetBody(o1);
	b2 = dGeomGetBody(o2);
	if(b1 && b2 && dAreConnectedExcluding(b1, b2, dJointTypeContact))
		return;
	n = dCollide(o1, o2, ncontacts, &contacts[0].geom, sizeof(dContact));
	if(n > 0)
	{
		for(i = 0; i < (unsigned int) n; i++)
		{
			contacts[i].surface.mode = dContactSlip1 | dContactSlip2 | dContactSoftERP | dContactSoftCFM | dContactApprox1 | dContactMu2;
			contacts[i].surface.mu = 0.75f;
			contacts[i].surface.mu2 = 0.75f;
			contacts[i].surface.slip1 = 0.01f;
			contacts[i].surface.slip2 = 0.01f;
			contacts[i].surface.soft_erp = 0.5f;
			contacts[i].surface.soft_cfm = 0.3f;
			c = dJointCreateContact(priv->world, priv->joints, &contacts[i]);
			dJointAttach(c, dGeomGetBody(contacts[i].geom.g1), dGeomGetBody(contacts[i].geom.g2));
		}
	}
}

static
void
my_machines_frame_play(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	GdkGLContext *context = gtk_widget_get_gl_context(GTK_WIDGET(self));
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(GTK_WIDGET(self));

	if(!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();

	unsigned int i, first, actives;
	const float *v;

	for(actives = 0, i = 0; i < priv->options.nmachines; i++)
		if(dBodyIsEnabled(priv->play.players[i].machine->body[0]))
		{
			actives++;
			if(i < priv->options.nplayers) my_machines_human(self, i);
			else my_machines_computer(self, i);
		}

	for(i = 0; i < 16; i++)
	{
		dSpaceCollide(priv->space, self, on_near);
		dWorldStep(priv->world, 0.01f);
		dJointGroupEmpty(priv->joints);
	}

	for(first = 0, i = 0; i < priv->options.nmachines; i++)
	{
		if(dBodyIsEnabled(priv->play.players[i].machine->body[0]) && (priv->play.players[i].checkpoint > priv->play.players[first].checkpoint || (priv->play.players[i].checkpoint == priv->play.players[first].checkpoint && priv->play.players[i].distance < priv->play.players[first].distance)))
			first = i;
	}

	if(actives > 1)
	{
		const float *p;
		float a[3], b[3];
		p = dBodyGetPosition(priv->play.players[first].machine->body[0]);
		H3D::Math::Vector::Sub(p, priv->play.camera, a);
		H3D::Math::Vector::ScalarProduct(a, 0.15f, b);
		H3D::Math::Vector::AddInPlace(priv->play.camera, b);
	}
	else
	{
		float position[3], rotation[16], target[3], forward[3];
		priv->play.players[first].points++;
		g_signal_emit_by_name(self, "my-win-point", first);
		H3D::Math::Point::Copy(&priv->play.track->content.checkpoints[priv->play.players[first].checkpoint % (priv->play.track->info.ncheckpoints)* 3], priv->play.camera);
		H3D::Math::Point::Copy(&priv->play.track->content.checkpoints[(priv->play.players[first].checkpoint + 1) % (priv->play.track->info.ncheckpoints) * 3], target);
		H3D::Math::Vector::Sub(target, priv->play.camera, forward);
		H3D::Math::Vector::NormalInPlace(forward);
		H3D::Math::Matrix::Rotation(forward, H3D::Math::Vector::Y, rotation);
		for(i = 0; i < priv->options.nmachines; i++)
		{
			if(!dBodyIsEnabled(priv->play.players[i].machine->body[0]))
				priv->play.players[i].machine->enable();
			priv->play.players[i].checkpoint = priv->play.players[first].checkpoint;
			H3D::Math::Point::Load(position, ((i % 4) < 2 ? -5.0f : 5.0f) + ((i & 1) ? 2.5f : -2.5f), 5.0f, float(i / 4) * 7.5f);
			H3D::Math::Vector::MatrixProductInPlace(position, rotation);
			H3D::Math::Point::AddInPlace(position, priv->play.camera);
			priv->play.players[i].machine->place(position, rotation);
		}
	}

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glLoadIdentity();

	gluLookAt(priv->play.camera[0], 70.0f, priv->play.camera[2], priv->play.camera[0], priv->play.camera[1], priv->play.camera[2], 0.0f, 0.0f, -1.0f);
	H3D::Helper::OpenGL::Frustum::Load(priv->play.frustum);

	priv->play.track->frame();

	for(i = 0; i < priv->options.nmachines; i++)
	{
		if(dBodyIsEnabled(priv->play.players[i].machine->body[0]))
		{
			if(H3D::Math::Frustum::Point(priv->play.frustum, dBodyGetPosition(priv->play.players[i].machine->body[0])))
				priv->play.players[i].machine->frame();
			else
			{
				if(priv->play.players[i].item != -1)
				{
					unsigned int checkpoint = ((float) rand() / RAND_MAX) * priv->play.track->info.ncheckpoints;
					H3D::Math::Point::Copy(&priv->play.track->content.checkpoints[checkpoint * 3], priv->play.items[priv->play.players[i].item].box->sphere);
					g_signal_emit_by_name(self, "my-gadget-released", i, priv->play.items[priv->play.players[i].item].type);
					priv->play.items[priv->play.players[i].item].player = -1;
					priv->play.players[i].item = -1;
				}
				priv->play.players[i].machine->disable();
				g_signal_emit_by_name(self, "my-out-point", i);
				if(actives == priv->options.nmachines && priv->play.players[i].points)
				{
					priv->play.players[i].points--;
					g_signal_emit_by_name(self, "my-loose-point", i);
				}
			}
		}
	}

	for(i = 0; i < priv->options.nitems; i++)
	{
		if(priv->play.items[i].frames > 0)
		{
			if(priv->play.items[i].player == -1)
				priv->play.items[i].gadget->frame();
			priv->play.items[i].frames--;
			if(priv->play.items[i].frames == 0)
			{
				if(priv->play.items[i].player != -1)
				{
					g_signal_emit_by_name(self, "my-gadget-released", priv->play.items[i].player, priv->play.items[i].type);
					priv->play.players[priv->play.items[i].player].item = -1;
					priv->play.items[i].player = -1;
				}
				else
					priv->play.items[i].gadget->disable();
				unsigned int checkpoint = ((float) rand() / RAND_MAX) * priv->play.track->info.ncheckpoints;
				H3D::Math::Point::Copy(&priv->play.track->content.checkpoints[checkpoint * 3], priv->play.items[i].box->sphere);
			}
		}
		else
			priv->play.items[i].box->frame();
	}

	alListener3f(AL_POSITION, priv->play.camera[0], 0, priv->play.camera[2]);
//	v = dBodyGetLinearVel(priv->play.players[first].machine->body[0]);
//	alListener3f(AL_VELOCITY, v[0], v[1], v[2]);

	gdk_gl_drawable_swap_buffers(drawable);
		
	gdk_gl_drawable_gl_end(drawable);

	if(priv->play.players[first].points == priv->options.npoints)
	{
		//priv->score.winner = first;
		my_machines_set_state(self, MY_MACHINES_STATE_MAIN);
		g_signal_emit_by_name(self, "my-win", first);
	}
}

static
void
my_machines_frame_pause(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	GdkGLContext *context = gtk_widget_get_gl_context(GTK_WIDGET(self));
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(GTK_WIDGET(self));

	if(!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glBindTexture(GL_TEXTURE_2D, priv->main.background);
	glPushMatrix();
		glTranslatef((float) rand() / RAND_MAX * 0.01f - 0.005f, (float) rand() / RAND_MAX * 0.01f - 0.005f, 0.0f);
		glBegin(GL_QUADS);
			glTexCoord2i(0, 0); glVertex2f(-0.01f, -0.01f);
			glTexCoord2i(1, 0); glVertex2f(1.01f, -0.01f);
			glTexCoord2i(1, 1); glVertex2f(1.01f, 1.01f);
			glTexCoord2i(0, 1); glVertex2f(-0.01f, 1.01f);
		glEnd();
	glPopMatrix();

	gdk_gl_drawable_swap_buffers(drawable);
		
	gdk_gl_drawable_gl_end(drawable);
}

static
void
my_machines_frame(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);

	switch(priv->state)
	{
		case MY_MACHINES_STATE_MAIN: my_machines_frame_main(self); break;
		case MY_MACHINES_STATE_PLAY: my_machines_frame_play(self); break;
		case MY_MACHINES_STATE_PAUSE: my_machines_frame_pause(self); break;
	}
}

static
void
my_machines_enter_main(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	GdkGLContext *context = gtk_widget_get_gl_context(GTK_WIDGET(self));
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(GTK_WIDGET(self));

	if(!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();

	alSourcePlay(priv->main.music);

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluOrtho2D(0.0f, 1.0f, 0.0f, 1.0f);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	gdk_gl_drawable_gl_end(drawable);

	priv->state = MY_MACHINES_STATE_MAIN;
}

static
void
my_machines_leave_main(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	alSourceStop(priv->main.music);
}

static
void
my_machines_enter_play(MyMachines *self)
{
	my_machines_load(self);
}

static
void
my_machines_leave_play(MyMachines *self)
{
	my_machines_unload(self);
}

static
void
my_machines_enter_pause(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	GdkGLContext *context = gtk_widget_get_gl_context(GTK_WIDGET(self));
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(GTK_WIDGET(self));
	unsigned int i;

	if(!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();

	for(i = 0; i < priv->options.nmachines; i++)
		if(dBodyIsEnabled(priv->play.players[i].machine->body[0]))
			alSourcePause(priv->play.players[i].machine->engine);

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluOrtho2D(0.0f, 1.0f, 0.0f, 1.0f);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	gdk_gl_drawable_gl_end(drawable);

	priv->state = MY_MACHINES_STATE_PAUSE;
}

static
void
my_machines_leave_pause(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	GdkGLContext *context = gtk_widget_get_gl_context(GTK_WIDGET(self));
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(GTK_WIDGET(self));
	unsigned int i;

	if(!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();
	for(i = 0; i < priv->options.nmachines; i++)
		if(dBodyIsEnabled(priv->play.players[i].machine->body[0]))
			alSourcePlay(priv->play.players[i].machine->engine);

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(45.0f, priv->aspect, 1.0f, 250.0f);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	gdk_gl_drawable_gl_end(drawable);

	priv->state = MY_MACHINES_STATE_PLAY;
}

static
gboolean
on_load(gpointer data)
{
	MyMachines *self = MY_MACHINES(data);
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	GdkGLContext *context = gtk_widget_get_gl_context(GTK_WIDGET(self));
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(GTK_WIDGET(self));

	if (!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();

	switch(priv->load.state)
	{
		case MY_MACHINES_LOAD_BEGIN:
			priv->load.state = MY_MACHINES_LOAD_TRACK;
			break;
		case MY_MACHINES_LOAD_TRACK:
			priv->play.track = new MyTrack();
			priv->play.track->load(priv->options.tracks[priv->options.track], priv->space);
			priv->load.state = MY_MACHINES_LOAD_ITEMS;
			break;
		case MY_MACHINES_LOAD_ITEMS:
		{
			float sphere[4];
			unsigned int checkpoint;
//			for(i = 0; i < priv->options.nitems; i++)
//			{
				priv->play.items[priv->load.items].box = new MyBox();
				checkpoint = ((float) rand() / RAND_MAX) * priv->play.track->info.ncheckpoints;
				priv->play.items[priv->load.items].type = (MyMachinesGadget) (((float) rand() / RAND_MAX) * (sizeof(priv->gadgets) / sizeof(priv->gadgets[0])));
				H3D::Math::Point::Copy(&priv->play.track->content.checkpoints[checkpoint * 3], sphere);
				sphere[3] = 2.0f;
				priv->play.items[priv->load.items].box->load(sphere);
				switch(priv->play.items[priv->load.items].type)
				{
					case MY_MACHINES_ITEM_CUBE: priv->play.items[priv->load.items].gadget = new MyCube(); break;
					case MY_MACHINES_ITEM_TURBO: priv->play.items[priv->load.items].gadget = new MyTurbo(); break;
					case MY_MACHINES_ITEM_FAKE: priv->play.items[priv->load.items].gadget = new MyFake(); break;
				}
				priv->play.items[priv->load.items].gadget->load(priv->world, priv->space);
				priv->play.items[priv->load.items].gadget->disable();
				priv->play.items[priv->load.items].player = -1;
				priv->play.items[priv->load.items].frames = 0;
//			}
			priv->load.items++;
			if(priv->load.items == priv->options.nitems) priv->load.state = MY_MACHINES_LOAD_MACHINES;
		}
			break;
		case MY_MACHINES_LOAD_MACHINES:
		{
			float position[3];
//			unsigned int i;
//			for(i = 0; i < priv->options.nmachines; i++)
//			{
			memset(priv->play.players[priv->load.machines].keys, 0, 5 * sizeof(bool));
			priv->play.players[priv->load.machines].checkpoint = 0;
			priv->play.players[priv->load.machines].distance = 0.0f;
			priv->play.players[priv->load.machines].points = 0;
			priv->play.players[priv->load.machines].item = -1;
			priv->play.players[priv->load.machines].machine = new MyMachine();
			H3D::Math::Point::Load(position, ((priv->load.machines % 4) < 2 ? -5.0f : 5.0f) + ((priv->load.machines & 1) ? 2.5f : -2.5f), 5.0f, float(priv->load.machines / 4) * 7.5f);
			gchar *machine = g_strdup_printf("%s %s", priv->options.machines[priv->options.machine], priv->machines[priv->load.machines]);
			priv->play.players[priv->load.machines].machine->load(machine, priv->world, priv->space, position);
			g_free(machine);
			priv->load.machines++;
//			}
			if(priv->load.machines == priv->options.nmachines) priv->load.state = MY_MACHINES_LOAD_SCENE;
		}
			break;
		case MY_MACHINES_LOAD_SCENE:
		{
			float listener[6] = {0.0f, -1.0f, 0.0f, 0.0f, 0.0f, -1.0f};
			H3D::Math::Point::LoadZero(priv->play.camera);
			alListener3f(AL_POSITION, priv->play.camera[0], 0.0f, priv->play.camera[2]);
			alListenerfv(AL_ORIENTATION, listener);

			glMatrixMode(GL_PROJECTION);
			glLoadIdentity();
			gluPerspective(45.0f, priv->aspect, 1.0f, 250.0f);
			glMatrixMode(GL_MODELVIEW);
			glLoadIdentity();

			priv->load.state = MY_MACHINES_LOAD_END;
			priv->state = MY_MACHINES_STATE_PLAY;
		}
			break;
		case MY_MACHINES_LOAD_END:
			g_assert_not_reached();
			break;
	}
	gdk_gl_drawable_gl_end(drawable);
	g_signal_emit_by_name(self, "my-load", priv->load.state);

	return priv->load.state != MY_MACHINES_LOAD_END;
}

static
gboolean
on_frame(gpointer data)
{
	MyMachines *self = (MyMachines *) data;

	my_machines_frame(self);

	return TRUE;
}

static
gboolean
on_realize(GtkWidget *widget, gpointer data)
{
	MyMachines *self = MY_MACHINES(widget);
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);

	ALuint buffer;

	GdkGLContext *context = gtk_widget_get_gl_context(widget);
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(widget);

	if (!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();

	srand(H3D::System::Ticks()); // TOFIX

	glEnable(GL_COLOR_MATERIAL);
	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LEQUAL);
	glEnable(GL_TEXTURE_2D);
	glClearColor(0, 0, 0.5, 0);
//	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
//	glEnable(GL_BLEND);

/*
	FT_Init_FreeType(&priv->freetype);
	FT_New_Face(priv->freetype, PACKAGE_DATADIR STRPS "VeraBd.ttf", 0, &priv->face);
	FT_Set_Pixel_Sizes(priv->face, 64, 64);

	priv->tess = gluNewTess();
	gluTessCallback(priv->tess, GLU_TESS_VERTEX, (void (*)())&glVertex3dv);
	gluTessCallback(priv->tess, GLU_TESS_BEGIN, (void (*)())&glBegin);
	gluTessCallback(priv->tess, GLU_TESS_END, (void (*)())&glEnd);
*/
	glGenTextures(1, &priv->main.background);
	glBindTexture(GL_TEXTURE_2D, priv->main.background);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);

/*
	GdkPixbuf *image = gdk_pixbuf_new_from_file(PACKAGE_DATADIR STRPS "textures" STRPS "Analog-Wind-1.jpg", NULL);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, gdk_pixbuf_get_width(image), gdk_pixbuf_get_height(image), 0, GL_RGB, GL_UNSIGNED_BYTE, gdk_pixbuf_get_pixels(image));
	g_object_unref(G_OBJECT(image));
*/
	H3D::Image::H3D *image = H3D::Image::JPEG::Load(PACKAGE_DATADIR STRPS "textures" STRPS "Analog-Wind-1.jpg");
	if(image)
	{
		glTexImage2D(GL_TEXTURE_2D, 0, H3D::Helper::OpenGL::Image::Format(image->info.bpp), image->info.width, image->info.height, 0, H3D::Helper::OpenGL::Image::Format(image->info.bpp), GL_UNSIGNED_BYTE, image->content.pixels);
		image->unload();
		delete image;
	}

	
	priv->world = dWorldCreate();
	priv->space = dHashSpaceCreate(0);
	priv->joints = dJointGroupCreate(0);
	dWorldSetGravity(priv->world, 0.0f, -9.8f, 0.0f);

//	alDistanceModel(AL_NONE);
	alGenSources(1, &priv->main.music);
	alGenBuffers(1, &buffer);
	H3D::Sound::H3D* sound = H3D::Sound::VORBIS::Load(PACKAGE_DATADIR STRPS "sounds" STRPS "engine.ogg");
	alBufferData(buffer, H3D::Helper::OpenAL::Sound::Format(sound->info.nchannels), sound->content.samples, sound->info.nframes * sound->info.nchannels * sizeof(short), sound->info.rate);
	sound->unload();
	delete sound;

	alSourcei(priv->main.music, AL_BUFFER, buffer);
	alSourcei(priv->main.music, AL_LOOPING, AL_TRUE);
	alSourcei(priv->main.music, AL_SOURCE_RELATIVE, AL_TRUE);

	gdk_gl_drawable_gl_end(drawable);

	my_machines_enter_main(self);
	g_signal_emit_by_name(self, "my-state-changed", MY_MACHINES_STATE_MAIN);

	priv->timeout = g_timeout_add(40, on_frame, MY_MACHINES(widget));

	return TRUE;
}

static
void
on_unrealize(GtkWidget *widget)
{
	MyMachines *self = MY_MACHINES(widget);
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	GdkGLContext *context = gtk_widget_get_gl_context(widget);
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(widget);
	ALuint buffer;

	g_source_remove(priv->timeout);

	if(my_machines_is_loaded(self)) my_machines_unload(self);

	if (!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();

	dJointGroupDestroy(priv->joints);
	dSpaceDestroy(priv->space);
	dWorldDestroy(priv->world);

//	gluDeleteTess(priv->tess);
	glDeleteTextures(1, &priv->main.background);

//	FT_Done_Face(priv->face);
//	FT_Done_FreeType(priv->freetype);

	alGetSourcei(priv->main.music, AL_BUFFER, (ALint*)&buffer);
	alDeleteSources(1, &priv->main.music);
	alDeleteBuffers(1, &buffer);

	gdk_gl_drawable_gl_end(drawable);
}

static
gboolean
my_machines_configure_event_main(MyMachines *self, GdkEventConfigure *event)
{
	GdkGLContext *context = gtk_widget_get_gl_context(GTK_WIDGET(self));
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(GTK_WIDGET(self));

	if (!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluOrtho2D(0.0f, 1.0f, 0.0f, 1.0f);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	
	gdk_gl_drawable_gl_end(drawable);

	return TRUE;
}

static
gboolean
my_machines_configure_event_play(MyMachines *self, GdkEventConfigure *event)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	GdkGLContext *context = gtk_widget_get_gl_context(GTK_WIDGET(self));
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(GTK_WIDGET(self));

	if (!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(45.0f, priv->aspect, 1.0f, 250.0f);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	
	gdk_gl_drawable_gl_end(drawable);

	return TRUE;
}

static
gboolean
my_machines_configure_event_pause(MyMachines *self, GdkEventConfigure *event)
{
	GdkGLContext *context = gtk_widget_get_gl_context(GTK_WIDGET(self));
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(GTK_WIDGET(self));

	if (!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluOrtho2D(0.0f, 1.0f, 0.0f, 1.0f);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	
	gdk_gl_drawable_gl_end(drawable);

	return TRUE;
}

static
gboolean
my_machines_configure_event(MyMachines *self, GdkEventConfigure *event)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	gboolean result;

	switch(priv->state)
	{
		case MY_MACHINES_STATE_MAIN: result = my_machines_configure_event_main(self, event);; break;
		case MY_MACHINES_STATE_PLAY: result = my_machines_configure_event_play(self, event); break;
		case MY_MACHINES_STATE_PAUSE: result = my_machines_configure_event_pause(self, event); break;
		default: result = FALSE; break;
	}
	return result;
}

static
gboolean
on_configure_event(GtkWidget *widget, GdkEventConfigure *event, gpointer data)
{
	MyMachines *self = (MyMachines *) data;
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	GdkGLContext *context = gtk_widget_get_gl_context(widget);
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(widget);

	priv->aspect = (float) widget->allocation.width / widget->allocation.height;

	if (!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();

	glViewport(0, 0, widget->allocation.width, widget->allocation.height);

	gdk_gl_drawable_gl_end(drawable);

	return my_machines_configure_event(self, event);
}

static
gboolean
on_expose_event(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
	if(event->count == 0) my_machines_frame(MY_MACHINES(widget));
	return TRUE;
}

static
gboolean
my_machines_key_press_event_play(MyMachines *self, GdkEventKey *event)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	switch(event->keyval)
	{
		case GDK_Up: priv->play.players[0].keys[0] = TRUE; break;
		case GDK_Down: priv->play.players[0].keys[1] = TRUE; break;
		case GDK_Left: priv->play.players[0].keys[2] = TRUE; break;
		case GDK_Right: priv->play.players[0].keys[3] = TRUE; break;
		case GDK_Return: priv->play.players[0].keys[4] = TRUE; break;
		case GDK_w: priv->play.players[1].keys[0] = TRUE; break;
		case GDK_s: priv->play.players[1].keys[1] = TRUE; break;
		case GDK_a: priv->play.players[1].keys[2] = TRUE; break;
		case GDK_d: priv->play.players[1].keys[3] = TRUE; break;
		case GDK_q: priv->play.players[1].keys[4] = TRUE; break;
		case GDK_i: priv->play.players[2].keys[0] = TRUE; break;
		case GDK_k: priv->play.players[2].keys[1] = TRUE; break;
		case GDK_j: priv->play.players[2].keys[2] = TRUE; break;
		case GDK_l: priv->play.players[2].keys[3] = TRUE; break;
		case GDK_u: priv->play.players[2].keys[4] = TRUE; break;
		case GDK_t: priv->play.players[3].keys[0] = TRUE; break;
		case GDK_g: priv->play.players[3].keys[1] = TRUE; break;
		case GDK_f: priv->play.players[3].keys[2] = TRUE; break;
		case GDK_h: priv->play.players[3].keys[3] = TRUE; break;
		case GDK_r: priv->play.players[3].keys[4] = TRUE; break;
	}
	return TRUE;
}
static
gboolean
my_machines_key_release_event_play(MyMachines *self, GdkEventKey *event)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);

	switch(event->keyval)
	{
		case GDK_Escape: my_machines_set_state(self, MY_MACHINES_STATE_MAIN); break;
		case GDK_Pause: my_machines_set_state(self, MY_MACHINES_STATE_PAUSE); break;
		case GDK_Up: priv->play.players[0].keys[0] = FALSE; break;
		case GDK_Down: priv->play.players[0].keys[1] = FALSE; break;
		case GDK_Left: priv->play.players[0].keys[2] = FALSE; break;
		case GDK_Right: priv->play.players[0].keys[3] = FALSE; break;
		case GDK_Return: priv->play.players[0].keys[4] = FALSE; break;
		case GDK_w: priv->play.players[1].keys[0] = FALSE; break;
		case GDK_s: priv->play.players[1].keys[1] = FALSE; break;
		case GDK_a: priv->play.players[1].keys[2] = FALSE; break;
		case GDK_d: priv->play.players[1].keys[3] = FALSE; break;
		case GDK_q: priv->play.players[1].keys[4] = FALSE; break;
		case GDK_i: priv->play.players[2].keys[0] = FALSE; break;
		case GDK_k: priv->play.players[2].keys[1] = FALSE; break;
		case GDK_j: priv->play.players[2].keys[2] = FALSE; break;
		case GDK_l: priv->play.players[2].keys[3] = FALSE; break;
		case GDK_u: priv->play.players[2].keys[4] = FALSE; break;
		case GDK_t: priv->play.players[3].keys[0] = FALSE; break;
		case GDK_g: priv->play.players[3].keys[1] = FALSE; break;
		case GDK_f: priv->play.players[3].keys[2] = FALSE; break;
		case GDK_h: priv->play.players[3].keys[3] = FALSE; break;
		case GDK_r: priv->play.players[3].keys[4] = FALSE; break;
	}
	return TRUE;
}

static
gboolean
my_machines_key_press_event_pause(MyMachines *self, GdkEventKey *event)
{
	return TRUE;
}

static
gboolean
my_machines_key_release_event_pause(MyMachines *self, GdkEventKey *event)
{
	switch(event->keyval)
	{
		case GDK_Escape: my_machines_set_state(self, MY_MACHINES_STATE_PLAY); break;
	}
	return TRUE;
}

static
gboolean
my_machines_key_press_event(MyMachines *self, GdkEventKey *event)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	gboolean result;

	switch(priv->state)
	{
		case MY_MACHINES_STATE_MAIN: result = FALSE;; break;
		case MY_MACHINES_STATE_PLAY: result = my_machines_key_press_event_play(self, event); break;
		case MY_MACHINES_STATE_PAUSE: result = my_machines_key_press_event_pause(self, event); break;
		default: result = FALSE; break;
	}
	return result;
}

static
gboolean
my_machines_key_release_event(MyMachines *self, GdkEventKey *event)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	gboolean result;

	switch(priv->state)
	{
		case MY_MACHINES_STATE_MAIN: result = FALSE;; break;
		case MY_MACHINES_STATE_PLAY: result = my_machines_key_release_event_play(self, event); break;
		case MY_MACHINES_STATE_PAUSE: result = my_machines_key_release_event_pause(self, event); break;
		default: result = FALSE; break;
	}
	return result;
}

static
gboolean
on_key_press_event(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	MyMachines *self = (MyMachines *) data;
	return my_machines_key_press_event(self, event);
}

static
gboolean
on_key_release_event(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	MyMachines *self = (MyMachines *) data;
	return my_machines_key_release_event(self, event);
}

static
void
my_machines_init(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);

	priv->config = gdk_gl_config_new_by_mode((GdkGLConfigMode) (GDK_GL_MODE_RGBA | GDK_GL_MODE_DEPTH | GDK_GL_MODE_DOUBLE));
	if(!priv->config) g_assert_not_reached();
	if(!gtk_widget_set_gl_capability(GTK_WIDGET(self), priv->config, NULL, TRUE, GDK_GL_RGBA_TYPE)) g_assert_not_reached();

	priv->timeout = 0;
	priv->machines[0] = "red";
	priv->machines[1] = "green";
	priv->machines[2] = "blue";
	priv->machines[3] = "yellow";
	priv->machines[4] = "cyan";
	priv->machines[5] = "violet";
	priv->machines[6] = "white";
	priv->machines[7] = "black";
	priv->gadgets[0] = "cube";
	priv->gadgets[1] = "turbo";
	priv->gadgets[2] = "fake";
	priv->options.tracks[0] = "crayons";
	priv->options.tracks[1] = "garden";
	priv->options.tracks[2] = "land";
	priv->options.tracks[3] = "bathtub";
	priv->options.tracks[4] = "square";
	priv->options.tracks[5] = "circle";
	priv->options.machines[0] = "car";
	priv->options.machines[1] = "formula 1";
	priv->options.machines[2] = "block";
	priv->options.nmachines = 4;
	priv->options.nplayers = 1;
	priv->options.nitems = 8;
	priv->options.track = 0;
	priv->options.machine = 0;
	priv->options.npoints = 3;
//	priv->edit = FALSE; //argc == 2 && !strcmp(argv[1], "-e");

	GTK_WIDGET_SET_FLAGS(GTK_WIDGET(self), GTK_CAN_FOCUS);
	gtk_widget_set_events(GTK_WIDGET(self), GDK_EXPOSURE_MASK | GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK);
	g_signal_connect(self, "realize", G_CALLBACK(on_realize), self);
	g_signal_connect(self, "unrealize", G_CALLBACK(on_unrealize), self);
	g_signal_connect(self, "configure-event", G_CALLBACK(on_configure_event), self);
	g_signal_connect(self, "key-press-event", G_CALLBACK(on_key_press_event), self);
	g_signal_connect(self, "key-release-event", G_CALLBACK(on_key_release_event), self);
//	g_signal_connect(self, "expose-event", G_CALLBACK(on_expose_event), self);
}

static
void
my_machines_finalize(GObject *object)
{
	G_OBJECT_CLASS(my_machines_parent_class)->finalize(object);
}

static
void
my_machines_dispose(GObject *object)
{
	G_OBJECT_CLASS(my_machines_parent_class)->dispose(object);
}
/*
static
void
my_machines_show(GtkWidget *widget)
{
	GTK_WIDGET_CLASS(my_machines_parent_class)->show(widget);

//	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(MY_MACHINES(widget));
//	priv->timeout = g_timeout_add(40, on_frame, MY_MACHINES(widget));
}

static
void
my_machines_hide(GtkWidget *widget)
{
//	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(MY_MACHINES(widget));
//	g_source_remove(priv->timeout);

	GTK_WIDGET_CLASS(my_machines_parent_class)->hide(widget);
}
*/
static
void
my_machines_class_init(MyMachinesClass *klass)
{
	G_OBJECT_CLASS(klass)->dispose = my_machines_dispose;
	G_OBJECT_CLASS(klass)->finalize = my_machines_finalize;
//	GTK_WIDGET_CLASS(klass)->show = my_machines_show;
//	GTK_WIDGET_CLASS(klass)->hide = my_machines_hide;
	g_type_class_add_private(klass, sizeof(MyMachinesPrivate));

	g_signal_new("my-state-changed", G_TYPE_FROM_CLASS(G_OBJECT_CLASS(klass)), (GSignalFlags) (G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS), NULL, NULL, NULL, g_cclosure_marshal_VOID__UINT, G_TYPE_NONE, 1, G_TYPE_UINT);
	g_signal_new("my-load", G_TYPE_FROM_CLASS(G_OBJECT_CLASS(klass)), (GSignalFlags) (G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS), NULL, NULL, NULL, g_cclosure_marshal_VOID__UINT, G_TYPE_NONE, 1, G_TYPE_UINT);
	g_signal_new("my-gadget-captured", G_TYPE_FROM_CLASS(G_OBJECT_CLASS(klass)), (GSignalFlags) (G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS), NULL, NULL, NULL, g_cclosure_user_marshal_VOID__UINT_UINT, G_TYPE_NONE, 2, G_TYPE_UINT, G_TYPE_UINT);
	g_signal_new("my-gadget-released", G_TYPE_FROM_CLASS(G_OBJECT_CLASS(klass)), (GSignalFlags) (G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS), NULL, NULL, NULL, g_cclosure_user_marshal_VOID__UINT_UINT, G_TYPE_NONE, 2, G_TYPE_UINT, G_TYPE_UINT);
	g_signal_new("my-win-point", G_TYPE_FROM_CLASS(G_OBJECT_CLASS(klass)), (GSignalFlags) (G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS), NULL, NULL, NULL, g_cclosure_marshal_VOID__UINT, G_TYPE_NONE, 1, G_TYPE_UINT);
	g_signal_new("my-loose-point", G_TYPE_FROM_CLASS(G_OBJECT_CLASS(klass)), (GSignalFlags) (G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS), NULL, NULL, NULL, g_cclosure_marshal_VOID__UINT, G_TYPE_NONE, 1, G_TYPE_UINT);
	g_signal_new("my-out-point", G_TYPE_FROM_CLASS(G_OBJECT_CLASS(klass)), (GSignalFlags) (G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS), NULL, NULL, NULL, g_cclosure_marshal_VOID__UINT, G_TYPE_NONE, 1, G_TYPE_UINT);
	g_signal_new("my-win", G_TYPE_FROM_CLASS(G_OBJECT_CLASS(klass)), (GSignalFlags) (G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS), NULL, NULL, NULL, g_cclosure_marshal_VOID__UINT, G_TYPE_NONE, 1, G_TYPE_UINT);
}

MyMachines *
my_machines_new(void)
{
	return MY_MACHINES(g_object_new(MY_TYPE_MACHINES, NULL));
}

void
my_machines_load(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);

	priv->load.state = MY_MACHINES_LOAD_BEGIN;
	priv->load.items = priv->load.machines = 0;
	g_signal_emit_by_name(self, "my-load", priv->load.state);

	g_idle_add(on_load, self);
}

void
my_machines_unload(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	GdkGLContext *context = gtk_widget_get_gl_context(GTK_WIDGET(self));
	GdkGLDrawable *drawable = gtk_widget_get_gl_drawable(GTK_WIDGET(self));
	unsigned int i;

	if (!gdk_gl_drawable_gl_begin(drawable, context)) g_assert_not_reached();
	switch(priv->load.state)
	{
		case MY_MACHINES_LOAD_END:
		case MY_MACHINES_LOAD_SCENE:
		case MY_MACHINES_LOAD_MACHINES:
			for(i = 0; i < priv->load.machines; i++)
			{
				priv->play.players[i].machine->unload();
				delete priv->play.players[i].machine;
			}
		case MY_MACHINES_LOAD_ITEMS:
			for(i = 0; i < priv->load.items; i++)
			{
				priv->play.items[i].box->unload();
				delete priv->play.items[i].box;
				priv->play.items[i].gadget->unload();
				delete priv->play.items[i].gadget;
			}
		case MY_MACHINES_LOAD_TRACK:
			priv->play.track->unload();
			delete priv->play.track;
		case MY_MACHINES_LOAD_BEGIN:
			break;
	}
	gdk_gl_drawable_gl_end(drawable);
	priv->load.state = MY_MACHINES_LOAD_BEGIN;
}

gboolean
my_machines_is_loaded(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);

	return priv->load.state != MY_MACHINES_LOAD_BEGIN;
}

void
my_machines_set_state(MyMachines *self, MyMachinesState state)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);

	switch(state)
	{
		case MY_MACHINES_STATE_MAIN:
			switch(priv->state)
			{
				case MY_MACHINES_STATE_MAIN: break;
				case MY_MACHINES_STATE_PLAY: my_machines_leave_play(self); my_machines_enter_main(self); g_signal_emit_by_name(self, "my-state-changed", state); break;
				case MY_MACHINES_STATE_PAUSE: my_machines_leave_pause(self); my_machines_leave_play(self), my_machines_enter_main(self); g_signal_emit_by_name(self, "my-state-changed", state); break;
			}
			break;
		case MY_MACHINES_STATE_PLAY:
			switch(priv->state)
			{
				case MY_MACHINES_STATE_MAIN: my_machines_leave_main(self); my_machines_enter_play(self); g_signal_emit_by_name(self, "my-state-changed", state); break;
				case MY_MACHINES_STATE_PLAY: break;
				case MY_MACHINES_STATE_PAUSE: my_machines_leave_pause(self); g_signal_emit_by_name(self, "my-state-changed", state); break;
			}
			break;
		case MY_MACHINES_STATE_PAUSE:
			switch(priv->state)
			{
				case MY_MACHINES_STATE_MAIN: break;
				case MY_MACHINES_STATE_PLAY: my_machines_enter_pause(self); g_signal_emit_by_name(self, "my-state-changed", state); break;
				case MY_MACHINES_STATE_PAUSE: break;
			}
			break;
	}
}

MyMachinesState
my_machines_get_state(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	return priv->state;
}

void
my_machines_set_machines(MyMachines *self, guint machines)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	priv->options.nmachines = machines;
}

guint
my_machines_get_machines(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	return priv->options.nmachines;
}

void
my_machines_set_players(MyMachines *self, guint players)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	priv->options.nplayers = players;
}

guint
my_machines_get_players(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	return priv->options.nplayers;
}

void
my_machines_set_items(MyMachines *self, guint items)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	priv->options.nitems = items;
}

void
my_machines_set_points(MyMachines *self, guint points)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	priv->options.npoints = points;
}

void
my_machines_set_track(MyMachines *self, guint track)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	priv->options.track = track;
}

void
my_machines_set_machine(MyMachines *self, guint machine)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	priv->options.machine = machine;
}

const gchar *
my_machines_get_gadget(MyMachines *self, MyMachinesGadget gadget)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	return priv->gadgets[gadget];
}

const gchar *
my_machines_get_player_color(MyMachines *self, guint player)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	return priv->machines[player];
}

guint
my_machines_get_player_points(MyMachines *self, guint player)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	return priv->play.players[player].points;
}

gint
my_machines_get_player_gadget(MyMachines *self, guint player)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	return priv->play.players[player].item >= 0 ? priv->play.items[priv->play.players[player].item].type : -1;
}

gfloat
my_machines_get_load_fraction(MyMachines *self)
{
	MyMachinesPrivate *priv = MY_MACHINES_GET_PRIVATE(self);
	return (float) (priv->load.state + priv->load.items + priv->load.machines) / (MY_MACHINES_LOAD_END + priv->options.nitems + priv->options.nmachines);
}

