// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include <stdlib.h>
#include <GL/glew.h>
#include <AL/alc.h>
#include <AL/al.h>
#include <ode/ode.h>
#include <gtk/gtk.h>
#include "mymachines.h"

void
on_game_new_activate(GtkAction *action, gpointer data)
{
	GtkBuilder *builder = GTK_BUILDER(data);
	GtkWidget *widget, *machines;

	widget = GTK_WIDGET(gtk_builder_get_object(builder, "properties"));

	machines = GTK_WIDGET(gtk_builder_get_object(builder, "area"));

	widget = GTK_WIDGET(gtk_builder_get_object(builder, "machines"));
	my_machines_set_machines(MY_MACHINES(machines), gtk_range_get_value(GTK_RANGE(widget)));

	widget = GTK_WIDGET(gtk_builder_get_object(builder, "players"));
	my_machines_set_players(MY_MACHINES(machines), gtk_range_get_value(GTK_RANGE(widget)));

	widget = GTK_WIDGET(gtk_builder_get_object(builder, "items"));
	my_machines_set_items(MY_MACHINES(machines), gtk_range_get_value(GTK_RANGE(widget)));
	
	widget = GTK_WIDGET(gtk_builder_get_object(builder, "points"));
	my_machines_set_points(MY_MACHINES(machines), gtk_range_get_value(GTK_RANGE(widget)));

	widget = GTK_WIDGET(gtk_builder_get_object(builder, "track"));
	my_machines_set_track(MY_MACHINES(machines), gtk_combo_box_get_active(GTK_COMBO_BOX(widget)));

	widget = GTK_WIDGET(gtk_builder_get_object(builder, "machine"));
	my_machines_set_machine(MY_MACHINES(machines), gtk_combo_box_get_active(GTK_COMBO_BOX(widget)));

	my_machines_set_state(MY_MACHINES(machines), MY_MACHINES_STATE_PLAY);
}

void
on_game_play_activate(GtkAction *action, gpointer data)
{
	GtkBuilder *builder = GTK_BUILDER(data);
	GtkWidget *widget;

	widget = GTK_WIDGET(gtk_builder_get_object(builder, "area"));
	my_machines_set_state(MY_MACHINES(widget), MY_MACHINES_STATE_PLAY);
}

void
on_game_pause_activate(GtkAction *action, gpointer data)
{
	GtkBuilder *builder = GTK_BUILDER(data);
	GtkWidget *widget;

	widget = GTK_WIDGET(gtk_builder_get_object(builder, "area"));
	my_machines_set_state(MY_MACHINES(widget), MY_MACHINES_STATE_PAUSE);
}

void
on_game_stop_activate(GtkAction *action, gpointer data)
{
	GtkBuilder *builder = GTK_BUILDER(data);
	GtkWidget *widget;

	widget = GTK_WIDGET(gtk_builder_get_object(builder, "area"));
	my_machines_set_state(MY_MACHINES(widget), MY_MACHINES_STATE_MAIN);
}

void
on_game_quit_activate(GtkAction *action, gpointer data)
{
	GtkBuilder *builder = GTK_BUILDER(data);
	GtkWidget *widget;

	widget = GTK_WIDGET(gtk_builder_get_object(builder, "window"));
	gtk_object_destroy(GTK_OBJECT(widget));
}

void
on_window_destroy(GtkWidget *widget, gpointer data)
{
	gtk_main_quit();
}

void
on_options_properties_activate(GtkAction *action, gpointer data)
{
	GtkBuilder *builder = GTK_BUILDER(data);
	GtkWidget *widget;
	gint response;

	widget = GTK_WIDGET(gtk_builder_get_object(builder, "properties"));
	while((response = gtk_dialog_run(GTK_DIALOG(widget))) == GTK_RESPONSE_HELP);
	gtk_widget_hide(widget);
}

void
my_label_set_score(GtkLabel *label, GtkBuilder *builder)
{
	GtkWidget *machines;
	GString *markup = g_string_new(NULL);
	guint i;
	gint gadget;

	machines = GTK_WIDGET(gtk_builder_get_object(builder, "area"));
	for(i = 0; i < my_machines_get_machines(MY_MACHINES(machines)); i++)
	{
		gadget = my_machines_get_player_gadget(MY_MACHINES(machines), i);
		if(gadget >= 0)
			g_string_append_printf(markup, "<span size='larger' background='gray' color='%s'>%i %s</span> ", my_machines_get_player_color(MY_MACHINES(machines), i), my_machines_get_player_points(MY_MACHINES(machines), i), my_machines_get_gadget(MY_MACHINES(machines), gadget));
		else
			g_string_append_printf(markup, "<span size='larger' background='gray' color='%s'>%i</span> ", my_machines_get_player_color(MY_MACHINES(machines), i), my_machines_get_player_points(MY_MACHINES(machines), i));
	}
	gtk_label_set_markup(label, markup->str);
	g_string_free(markup, TRUE);

}

void
on_help_contents_activate(GtkAction *action, gpointer data)
{
	g_spawn_command_line_async("yelp " PACKAGE_DATADIR "/doc/" PACKAGE_YELP, NULL);
}

void
on_help_about_activate(GtkAction *action, gpointer data)
{
	GtkBuilder *builder = GTK_BUILDER(data);
	GtkWidget *widget;

	widget = GTK_WIDGET(gtk_builder_get_object(builder, "about"));
	gtk_dialog_run(GTK_DIALOG(widget));
	gtk_widget_hide(widget);
}

void
on_machines_value_changed(GtkRange *range, gpointer data)
{
	GtkBuilder *builder = GTK_BUILDER(data);
	GtkWidget *widget;
	GtkAdjustment *adjustment;

	widget = GTK_WIDGET(gtk_builder_get_object(builder, "players"));
	adjustment = gtk_range_get_adjustment(GTK_RANGE(widget));
	gtk_range_set_range(GTK_RANGE(widget), adjustment->lower, MIN(5, gtk_range_get_value(range) + 1));
	gtk_widget_queue_draw(widget);
}

void
on_my_state_changed(MyMachines *machines, MyMachinesState state, gpointer data)
{
	GtkBuilder *builder = GTK_BUILDER(data);
	GtkWidget *widget;
	GtkAction *action;

	switch(state)
	{
		case MY_MACHINES_STATE_MAIN:
			action = GTK_ACTION(gtk_builder_get_object(builder, "game_new"));
			gtk_action_set_sensitive(action, TRUE);
			action = GTK_ACTION(gtk_builder_get_object(builder, "game_play"));
			gtk_action_set_sensitive(action, FALSE);
			action = GTK_ACTION(gtk_builder_get_object(builder, "game_pause"));
			gtk_action_set_sensitive(action, FALSE);
			action = GTK_ACTION(gtk_builder_get_object(builder, "game_stop"));
			gtk_action_set_sensitive(action, FALSE);
			action = GTK_ACTION(gtk_builder_get_object(builder, "options_properties"));
			gtk_action_set_sensitive(action, TRUE);
			widget = GTK_WIDGET(gtk_builder_get_object(builder, "score"));
			gtk_widget_hide(widget);
			widget = GTK_WIDGET(gtk_builder_get_object(builder, "statusbar"));
			gtk_statusbar_pop(GTK_STATUSBAR(widget), 0);
			break;
		case MY_MACHINES_STATE_PLAY:
			action = GTK_ACTION(gtk_builder_get_object(builder, "game_new"));
			gtk_action_set_sensitive(action, FALSE);
			action = GTK_ACTION(gtk_builder_get_object(builder, "game_play"));
			gtk_action_set_sensitive(action, FALSE);
			action = GTK_ACTION(gtk_builder_get_object(builder, "game_pause"));
			gtk_action_set_sensitive(action, TRUE);
			action = GTK_ACTION(gtk_builder_get_object(builder, "game_stop"));
			gtk_action_set_sensitive(action, TRUE);
			action = GTK_ACTION(gtk_builder_get_object(builder, "options_properties"));
			gtk_action_set_sensitive(action, FALSE);
			widget = GTK_WIDGET(gtk_builder_get_object(builder, "statusbar"));
			gtk_statusbar_pop(GTK_STATUSBAR(widget), 0);
			break;
		case MY_MACHINES_STATE_PAUSE:
			action = GTK_ACTION(gtk_builder_get_object(builder, "game_play"));
			gtk_action_set_sensitive(action, TRUE);
			action = GTK_ACTION(gtk_builder_get_object(builder, "game_pause"));
			gtk_action_set_sensitive(action, FALSE);
			widget = GTK_WIDGET(gtk_builder_get_object(builder, "statusbar"));
			gtk_statusbar_pop(GTK_STATUSBAR(widget), 0);
			gtk_statusbar_push(GTK_STATUSBAR(widget), 0, "Game paused");
			break;
	}
}

void
on_my_load(MyMachines *machines, MyMachinesLoadState state, gpointer data)
{
	GtkBuilder *builder = GTK_BUILDER(data);
	GtkWidget *widget;

	widget = GTK_WIDGET(gtk_builder_get_object(builder, "progressbar"));
	gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(widget), my_machines_get_load_fraction(machines));
	switch(state)
	{
		case MY_MACHINES_LOAD_BEGIN:
			gtk_widget_show(widget);
			break;
		case MY_MACHINES_LOAD_TRACK:
			widget = GTK_WIDGET(gtk_builder_get_object(builder, "statusbar"));
			gtk_statusbar_pop(GTK_STATUSBAR(widget), 0);
			gtk_statusbar_push(GTK_STATUSBAR(widget), 0, "Loading track");
			break;
		case MY_MACHINES_LOAD_ITEMS:
			widget = GTK_WIDGET(gtk_builder_get_object(builder, "statusbar"));
			gtk_statusbar_pop(GTK_STATUSBAR(widget), 0);
			gtk_statusbar_push(GTK_STATUSBAR(widget), 0, "Loading items");
			break;
		case MY_MACHINES_LOAD_MACHINES:
			widget = GTK_WIDGET(gtk_builder_get_object(builder, "statusbar"));
			gtk_statusbar_pop(GTK_STATUSBAR(widget), 0);
			gtk_statusbar_push(GTK_STATUSBAR(widget), 0, "Loading machines");
			break;
		case MY_MACHINES_LOAD_SCENE:
			widget = GTK_WIDGET(gtk_builder_get_object(builder, "statusbar"));
			gtk_statusbar_pop(GTK_STATUSBAR(widget), 0);
			gtk_statusbar_push(GTK_STATUSBAR(widget), 0, "Loading scene");
			break;
		case MY_MACHINES_LOAD_END:
			gtk_widget_hide(widget);
			widget = GTK_WIDGET(gtk_builder_get_object(builder, "score"));
			my_label_set_score(GTK_LABEL(widget), builder);
			gtk_widget_show(widget);
			widget = GTK_WIDGET(gtk_builder_get_object(builder, "statusbar"));
			gtk_statusbar_pop(GTK_STATUSBAR(widget), 0);
	}
}

void
on_my_gadget_captured(MyMachines *machines, guint player, MyMachinesGadget gadget, gpointer data)
{
	GtkBuilder *builder = GTK_BUILDER(data);
	GtkWidget *widget;
	gchar *text;

	widget = GTK_WIDGET(gtk_builder_get_object(builder, "score"));
	my_label_set_score(GTK_LABEL(widget), builder);

	widget = GTK_WIDGET(gtk_builder_get_object(builder, "statusbar"));
	gtk_statusbar_pop(GTK_STATUSBAR(widget), 0);
	text = g_strdup_printf("%s player gets %s gadget", my_machines_get_player_color(machines, player), my_machines_get_gadget(machines, gadget));
	text[0] = g_ascii_toupper(text[0]);
	gtk_statusbar_push(GTK_STATUSBAR(widget), 0, text);
	g_free(text);
}

void
on_my_gadget_released(MyMachines *machines, guint player, MyMachinesGadget gadget, gpointer data)
{
	GtkBuilder *builder = GTK_BUILDER(data);
	GtkWidget *widget;
	gchar *text;

	widget = GTK_WIDGET(gtk_builder_get_object(builder, "score"));
	my_label_set_score(GTK_LABEL(widget), builder);

	widget = GTK_WIDGET(gtk_builder_get_object(builder, "statusbar"));
	gtk_statusbar_pop(GTK_STATUSBAR(widget), 0);
	text = g_strdup_printf("%s player looses %s gadget", my_machines_get_player_color(machines, player), my_machines_get_gadget(machines, gadget));
	text[0] = g_ascii_toupper(text[0]);
	gtk_statusbar_push(GTK_STATUSBAR(widget), 0, text);
	g_free(text);
}

void
on_my_win_point(MyMachines *machines, guint player, gpointer data)
{
	GtkBuilder *builder = GTK_BUILDER(data);
	GtkWidget *widget;
	gchar *text;

	widget = GTK_WIDGET(gtk_builder_get_object(builder, "score"));
	my_label_set_score(GTK_LABEL(widget), builder);

	widget = GTK_WIDGET(gtk_builder_get_object(builder, "statusbar"));
	gtk_statusbar_pop(GTK_STATUSBAR(widget), 0);
	text = g_strdup_printf("%s player wins point", my_machines_get_player_color(machines, player));
	text[0] = g_ascii_toupper(text[0]);
	gtk_statusbar_push(GTK_STATUSBAR(widget), 0, text);
	g_free(text);
}

void
on_my_loose_point(MyMachines *machines, guint player, gpointer data)
{
	GtkBuilder *builder = GTK_BUILDER(data);
	GtkWidget *widget;
	gchar *text;

	widget = GTK_WIDGET(gtk_builder_get_object(builder, "score"));
	my_label_set_score(GTK_LABEL(widget), builder);

	widget = GTK_WIDGET(gtk_builder_get_object(builder, "statusbar"));
	gtk_statusbar_pop(GTK_STATUSBAR(widget), 0);
	text = g_strdup_printf("%s player looses point", my_machines_get_player_color(machines, player));
	text[0] = g_ascii_toupper(text[0]);
	gtk_statusbar_push(GTK_STATUSBAR(widget), 0, text);
	g_free(text);
}

void
on_my_out_point(MyMachines *machines, guint player, gpointer data)
{
	GtkBuilder *builder = GTK_BUILDER(data);
	GtkWidget *widget;
	gchar *text;

	widget = GTK_WIDGET(gtk_builder_get_object(builder, "statusbar"));
	gtk_statusbar_pop(GTK_STATUSBAR(widget), 0);
	text = g_strdup_printf("%s player is out of point", my_machines_get_player_color(machines, player));
	text[0] = g_ascii_toupper(text[0]);
	gtk_statusbar_push(GTK_STATUSBAR(widget), 0, text);
	g_free(text);
}

void
on_my_win(MyMachines *machines, guint player, gpointer data)
{
	GtkBuilder *builder = GTK_BUILDER(data);
	GtkWidget *widget;
	gchar *text;

	widget = GTK_WIDGET(gtk_builder_get_object(builder, "statusbar"));
	gtk_statusbar_pop(GTK_STATUSBAR(widget), 0);
	text = g_strdup_printf("%s player wins", my_machines_get_player_color(machines, player));
	text[0] = g_ascii_toupper(text[0]);
	gtk_statusbar_push(GTK_STATUSBAR(widget), 0, text);
	g_free(text);
}

void
my_fix(GtkBuilder *builder)
{
	GtkWidget *widget;
	widget = GTK_WIDGET(gtk_builder_get_object(builder, "track"));
	gtk_combo_box_set_active(GTK_COMBO_BOX(widget), 0);
	widget = GTK_WIDGET(gtk_builder_get_object(builder, "machine"));
	gtk_combo_box_set_active(GTK_COMBO_BOX(widget), 0);
}

int
main(int argc, char **argv)
{
	int result;
	GtkBuilder *builder;
	GtkWidget *window, *widget;
	ALCdevice *device;
	ALCcontext *acontext;

	glewInit();
	gtk_init(&argc, &argv);
	device = alcOpenDevice(NULL);
	if(device)
	{
		acontext = alcCreateContext(device, NULL);
		if(acontext)
		{
			alcMakeContextCurrent(acontext);
			dInitODE();
				gtk_window_set_default_icon_from_file(DATADIR "/icons/" PACKAGE_ICON, NULL);
				builder = gtk_builder_new();
					gtk_builder_add_from_file(builder, PACKAGE_DATADIR "/ui/" PACKAGE_UI, NULL);
					my_fix(builder);
					gtk_builder_connect_signals(builder, builder);
						window = GTK_WIDGET(gtk_builder_get_object(builder, "window"));
						widget = GTK_WIDGET(gtk_builder_get_object(builder, "area"));
						g_signal_connect(widget, "my-state-changed", G_CALLBACK(on_my_state_changed), builder);
						g_signal_connect(widget, "my-load", G_CALLBACK(on_my_load), builder);
						g_signal_connect(widget, "my-gadget-captured", G_CALLBACK(on_my_gadget_captured), builder);
						g_signal_connect(widget, "my-gadget-released", G_CALLBACK(on_my_gadget_released), builder);
						g_signal_connect(widget, "my-win-point", G_CALLBACK(on_my_win_point), builder);
						g_signal_connect(widget, "my-loose-point", G_CALLBACK(on_my_loose_point), builder);
						g_signal_connect(widget, "my-out-point", G_CALLBACK(on_my_out_point), builder);
						g_signal_connect(widget, "my-win", G_CALLBACK(on_my_win), builder);
						gtk_widget_show(window);
							gtk_main();
					g_object_unref(G_OBJECT(builder));
			dCloseODE();
			alcMakeContextCurrent(NULL);
			alcDestroyContext(acontext);
			result = EXIT_SUCCESS;
		}
		else result = EXIT_FAILURE;
		alcCloseDevice(device);
	}
	else result = EXIT_FAILURE;
	return result;
}
